/* enhance -- perform enhancement of clean speech signals */

/* M.A.Huckvale - University College London */

/* version 1.0 - January 1998 */

#define PROGNAME "enhance"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH ENHANCE SFS1 UCL
.SH NAME
enhance -- enhance clean speech signal
.SH SYNOPSIS
.B enhance
(-i item) (-c mu-law) (-a mu-law) (-A label) file
.SH DESCRIPTION
.I enhance
is a program to perform a range of enhancement operations on
operations on speech signals.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select input item.
.TP 11
.BI -c mulaw
Select compression-style enhancement using a mu-law compression function:
output-energy = log(1+mu.input-energy)/log(1+mu).  A mu value of 10 gives
modest compression.
.TP 11
.BI -a mulaw
Select annotation-style enhancement.  All selected regions
are enhanced according to a mu-law compression
function as '-c'.
.TP 11
.BI -A label
Add the given label as a type of region to enhance.
.TP 11
.BI -X label
Select all regions except the labelled region for enhancement.
.SH INPUT ITEMS
.IP 1.xx 11
Any speech item.
.SH VERSION/AUTHOR
1.0 - Mark Huckvale.
*/
/*--------------------------------------------------------------------------*/


/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"			/* header structures */

/* manifest constants */
#define WINTIME	0.030		/* process in 30ms windows */
#define DBRANGE	40.0		/* dynamic range */

/* global data */
struct item_header spitem;	/* input item header data */
struct item_header opitem;	/* output item header */
struct item_header anitem;	/* annotation item */
struct an_rec *an;

/* annotation lists */
#define MAXLABEL	32
char	*label[MAXLABEL];
int	nlabel=0;
int	exclude=0;

/* buffering */
short	*buff;
short	*hold;
double	*window;
int	buffsize;
double	maxenergy;

/* operations selected */
int	compmode=1;		/* compression mode */
int	anmode=0;		/* annotation mode */
double	mu=10.0;		/* mu value for compression */

/* save string */
char *strsave(char *str)
{
	char *ptr=malloc(strlen(str)+1);
	strcpy(ptr,str);
	return(ptr);
}

/* check if annotation on list */
int onlist(char *str)
{
	int	i;
	for (i=0;i<nlabel;i++)
		if (strcmp(label[i],str)==0)
			return(1);
	return(0);
}

/* check window against annotations: 0=enhance, 1=don't enhance */
int checkwin(double start,double stop)
{
	double	t1,t2;
	int	i,j,match;

	for (i=0;i<anitem.numframes;i++) {
		/* check if this annotation mentioned on command line */
		match=onlist(an[i].label);
		/* run forward to end of equivalent block */
		for (j=i+1;j<anitem.numframes;j++)
			if (onlist(an[j].label)!=match) break;
		j--;
		/* find start and stop time of block */
		t1 = anitem.offset + an[i].posn*anitem.frameduration;
		t2 = anitem.offset + (an[j].posn + an[j].size)*anitem.frameduration;
		/* check whether to exclude */
		if (exclude) {
			if (match) {
				/* exclude this window only if completely in region */
				if ((t1 <= start) && (stop <= t2))
					return(1);
			}
		}
		else {
			if (match) {
				/* include this window if any part in region */
				if ((t1 <= start) && (start <= t2))
					return(0);
				if ((t1 <= stop) && (stop <= t2))
					return(0);
			}
		}
	}
	return(!exclude);
}
	
/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
	/* local variables */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument */
	int		errflg=0;	/* option error flag */
	int		c;		/* option char */
	int		it;
	char		*ty;
	char		*sptype="0";
	char		*antype="0";
	char		filename[SFSMAXFILENAME];	
					/* database file name */
	int		fid,ofid;
	int		i,j;
	double		energy;
	double		minenergy;
	double		newen;
	double		sumsq,factor;
	double		val;
	double		omega;
	int		overload=0;
 
	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:c:a:A:X:")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Enhance clean speech V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* item spec */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					sptype = ty;
				else if (it == AN_TYPE)
					antype = ty;
				else
					error("unsuitable item specification %s",optarg);
			}
			else
				error("illegal item specification %s",optarg);
			break;
		case 'c' :	/* compression mode */
			mu = atof(optarg);
			compmode=1;
			anmode=0;
			break;
		case 'a' :	/* annotation mode */
			mu = atof(optarg);
			compmode=0;
			anmode=1;
			break;
		case 'A' :	/* add annotated region */
			if (nlabel && exclude)
				error("cannot mix -A and -X switches");
			exclude=0;
			label[nlabel++] = strsave(optarg);
			break;
		case 'X' :	/* exclude annotated region */
			if (nlabel && !exclude)
				error("cannot mix -A and -X switches");
			exclude=1;
			label[nlabel++] = strsave(optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-c mu-value) (-a mu-value) (-A addlabel|-X exclabel) file\n",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open file */
	if ((fid=sfsopen(filename,"w",NULL)) < 0) {
		if (fid==-1)
			error("unable to find file '%s'",filename);
		else
			error("access error on '%s'",filename);
	}

	if (anmode) {
		/* locate input item */
		if (!sfsitem(fid,AN_TYPE,antype,&anitem))
			error("unable to find input item in '%s'",filename);
		an = (struct an_rec *)sfsbuffer(&anitem,anitem.numframes);
		sfsread(fid,0,anitem.numframes,an);
printf("read in %d annotations\n",anitem.numframes);
	}

	/* locate input item */
	if (!sfsitem(fid,SP_TYPE,sptype,&spitem))
		error("unable to find input item in '%s'",filename);

	/* calculate analysis window size in samples */
	buffsize = (int)(0.5 + WINTIME / spitem.frameduration);
	if (buffsize & 1) buffsize++;
	
	/* get buffers */
	buff = (short *)sfsbuffer(&spitem,buffsize);
	hold = (short *)sfsbuffer(&spitem,buffsize);
	memset(hold,0,buffsize*sizeof(short));
	window = (double *)calloc(buffsize,sizeof(double));

	/* create output item header */
	sfsheader(&opitem,spitem.datatype,spitem.floating,
			spitem.datasize,spitem.framesize,
			spitem.frameduration,spitem.offset,
			spitem.windowsize,spitem.overlap,spitem.lxsync);
	if (compmode)
		sprintf(opitem.history,"%s(%d.%02d;mu=%g)",
			PROGNAME,
			spitem.datatype,spitem.subtype,mu);
	else {
		sprintf(opitem.history,"%s(%d.%02d,%d.%02d;mu=%g%s%s",
			PROGNAME,
			spitem.datatype,spitem.subtype,
			anitem.datatype,anitem.subtype,mu,
			(exclude)?",exclude":"",
			(nlabel>0)?",labels=":"");
		for (i=0;i<nlabel;i++) {
			if (i) strcat(opitem.history,",");
			strcat(opitem.history,label[i]);
		}
		strcat(opitem.history,")");
	}

	/* open output channel */
	if ((ofid=sfschannel(filename,&opitem)) < 0)
		error("unable to open output file",NULL);

	/* calculate raised cosine window */
	omega = 8.0*atan(1.0)/(buffsize-1);
	for (i=0;i<buffsize;i++) {
		window[i] = 0.5 - 0.5*cos(i*omega);
	}

	/* find window with maximum energy */
	maxenergy = 0;
	for (i=0;sfsread(fid,i,buffsize,buff)==buffsize;i+=buffsize/2) {

		/* calculate energy in window */
		sumsq = 0;
		for (j=0;j<buffsize;j++) {
			val = buff[j] * window[j];
			sumsq += val * val;
		}
		energy = 10*log10(sumsq/buffsize);
		if (energy > maxenergy) maxenergy = energy;
	}

	printf("Maximum energy in window = %.2fdB\n",maxenergy);
	minenergy = maxenergy - DBRANGE;

	/* process file */
	for (i=0;sfsread(fid,i,buffsize,buff)==buffsize;i+=buffsize/2) {

		/* calculate energy in window */
		sumsq = 0;
		for (j=0;j<buffsize;j++) {
			val = buff[j] * window[j];
			sumsq += val * val;
		}
		energy = 10*log10(sumsq/buffsize);

		/* calculate new energy value */
		if (energy < minenergy)
			newen = energy;
		else
			newen = minenergy + DBRANGE*log(1+mu*(energy-minenergy)/DBRANGE) / log(1+mu);

		if (compmode) {
			/* calculate multiplying factor */
			factor = pow(10.0,(newen-energy)/20);
/*
printf("Old: %5.2fdB  New: %5.2fdB  Factor: %5.2f\n",energy,newen,factor);
*/
		}
		else {
			if (checkwin(spitem.offset+i*spitem.frameduration,
				     spitem.offset+(i+buffsize-1)*spitem.frameduration))
				factor = 1.0;
			else
				factor = pow(10.0,(newen-energy)/20);
		}

		/* scale output and add to hold buffer */
		for (j=0;j<buffsize;j++) {
			val = hold[j] + buff[j] * window[j] * factor;
			if (val < -32768) {
				overload=1;
				hold[j] = -32768;
			}
			else if (val > 32767) {
				overload=1;
				hold[j] = 32767;
			}
			else
				hold[j] = (short)val;
		}

		/* write out result of processing */
		if (sfswrite(ofid,buffsize/2,hold) != buffsize/2)
			error("write error on output file",NULL);

		/* shift hold buffer and reset */
		for (j=0;j<buffsize/2;j++) {
			hold[j] = hold[buffsize/2 + j];
			hold[buffsize/2 + j] = 0;
		}
	}
	/* write out result of processing last buffer */
	if (sfswrite(ofid,buffsize/2,hold) != buffsize/2)
		error("write error on output file",NULL);
	if (overload)
		fprintf(stderr,"%s: WARNING - overload on output\n",PROGNAME);

	/* update file */
	if (!sfsupdate(filename))
		error("update error on %s",filename);

	/* ... that's all folks */
	exit(0);
}

