/* ansi -- DIG routines for UCL/ANSI Graphics Terminal Emulators */

#include "SFSCONFG.h"
#ifdef EVa

#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <fcntl.h>
#include <termio.h>
#include "dig.h"
#include "digdata.h"


/* global data */
static struct {
	short subtype;		/* 0=Archimedes, 1=IBM PC */
	short mode;		/* 0=?, 1=initialised */
	short currbun;		/* current bundle */
	short rop;		/* raster operation */
	short xmouse,ymouse,smouse; /* mouse status */
} digDEVa;

/* device characteristics */
int ansi_params(subtype,digdev)
int	subtype;
struct digdevice *digdev;
{

	/* NOTE: the UCL/ANSI emulator should have capability to return
			this information under command */

	/* get device characteristics from subtype */
	digDEVa.subtype = subtype;
	switch (subtype) {
	case 0:		/* Archimedes */
		digdev->nhoriz = 640;
		digdev->nvert = 256;
		digdev->aspect = 2.0;
		digdev->chwidth = 16;
		digdev->chheight = 8;
		digdev->greylevels = 16;
		digdev->greypixwidth = 2;
		digdev->greypixheight = 1;
		break;
	case 1:		/* IBM PC - Hercules Monochrome */
		digdev->nhoriz = 720;
		digdev->nvert = 348;
		digdev->aspect = 1.578;
		digdev->chwidth = 8;
		digdev->chheight = 8;
		digdev->greylevels = 2;
		digdev->greypixwidth = 1;
		digdev->greypixheight = 1;
		break;
	case 2:		/* IBM PC - Colour EGA */
		digdev->nhoriz = 640;
		digdev->nvert = 350;
		digdev->aspect = 1.2;
		digdev->chwidth = 8;
		digdev->chheight = 8;
		digdev->greylevels = 7;
		digdev->greypixwidth = 2;
		digdev->greypixheight = 1;
		break;
	case 3:		/* IBM PC - Colour VGA */
		digdev->nhoriz = 640;
		digdev->nvert = 480;
		digdev->aspect = 1.0;
		digdev->chwidth = 8;
		digdev->chheight = 8;
		digdev->greylevels = 7;
		digdev->greypixwidth = 1;
		digdev->greypixheight = 1;
		break;
	case 4:		/* IBM PC - Super VGA */
		digdev->nhoriz = 800;
		digdev->nvert = 600;
		digdev->aspect = 1.0;
		digdev->chwidth = 8;
		digdev->chheight = 8;
		digdev->greylevels = 7;
		digdev->greypixwidth = 1;
		digdev->greypixheight = 1;
		break;
	default:
		error("(dig) unknown ANSI terminal type");
	}
	return(0);
}

/* open graphics */
int ansi_open()
{
	if (!(digDEVa.mode)) {
		printf("\033[1g");	/* init graphics */
		fflush (stdout) ;
		sleep(1);
		(digDEVa.mode)++;
		digDEVa.currbun=24;
		digDEVa.rop=0;
	}
	return(0);
}

/* interrupt graphics */
int ansi_inter()
{
	/* should somehow recover from partially output drawing command */
	printf("Interrupt");
	fflush(stdout);
	return(0);
}

/* close graphics - return to text mode */
int ansi_close()
{
	printf("\033[0g");	/* close graphics */
	fflush(stdout);
	digDEVa.mode=0;
	return(0);
}

/* clear screen */
int ansi_clear()
{
	printf("\033[k");
	fflush(stdout);
	return(0);
}

/* poly line */
int ansi_polyline(bundle,buff,len)
int	bundle;
short	*buff;
int	len;
{
	int	x,y;

	/* check colour */
	if (bundle != digDEVa.currbun) {
		printf("\033[%dh",bundle);
		digDEVa.currbun=bundle;
	}

	/* position to first point */
	x = *buff++;
	y = *buff++;
	printf("\033[%d;%dp",x,y);
	len -= 2;

	/* plot rest of points */
	while (len > 0) {
		x = *buff++;
		y = *buff++;
		printf("\033[%d;%dd",x,y);
		len -= 2;
	}
	fflush(stdout);
	return(0);
}

/* draw 'hardware' text */
int ansi_text(bundle,x,y,str)
int	bundle;
int	x,y;
char	*str;
{
	printf("\033[0c\033[%dh\033[%d;%dp%s\033[1c",bundle,x,y,str);
	digDEVa.currbun=bundle;
	fflush(stdout);
	return(0);
}

/* draw a filled rectangle */
int ansi_fillrect(bundle,x1,y1,x2,y2)
int	bundle;
int	x1,y1,x2,y2;
{
	printf("\033[%dh\033[%d;%dp\033[%d;%db",bundle,x1,y1,x2,y2);
	digDEVa.currbun=bundle;
	return(0);
}

/* change line/text raster operation */
int ansi_rasterop(mode)
int	mode;
{
	int	ret;
	printf("\033[%dl",mode);
	ret=digDEVa.rop;
	digDEVa.rop=mode;
	return(ret);
}

/* initialise pointing device */
int ansi_pointer(flag,xm,ym)
int	flag;
int	xm,ym;
{
	extern int	errno;
	struct termio 	newmode;

	if (flag) {
		/* open channel to terminal in raw mode */
		if (digdata.tty == -1) {
			if ((digdata.tty = open ("/dev/tty",O_RDONLY)) >= 0) {
				ioctl(digdata.tty,TCGETA,&digdata.oldmode);
				newmode = digdata.oldmode;
				newmode.c_lflag = newmode.c_lflag & ~ICANON;
				newmode.c_lflag = newmode.c_lflag & ~ECHO;
				newmode.c_cc[4]=1;		/* 1 character burst */
				newmode.c_cc[5]=0;		/* wait */
				ioctl(digdata.tty,TCSETA,&newmode);
			}
			else {
				fprintf(stderr,"diginitmouse: unable to open keyboard, errno=%d\n",errno);
				return(-1);
			}
		}

		printf("\033[1o");
		fflush(stdout);
		digDEVa.xmouse = xm;
		digDEVa.ymouse = ym;
		digDEVa.smouse = 0;
	}
	else {
		/* restore keyboard */
		if (digdata.tty >= 0) {
			ioctl(digdata.tty,TCFLSH,0);
			ioctl(digdata.tty,TCSETA,&digdata.oldmode) ;
			close(digdata.tty);
		}
		digdata.tty = -1;
		printf("\033[0o");
		fflush(stdout);
	}
	return(0);
}

/* get mouse/key status */
int ansi_mouse(but,xm,ym,ich)
int	*but;
int	*xm;
int	*ym;
int	*ich;
{
	int	p1,p2,p3;
	char	ch;

	*xm = digDEVa.xmouse;
	*ym = digDEVa.ymouse;
	*but = digDEVa.smouse;

	/* check channel opened */
	if (digdata.tty < 0) {
		fprintf(stderr,"diggetmousech: diginitmouse not called\n");
		return(0);
	}

	/* get a character */
	ch = '\0' ;
	read(digdata.tty,&ch,1);
	ch = ch & 127;		/* zap parity bit */
	*ich = ch;

	/* check for MOUSE command */
	if (ch==ESC) {
		read(digdata.tty,&ch,1);
		if (ch=='[') {
			read(digdata.tty,&ch,1);
			p1= 0;
			p2= -1;
			p3= -1;
			while (isdigit(ch)) {
				p1 = 10 * p1 + ch - '0';
				read(digdata.tty,&ch,1);
			}
			if (ch != ';') {
				*ich = ch;
				return(0);
			}
			read(digdata.tty,&ch,1);
			p2=0;
			while (isdigit(ch)) {
				p2 = 10 * p2 + ch - '0';
				read(digdata.tty,&ch,1);
			}
			if (ch != ';') {
				*ich=ch;
				return(0);
			}
			read(digdata.tty,&ch,1);
			p3=0;
			while (isdigit(ch)) {
				p3 = 10 * p3 + ch - '0';
				read(digdata.tty,&ch,1);
			}
			if ((ch=='o') && (p3 >=0)) {
				*xm=digDEVa.xmouse=p1;
				*ym=digDEVa.ymouse=p2;
				*but=digDEVa.smouse=p3;
				*ich=0;
				return(DIG_MOUSE);
			}
			else
				*ich=ch;
		}
	}
	return((*ich)?DIG_KEY:0);
}

/* display text on prompt line */
int ansi_prompt(str)
char	*str;
{
	printf("\033[H\033[K%s",str);
	fflush(stdout);
	return(0);
}

/* grey-level display */
int ansi_gscale(x,y,height,buff)
int	x,y;
int	height;
unsigned char	*buff;
{
	unsigned char	*ip,*op;
	int		i,j,num,cnt;
	unsigned char	lastcol;

	if (digDEVa.subtype<3) return(0);

	/* do run-length compression of buffer */
	lastcol = *buff;
	ip=buff+1;
	op=buff;
	num=1;
	for (i=1;i<height;i++,ip++) {
		if ((*ip==lastcol) && (num < 63))
			num++;
		else if (num > 2) {
			*op++ = 32+num;
			*op++ = 96+(7-lastcol);
			lastcol = *ip;
			num = 1;
		}
		else {
			for (j=0;j<num;j++) *op++ = 96+(7-lastcol);
			lastcol = *ip;
			num = 1;
		}
	}
	if (num > 2) {
		*op++ = 32+num;
		*op++ = 96+(7-lastcol);
	}
	else {
		for (j=0;j<num;j++) *op++ = 96+(7-lastcol);
	}
	cnt = op - buff;

	printf("\033[%d;%dp",x,y);
	printf("\033[%d;%dx",height,cnt);
	for (i=0;i<cnt;i++,buff++) putchar(*buff);
	return(0);
}

#endif
