/* envelope -- generate amplitude envelope of speech waveform */

/* m.a. huckvale - august 1989 */

/* version 1.1 - July 1993
	- add dB output option
*/

#define PROGNAME "envelope"
#define PROGVERS "1.1"
char *progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH ENVELOPE SFS1 UCL
.SH NAME
envelope - generate amplitude envelope track of speech waveform
.SH SYNOPSIS
.B envelope
(-i item) (-w windowsize) (-h) (-d) file
.SH DESCRIPTION
.I envelope
produces a smoothed amplitude envelope of a speech pressure waveform
as a track item at the same sampling frequency.
The envelope is calculated from the sum-squared sample values in
a given window.  The window is supplied on the command line.  A rectangular
window is used by default, but a hamming window is an option.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select input item number.
.TP 11
.BI -w windowsize
Specify the envelope window width in seconds.  Default 1ms.
.TP 11
.B -h
Use hamming window.  Default rectangular.
.TP 11
.B -d
Output in dB (10log10(envelope)).
.SH INPUT ITEMS
.IP 1.xx 11
Any speech waveform.
.SH OUTPUT ITEMS
.IP 16 11
Amplitude envelope.
.SH VERSION/AUTHOR
1.0 - Mark Huckvale.
*/
/*--------------------------------------------------------------------------*/

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"			/* header structures */

#define MAX(x,y)	(((x)>(y))?(x):(y))

/* global data */
struct item_header 	spitem;		/* item header data */
short			*sp;		/* speech data buffer (1 x 20ms window) */
struct item_header 	tritem;		/* item header data */
float			*tr;		/* track item buffer */

/* options */
double		windowtime=0.001;	/* window width (seconds) */
int		winsize;		/* window width (samples) */
int		hamming=0;		/* Hamming window reqd */
int		trdb=0;			/* output track in dB */

/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
	/* local variables */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument */
	int		errflg=0;	/* option error flag */
	int		c;		/* option char */
	int		it;
	char		*ty;
	char		*sptype = "0";	/* input sp sub-type = last */
	char		filename[SFSMAXFILENAME];
					/* database file name */
	int		fid,ofid;	/* file descriptors */

	/* processing variables */
	int		i,j;
	int		spbufsize,trbufsize,trbufptr=0;
	float		*fsamp;
	float		sqsamp(),hwindow(),rwindow();

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:w:hd")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Amplitude envelope V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* item spec */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					sptype=ty;
				else
					error("unsuitable item specification %s",optarg);
			}
			else
				error("illegal item specification %s",optarg);
			break;
		case 'w' :	/* window size */
			windowtime = atof(optarg);
			if (windowtime <= 0.0)
				error("bad window time '%s'",optarg);
			break;
		case 'h' :	/* hamming window */
			hamming++;
			break;
		case 'd' :	/* dB output */
			trdb++;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-w windowsize) (-h) (-d) file\n",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open file */
	if ((fid=sfsopen(filename,"w",NULL)) < 0) {
		if (fid==-1)
			error("'%s': file not found",filename);
		else
			error("access error on '%s'",filename);
	}

	/* locate input speech item */
	if (!sfsitem(fid,SP_TYPE,sptype,&spitem))
		error("cannot find input SP item in '%s'",filename);

	/* get window parameters */
	winsize = (int)(0.5+windowtime/spitem.frameduration);
	if (winsize==0)
		error("window size less than one speech sample",NULL);
	if (winsize >= spitem.numframes)
		error("window size larger than speech signal!",NULL);

	/* generate output item header */
	sfsheader(&tritem,TR_TYPE,1,4,1,spitem.frameduration,spitem.offset+windowtime/2,winsize,winsize-1,0);
	sprintf(tritem.history,"%s(%d.%02d;window=%g%s%s)",
		PROGNAME,spitem.datatype,spitem.subtype,
		windowtime,(hamming)?",hamming":"",
		(trdb)?",decibel":"");

	/* get input buffer */
	spbufsize = MAX(4096,winsize);
	if ((sp=(short *)sfsbuffer(&spitem,spbufsize)) == NULL)
		error("cannot get buffer for speech",NULL);

	/* get output buffer */
	trbufsize = MAX(512,winsize);
	if ((tr=(float *)sfsbuffer(&tritem,trbufsize)) == NULL)
		error("cannot get buffer for output",NULL);

	/* get processing buffer */
	if ((fsamp=(float *)calloc(winsize,sizeof(float)))==NULL)
		error("could not get processing buffer",NULL);

	/* open output channel */
	if ((ofid=sfschannel(filename,&tritem)) < 0)
		error("cannot open temporary file",NULL);

	/* pre-load buffer */
	for (i=0;i<winsize-1;i++) fsamp[i]=sqsamp(i,fid,sp,spbufsize);

	/* perform processing */
	for (i=winsize-1;i<spitem.numframes;i++) {

		/* get last value into buffer */
		fsamp[winsize-1] = sqsamp(i,fid,sp,spbufsize);

		/* calculate envelope */
		if (hamming)
			tr[trbufptr] = hwindow(fsamp,winsize);
		else
			tr[trbufptr] = rwindow(fsamp,winsize);
		if (trdb) {
			if (tr[trbufptr]<1.0) tr[trbufptr]=(float)1.0;
			tr[trbufptr] = (float)(10.0*log10(tr[trbufptr]));
		}

		/* write out values to file */
		trbufptr++;
		if (trbufptr==trbufsize) {
			sfswrite(ofid,trbufsize,tr);
			trbufptr=0;
			if (ttytest()) {
				printf("\rFrame %d/%d",(i+1)-(winsize-1),spitem.numframes-winsize+1);
				fflush(stdout);
			}
		}

		/* shift buffer */
		for (j=0;j<winsize-1;j++)
			fsamp[j] = fsamp[j+1];
	}

	/* finish output buffer */
	sfswrite(ofid,trbufptr,tr);
	if (ttytest()) {
		printf("\r                          \r");
		fflush(stdout);
	}

	/* and update file */
	if (!sfsupdate(filename))
		error("backup error on '%s'",filename);

	/* ... that's all folks */
	exit(0);
}

/* return squared sample values */
float	sqsamp(sno,fid,buff,buffsize)
int	sno;
int	fid;
short	*buff;
int	buffsize;
{
	static int	first= -1;
	float		val;

	if ((first < 0) || (sno >= (first+buffsize))) {
		sfsread(fid,sno,buffsize,buff);
		first=sno;
	}

	val = buff[sno-first];
	return(val*val);
}

/* calculate rectangular window */
float	rwindow(buff,size)
float	*buff;
int	size;
{
	register float	sum=(float)0.0;

	while (size-- > 0)
		sum += *buff++;
	return(sum);
}

/* calculate hamming window */
float	hwindow(buff,size)
float	*buff;
int	size;
{
	register float	sum=(float)0.0;
	static int 	donewindow=0;
	static float	omega;
	static float	*hwin;
	int		i;
	float		*hw;

	/* first time caculate window */
	if (!donewindow) {
		hwin = (float *)calloc(size,sizeof(float));
		omega = (float)(8.0 * atan(1.0) / (size-1));
		for (i=0;i<size;i++)
			hwin[i] = (float)(0.54 - (0.46 * cos(i * omega)));
		donewindow=1;
/*
		printf("Hamming window=");
		for (i=0;i<size;i++) printf("%g ",hwin[i]);
		printf("\n");
*/
	}

	/* calculate windowed buffer */
	hw = hwin;
	while (size-- > 0)
		sum += *buff++ * *hw++;
	return(sum);
}

