/* voc19 -- JSRU style 19-channel filterbank */

/* M.A.Huckvale - November 1988 */

/* version 1.1
	- fix windowsize in header
*/
/* version 1.2 - September 1989
	- switch for 200 frames/sec
	- filter frequencies in params field
*/
/* version 1.3 - January 1990
	- fix window size error for fast frame
*/
/* version 2.0 - February 1996
	- replace filtering with general purpose routines
*/

#define PROGNAME "voc19"
#define PROGVERS "2.0"
char	*progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH VOC19 SFS1 UCL SFS
.SH NAME
voc19 - JSRU style 19-channel vocoder
.SH SYNOPSIS
.B voc19
(-i item) (-f) file
.SH DESCRIPTION
.I voc19
is a straightforward implementation of a 19-channel filterbank analyser using
two second-order section Butterworth bandpass filters spaced as in Holmes paper
published in IEE proceedings.
Energy smoothing done at 50Hz to give default 100Hz framerate.
.PP
There is no 
limit to the length of the input signal.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version number.
.TP 11
.BI -i item
Select input item.
.TP 11
.B -f
Faster frame rate.  Sample output at 200 frames/sec, filtering remains at
50Hz to avoid pitch pulses.
.SH INPUT ITEMS
.IP SP
Any speech signal sampled at greater than 8kHz.
.SH OUTPUT ITEMS
.IP CO 11
Vocoder energies.
.SH VERSION/AUTHOR
2.0 - Mark Huckvale
.SH SEE ALSO
voc8(1), voc26(1)
*/
/*--------------------------------------------------------------------------*/

/* standard definitions */
#include "SFSCONFG.h"
#include <stdio.h>		/* standard io library */
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"		/* database structures */
#include "filter.h"

/* manifest constants */
#define NCHAN 19		/* # channels */
int	window;			/* window size in samples */
int	fastframe=0;		/* faster frame rate */

/* channel filters */
FILTER	*chan[NCHAN];
FILTER	*schan[NCHAN];

/* design filters */
void design(srate,frate)
double	srate;
double	frate;
{
	int	i;

	chan[ 0] = filter_design(FILTER_BAND_PASS,4, 180, 300,srate);
	chan[ 1] = filter_design(FILTER_BAND_PASS,4, 300, 420,srate);
	chan[ 2] = filter_design(FILTER_BAND_PASS,4, 420, 540,srate);
	chan[ 3] = filter_design(FILTER_BAND_PASS,4, 540, 660,srate);
	chan[ 4] = filter_design(FILTER_BAND_PASS,4, 660, 780,srate);
	chan[ 5] = filter_design(FILTER_BAND_PASS,4, 780, 900,srate);
	chan[ 6] = filter_design(FILTER_BAND_PASS,4, 900,1075,srate);
	chan[ 7] = filter_design(FILTER_BAND_PASS,4,1075,1225,srate);
	chan[ 8] = filter_design(FILTER_BAND_PASS,4,1225,1375,srate);
	chan[ 9] = filter_design(FILTER_BAND_PASS,4,1375,1525,srate);
	chan[10] = filter_design(FILTER_BAND_PASS,4,1525,1700,srate);
	chan[11] = filter_design(FILTER_BAND_PASS,4,1700,1900,srate);
	chan[12] = filter_design(FILTER_BAND_PASS,4,1900,2100,srate);
	chan[13] = filter_design(FILTER_BAND_PASS,4,2100,2300,srate);
	chan[14] = filter_design(FILTER_BAND_PASS,4,2300,2550,srate);
	chan[15] = filter_design(FILTER_BAND_PASS,4,2550,2850,srate);
	chan[16] = filter_design(FILTER_BAND_PASS,4,2850,3150,srate);
	chan[17] = filter_design(FILTER_BAND_PASS,4,3150,3475,srate);
	chan[18] = filter_design(FILTER_BAND_PASS,4,3475,4000,srate);

	for (i=0;i<NCHAN;i++)
		schan[i] = filter_design(FILTER_LOW_PASS,2,frate/2,frate/2,srate);
	
}

/* filter processing */
void process(fid,window,numframes,ofid,co)
int		fid;		/* input file */
int		window;		/* input window size */
int		numframes;	/* # input frames */
int		ofid;		/* output file */
struct co_rec	*co;		/* output buffer */
{
	int	i;				/* sample */
	int	j;				/* channel */
	int	k;				/* window value */
	float	smrect;				/* smoothing filter output */
	int	cnt=1;				/* framecounter */
	int	winsize;			/* window size */
	short	*sp,*fsp,*rsp;

	/* get analysis window size */
	if (fastframe)
		winsize = window/2;
	else
		winsize = window;

	/* estimate # output frames */
	numframes /= winsize;

	sp = calloc(winsize,sizeof(short));
	fsp = calloc(winsize,sizeof(short));
	rsp = calloc(winsize,sizeof(short));

	/* process each window in turn */
	for (i=0;sfsread(fid,i,winsize,sp)==winsize;i+=winsize) {

		/* for each channel in turn */
		for (j=0;j<NCHAN;j++) {

			/* filter the window */
			filter_signal(chan[j],sp,fsp,winsize);

			/* rectify */
			for (k=0;k<winsize;k++)
				if (fsp[k]<0) fsp[k] = -fsp[k];

			/* smooth */
			filter_signal(schan[j],fsp,rsp,winsize);
				
			/* store energy for filter */
			smrect = rsp[winsize-1];
			if (smrect < 1) smrect = (float)0.5;
			co->data[j] = (float)(20.0 * log10(smrect));
			
		}

		/* write output record */
		co->posn=i;
		co->size=winsize;
		co->flag=0;
		co->mix=(float)0.0;
		co->gain=(float)1.0;
		if (sfswrite(ofid,1,co)!=1)
			error("write error on output file",NULL);

		/* report progress */
		if (ttytest()) {
			printf("\rFrame %d/%d",cnt++,numframes);
			fflush(stdout);
		}
	}

	free(sp);
	free(fsp);
	free(rsp);
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it=SP_TYPE;	/* item type selection */
	char		*ty="0";	/* item match selection */
	/* input file variables */
	char		filename[SFSMAXFILENAME]; /* sfs file name */
	int		fid;		/* input file descriptor */
	struct item_header spitem;	/* item header for speech data */
	/* output file variables */
	int		ofid;		/* output file descriptor */
	struct item_header coitem;	/* item header for vocoder data */
	struct co_rec	*co;
	double		srate;
	
	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:f")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: 19-channel vocoder V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it != SP_TYPE)
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'f' :	/* fast frame rate */
			fastframe++;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-f) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open data file */
	if ((fid=sfsopen(filename,"w",NULL)) < 0) {
		if (fid==-1)
			error("cannot find file: %s",filename);
		else
			error("access error on %s",filename);
	}

	/* locate input speech */
	if (!sfsitem(fid,it,ty,&spitem))
		error("cannot find SP data in %s",filename);

	/* design filters */
	srate = 1.0/spitem.frameduration;
	window = (int)((0.5+(0.01/spitem.frameduration)));
	if (srate < 8000)
		error("input signal sampled at too low a rate");
	design(srate,100.0);

	/* create output item header */
	if (fastframe) {
		sfsheader(&coitem,CO_TYPE,1,sizeof(float),
			19+sfsstruct[CO_TYPE]/sizeof(float),
			spitem.frameduration,spitem.offset,window/2,0,0);
		sprintf(coitem.history,"%s(%d.%02d;fastframe)",
				PROGNAME,spitem.datatype,spitem.subtype);
	}
	else {
		sfsheader(&coitem,CO_TYPE,1,sizeof(float),
			19+sfsstruct[CO_TYPE]/sizeof(float),
			spitem.frameduration,spitem.offset,window,0,0);
		sprintf(coitem.history,"%s(%d.%02d)",
				PROGNAME,spitem.datatype,spitem.subtype);
	}
	strcpy(coitem.params,"labels=240|360|480|600|720|840|1000|1150|1300|1450|1600|1800|2000|2200|2400|2700|3000|3300|3750");

	/* open output channel */
	if ((ofid=sfschannel(filename,&coitem)) < 0)
		error("cannot open output channel",NULL);

	/* get buffer */
	if ((co=(struct co_rec *)sfsbuffer(&coitem,1))==NULL)
		error("cannot get output buffer",NULL);

	/* do processing */
	process(fid,window,spitem.numframes,ofid,co);

	/* update data file */
	if (!sfsupdate(filename))
		error("update error on %s",filename);

	/* clear progress */
	if (ttytest()) {
		printf("\r                       \r");
		fflush(stdout);
	}

	/* that's all folks */
	exit(0);
}

