/* fxload -- load a Fx item from an ASCII list of numbers */

/* M.A.Huckvale - January 1994 */

/* version 1.0 */

/*-------------------------------------------------------------------*/
/**MAN
.TH FXLOAD SFS1 UCL
.SH NAME
fxload -- load a Fx item from a text file of numbers
.SH SYNOPSIS
.B fxload
(-f freq) (-h history) (-O timeoffset) textfile file
.SH DESCRIPTION
.I fxload
constructs a new fx item in the given file from text numbers
a text file.  The numbers can be separated by any white-space.
.PP
.B Options
.TP 11
.B -I
Identify the program version.
.TP 11
.BI -f freq
Use a sampling interval based on 1/freq seconds for sampling the time of each
fx value. Default 100. 
.TP 11
.BI -h history
Place a "history=" indicator in the created item header, with the argument
.I history.
.TP 11
.BI -O timeoffset
Set "offset" field in output item to given value.
.SH OUTPUT ITEMS
.IP FX 11
Loaded fx item
.SH VERSION/AUTHOR
1.0 - Mark Huckvale
*/
/*---------------------------------------------------------------*/

/* program name and version */
#define	PROGNAME "fxload"
#define PROGVERS "1.0s"
char	*progname = PROGNAME;

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "sfs.h"		/* database filing system structures */

/* global data */
struct item_header 	fxitem;			/* trcak item header  */
char			textname[SFSMAXFILENAME];	/* text file name */
char			filename[SFSMAXFILENAME];	/* dbase file name */
double			fdur=1.0E-2;		/* default sample duration */
char			*fxhist;		/* fx history */
double			offset=0;

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;			/* option index */
	extern char	*optarg;		/* option argument ptr */
	int		errflg = 0;		/* option error flag */
	int		c;			/* option switch */

	/* processing variables */
	FILE		*ip;			/* input text file */
	int		ofid;			/* output file descriptors */
	float		val;			/* input value */
	short		fxval;			/* FX value */
	double		srate,atof();

	/* decode switches */
	while ( (c = getopt(argc,argv,"If:h:O:")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Load fundamental frequency V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'f' :	/* select sampling frequency */
			srate = atof(optarg);
			if (srate<=0)
				error("bad sampling frequency: %s",optarg);
			else
				fdur = 1.0/srate;
			break;
		case 'h' :	/* history */
			fxhist=optarg;
			break;
		case 'O' :	/* offset */
			offset=atof(optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	/* check command line */
	if (errflg || (argc<3))
		error("usage: %s (-I) (-f freq) (-h fxhist) (-O timoffset) textfile file",PROGNAME);

	/* get text file name */
	if (optind < argc) 
		strcpy(textname,argv[optind]);
	else
		error("no text file specified",NULL);
	optind++;

	/* get data filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open text file */
	if ((ip=fopen(textname,"r"))==NULL)
		error("cannot open %s",textname);

	/* open data file */
	if ((ofid=sfsopen(filename,"w",NULL)) < 0) {
		if (ofid==-1)
			error("cannot find file %s",filename);
		else
			error("access error on %s",filename);
	}
	sfsclose(ofid);

	/* create output item header */
	sfsheader(&fxitem,FX_TYPE,0,2,1,fdur,offset,1,0,0);
	if (fxhist) 
		sprintf(fxitem.history,"%s(file=%s,rate=%g,offset=%g,history=%s)",PROGNAME,textname,1.0/fdur,offset,fxhist);
	else
		sprintf(fxitem.history,"%s(file=%s,rate=%g,offset=%g)",PROGNAME,textname,1.0/fdur,offset);

	/* open output channel */
	if ((ofid=sfschannel(filename,&fxitem)) < 0)
		error("unable to open temporary file",NULL);

	/* process input file */
	while (fscanf(ip,"%f",&val) == 1) {
		fxval = (short)val;
		if (sfswrite(ofid,1,&fxval) != 1)
			error("write error on output file",NULL);
	}

	/* update data file */
	if (!sfsupdate(filename))
		error("backup error on %s",filename);

	/* that's all folks */
	exit(0);

}
