/* cnv2sfs -- general purpose conversion of audio file to SFS */

/* Mark Huckvale - University College London */

/* version 1.0 - March 2001
	- uses File code from SOX
*/
/* version 1.1 - April 2001
	- add options for stereo load
*/

#define PROGNAME "cnv2sfs"
#define PROGVERS "1.2"
char *progname=PROGNAME;

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "sfs.h"
#include "st.h"
#ifdef SPHERE
#include <sp/sphere.h>
#endif

/*-------------------------------------------------------------------------*/
/**MAN
.TH CNV2SFS SFS1 UCL
.SH NAME
cnv2sfs -- general purpose audio file to SFS conversion
.SH SYNOPSIS
.B cnv2sfs
(-i itemtype|-C chantype) audiofile sfsfile
.SH DESCRIPTION
.I cnv2sfs
is a general purpose program for converting audiofiles
to SFS format.  It knows about a large number of file formats.
The basic input code is borrowed from the SOX application.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i itemtype
Specify item type in sfs file.  1=Speech, 2=Lx.  Default: Speech.
.TP 11
.BI -C chantype
Specify speech and/or Lx for stereo linkage.  Use -C11 for stereo
speech, -C12 for speech and Lx, -C21 for Lx and speech, -C22 for stereo Lx.
Default: stereo speech.
.SH OUTPUT ITEMS
.IP SP
Speech item.
.IP LX
Laryngograph item.
.SH HISTORY
.IP file
Input filename.
.IP channel
Channel number in input file.
.SH VERSION/AUTHOR
.IP 1.1
Mark Huckvale
.SH SEE ALSO
slink
*/

/* global data */
struct st_soundstream informat;

struct main_header	head;
struct item_header	spitem;
struct item_header	spitem2;
long			xbuff[4096];
short			sp[4096];
int			dostereo=11;

/* called from util.c:fail */
void cleanup()
{
	/* Close the input file before exiting */
	if (informat.fp)
		fclose(informat.fp);
}

/* main program */
void main(int argc,char **argv)
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	/* matching variables */
	int		it=SP_TYPE;
	char		*ty;
	/* file variables */
	char		ifilename[SFSMAXFILENAME]; /* SFS data file name */
	FILE		*ip;
	char		ofilename[SFSMAXFILENAME]; /* SFS data file name */
	int		ofid;		/* file descriptor */
	int		ofid2=0;		/* file descriptor */
	int		i,j,len;
#ifdef SPHERE
	char		hbuf[64];
	SP_FILE 	*sp_in;
	SP_INTEGER	smr,scount;
#endif

	/* decode switches */
	while ( (c = getopt(argc,argv,"IC:i:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Audio File Conversion to SFS V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) != 0)
				error("bad 'i' option : %s",optarg);
			break;
		case 'C' :	/* stereo channel types */
			dostereo=atoi(optarg);
			it = SP_TYPE;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i itemtype|-C chantype) audiofile sfsfile",PROGNAME);

	/* get in filename */
	if (optind < argc)
		strcpy(ifilename,sfsfile(argv[optind]));
	else
		error("no audio file specified",NULL);
	optind++;

	/* get out filename */
	if (optind < argc)
		strcpy(ofilename,sfsfile(argv[optind]));
	else
		error("no output file specified",NULL);

	/* open output file */
	if ((ofid=sfsopen(ofilename,"r",&head))<0) {
		/* file doesn't exist - try to create it */
		if ((ofid=sfsopen(ofilename,"c",NULL))<0)
			error("could not create file '%s'",ofilename);
		sfsclose(ofid);
	}
	else
		sfsclose(ofid);

	/* open input file */
	if ((ip=fopen(ifilename,"rb"))==NULL)
		error("could not open '%s'",ifilename);

#ifdef SPHERE
	/* do special processing for NIST/SPHERE */
	if ((fread(hbuf,1,7,ip)==7)&&(strncmp(hbuf,"NIST_1A",7)==0)) {

		fclose(ip);

		/* open file with SPHERE library routine */
		if ((sp_in=sp_open(ifilename,"r")) == SPNULL) {
			fprintf(stderr,"could not open '%s'\n",ifilename);
			exit(1);
		}

		/* get sampling rate */
		sp_h_get_field(sp_in,SAMPLE_RATE_FIELD,T_INTEGER,(void *)&smr);

		/* get # samples */
		sp_h_get_field(sp_in,SAMPLE_COUNT_FIELD,T_INTEGER,(void *)&scount);

		sp_set_data_mode(sp_in,"SE-PCM-2:SBF-01");

		sfsheader(&spitem,it,0,2,1,1.0/smr,0.0,1,0,0);
		sprintf(spitem.history,"%s(file=%s)",PROGNAME,ifilename);

		/* open an output channel */
		if ((ofid=sfschannel(ofilename,&spitem))<=0)
			error("could not open output channel to '%s'",ofilename);

		while (scount > 0) {
			len = sp_read_data(sp,(scount > 4096) ? 4096 : scount,sp_in);
			if (sfswrite(ofid,len,sp)!=len)
				error("output write error");
			scount -= len;
		}

		sp_close(sp_in);
	}
	else
#endif
	{
		rewind(ip);

		/* get input format */
		informat.info.rate      = 0;
		informat.info.size      = -1;
		informat.info.encoding  = -1;
		informat.info.channels  = -1;
		informat.swap      = 0;
		informat.fp        = ip;
		informat.seekable  = 1;
		informat.filename  = ifilename;
		informat.filetype = "auto";
		informat.comment = NULL;
		st_gettype(&informat);

		/* Read and write starters can change their formats. */
		if ((informat.h->startread)(&informat) == ST_EOF)
			error("Failed to read %s, error %s",ifilename,informat.st_errstr);
		st_checkformat(&informat);

		printf("Input file    : %s\n",ifilename);
		printf("Sampling Rate : %d\n",informat.info.rate);
		printf("Sample Size   : %s\n",st_sizes_str[informat.info.size]);
		printf("Sample Coding : %s\n",st_encodings_str[informat.info.encoding]);
		printf("No. Channels  : %d\n",informat.info.channels);
		if (informat.comment)
			printf("File Comment  : %s\n", informat.comment);

		/* create an output item header */
		if (informat.info.channels==2) {
			if (dostereo/10==2)
				sfsheader(&spitem,LX_TYPE,0,2,1,1.0/informat.info.rate,0.0,1,0,0);
			else
				sfsheader(&spitem,it,0,2,1,1.0/informat.info.rate,0.0,1,0,0);
			sprintf(spitem.history,"%s(file=%s,channel=1)",PROGNAME,ifilename);
		}
		else {
			sfsheader(&spitem,it,0,2,1,1.0/informat.info.rate,0.0,1,0,0);
			sprintf(spitem.history,"%s(file=%s)",PROGNAME,ifilename);
		}

		/* open an output channel */
		if ((ofid=sfschannel(ofilename,&spitem))<=0)
			error("could not open output channel to '%s'",ofilename);

		/* stereo */
		if ((informat.info.channels==2)&&(dostereo%10!=0)) {
			/* create an output item header */
			if (dostereo%10==2)
				sfsheader(&spitem2,LX_TYPE,0,2,1,1.0/informat.info.rate,0.0,1,0,0);
			else
				sfsheader(&spitem2,it,0,2,1,1.0/informat.info.rate,0.0,1,0,0);
			sprintf(spitem2.history,"%s(file=%s,channel=2)",PROGNAME,ifilename);

			/* open an output channel */
			if ((ofid2=sfschannel(ofilename,&spitem2))<=0)
				error("could not open output channel to '%s'",ofilename);
		}

		/* read the input data */
		while ((len=(informat.h->read)(&informat, xbuff, 4096))>0) {
			if (informat.info.channels==2) {
				for (i=0,j=0;i<len;i+=2,j++)
					sp[j]=(short)(xbuff[i] >> 16);
				if (sfswrite(ofid,len/2,sp)!=len/2)
					error("output write error");
				if (dostereo%10!=0) {
					for (i=1,j=0;i<len;i+=2,j++)
						sp[j]=(short)(xbuff[i] >> 16);
					if (sfswrite(ofid2,len/2,sp)!=len/2)
						error("output write error");
				}
			}
			else {
				for (i=0;i<len;i++)
					sp[i]=(short)(xbuff[i] >> 16);
				if (sfswrite(ofid,len,sp)!=len)
					error("output write error");
			}
		}
		fclose(ip);
	}

	if (!sfsupdate(ofilename))
		error("update failed on '%s'",ofilename);
	exit(0);
}
