/* beeper - generate pure-tone beeps to order */

/* Mark Huckvale - University College London */

/* version 1.0 - March 1996 */

#define PROGNAME "beeper"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH BEEPER SFS1 UCL 
.SH NAME
beeper - generate pure-tone beeps to order
.SH SYNOPSIS
.B beeper
(-f start (stop)) (-a start (stop)) (-n number) (-c controlfile) (-o sfsfile)
(-r samplerate) (-d duration) (-g gap) (-R ramptime) (-l) (-v)
.SH DESCRIPTION
.I beeper
is a program to generate sequences of pure-tone beeps to order.  Command
line options control the frequency, amplitude, number and timing of
a sequence of beeps.  The beep signal may optionally be saved in an
SFS file.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -f start (stop)
Specify the first and (optionally) last tone frequency in Hertz.  Default 1000Hz.
.TP 11
.BI -a start (stop)
Specify the first and (optionally) last tone amplitude in dB with respect to a
signal with a peak amplitude of 1.  Default 70dB.  Maximum 90dB.
.TP 11
.B -n number
Specify the number of tones to produce.  Default 1.
.TP 11
.BI -c controlfile
Specify the frequency and tone of each beep in a control file.  The file
consists of one line per beep, with the first number on the line
specifying the frequency in Hertz, and the second the amplitude in dB.
.TP 11
.BI -o sfsfile
Save the resulting signal into an SFS file instead of replaying.
Default is to replay.
.TP 11
.BI -r samplerate
Specify the output sampling rate.  Default 20000Hz.
.TP 11
.BI -d duration
Specify the duration of each beep in seconds.  Default 0.5 seconds.
.TP 11
.BI -g gapduration
Specify the duration of the gap between beeps in seconds.  Default 0.5 seconds.
.TP 11
.BI -R ramptime
Specify the time to ramp the amplitude at the ends of the beep in seconds.
Default 0.05 seconds.
.TP 11
.B -l
Use logarithmic frequency steps.  Default linear;
.TP 11
.B -v
Verbose.  Report details of signal generated.
.SH OUTPUT ITEMS
.IP SPEECH
Beep sequence.
.SH HISTORY
.IP fstart=
Starting frequency.
.IP fend=
Ending frequency.
.IP astart=
Starting amplitude.
.IP aend=
Ending amplitude.
.IP file=
Control filename.
.IP srate=
Sampling rate.
.IP dur=
Beep duration.
.IP gap=
Gap duration.
.IP ramp=
Ramp duration.
.SH VERSION/AUTHOR
.IP 1.0
Mark Huckvale
.SH SEE ALSO
steptone
*/
/*--------------------------------------------------------------------------*/

/* include files */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <malloc.h>
#include <ctype.h>
#include "sfs.h"

/* global data */
struct item_header	spitem;
short			*sp;
int			splen;

/* parameters */
double	fstart=1000.0;		/* starting frequency */
double	fend=1000.0;		/* ending frequency */
double	astart=70;		/* starting amplitude */
double	aend=70;		/* ending amplitude */
int	nbeep=1;		/* number of beeps */
char	cfilename[SFSMAXFILENAME];	/* control file name */
int	docontrol=0;		/* read control */
double	srate=20000.0;		/* sampling rate */
char	sfilename[SFSMAXFILENAME];	/* SFS file name */
int	dosave=0;		/* save to SFS file */
double	btime=0.5;		/* beep time */
double	bgap=0.5;		/* beep gap time */
double	bramp=0.05;		/* ramp time */
int	dologstep=0;		/* log steps */
int	verbose=1;

/* sine table */
#define SINETABLEN	4096
double	sinetab[SINETABLEN];

/* control data */
struct beep_rec {
	double	freq;		/* frequency */
	double	amp;		/* amplitude (db) */
	double	size;		/* amplitude (lin) */
	int	slen;		/* # signal samples */
	int	rlen;		/* # ramp samples */
	int	glen;		/* # gap samples */
} *btab;

/* count # lines in control file */
int	countlines(cfilename)
char *cfilename;
{
	FILE	*ip;
	int	n=0;
	double	f,a;

	if ((ip=fopen(cfilename,"r"))==NULL)
		error("could not open '%s'",cfilename);
	while (fscanf(ip,"%lf %lf\n",&f,&a)==2) n++;
	fclose(ip);
	return(n);
}

/* load control data into table */
int	loadlines(cfilename)
char *cfilename;
{
	FILE	*ip;
	int	n=0;
	double	f,a;

	if ((ip=fopen(cfilename,"r"))==NULL)
		error("could not open '%s'",cfilename);
	while (fscanf(ip,"%lf %lf\n",&f,&a)==2) {
		btab[n].freq = f;
		btab[n].amp = a;
		n++;
	}
	fclose(ip);
	return(n);
}

/* convert dB scale to linear */
double	dbtolin(dbval)
double dbval;
{
	return(pow(10.0,dbval/20.0));
}

/* make a sine table */
void	makesinetable()
{
	double	omega,step;
	int	i;

	step = 8.0*atan(1.0)/SINETABLEN;
	omega = 0.0;
	for (i=0;i<SINETABLEN;i++) {
		sinetab[i] = sin(omega);
		omega += step;
	}
}

/* make a beep */
int	makebeep(buf,brec)
short *buf;struct beep_rec *brec;
{
	int	i;
	int	corelen;
	double	step;
	double	pos=0;

	corelen = brec->slen - brec->rlen;
	step = brec->freq*SINETABLEN/srate;

	/* initial ramp up */
	for (i=0;i<brec->rlen;i++) {
		*buf++ = (short)(i * brec->size * sinetab[(int)pos] / brec->rlen);
		pos += step;
		if (pos >= SINETABLEN) pos -= SINETABLEN;
	}
	/* core */
	for (;i<corelen;i++) {
		*buf++ = (short)(brec->size * sinetab[(int)pos]);
		pos += step;
		if (pos >= SINETABLEN) pos -= SINETABLEN;
	}
	/* final ramp down */
	for (i=0;i<brec->rlen;i++) {
		*buf++ = (short)((brec->rlen-i) * brec->size * sinetab[(int)pos] / brec->rlen);
		pos += step;
		if (pos >= SINETABLEN) pos -= SINETABLEN;
	}
	/* silent gap */
	for (i=0;i<brec->glen;i++)
		*buf++ = 0;

	return(brec->slen+brec->glen);
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		i;
	int		pos;
	
	/* decode switches */
	while ( (c = getopt(argc,argv,"If:a:n:c:o:r:d:g:R:lv")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Make programmed beeps V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'f' :	/* frequency */
			fstart = atof(optarg);
			if (isdigit(argv[optind][0]))
				fend = atof(argv[optind++]);
			else
				fend = fstart;
			break;
		case 'a' :	/* amplitude */
			astart = atof(optarg);
			if (isdigit(argv[optind][0]))
				aend = atof(argv[optind++]);
			else
				aend = astart;
			break;
		case 'n' :	/* number */
			nbeep = atoi(optarg);
			break;
		case 'c' :	/* control file */
			strcpy(cfilename,optarg);
			docontrol=1;
			break;
		case 'o' :	/* SFS file */
			strcpy(sfilename,optarg);
			dosave=1;
			break;
		case 'r' :	/* sample rate */
			srate = atof(optarg);
			break;
		case 'd' :	/* duration */
			btime = atof(optarg);
			break;
		case 'g' :	/* gap */
			bgap = atof(optarg);
			break;
		case 'R' :	/* ramp */
			bramp = atof(optarg);
			break;
		case 'l' :	/* logarithmic */
			dologstep=1;
			break;
		case 'v' :	/* verbose */
			verbose=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg)
		error("usage: %s (-I) {(-f startfreq stopfreq) (-a startamp stopamp) (-n numbeep)|(-c controlfile)} (-o sfsfile) (-r samplerate) (-d duration) (-g gapduration) (-R rampduration) (-l) (-v)",PROGNAME);

	/* set up control structure */
	if (docontrol)
		nbeep = countlines(cfilename);
	if ((btab=(struct beep_rec *)calloc(nbeep,sizeof(struct beep_rec)))==NULL)
		error("could not get memory");
	if (docontrol)
		loadlines(cfilename);
	else {
		btab[0].freq = fstart;
		btab[0].amp = astart;
		btab[nbeep-1].freq = fend;
		btab[nbeep-1].amp = aend;
		if (dologstep==0) {
			for (i=1;i<nbeep-1;i++) {
				btab[i].freq = btab[0].freq + i*(btab[nbeep-1].freq-btab[0].freq)/(nbeep-1);
				btab[i].amp = btab[0].amp + i*(btab[nbeep-1].amp-btab[0].amp)/(nbeep-1);
			}
		}
		else {
			for (i=1;i<nbeep-1;i++) {
				btab[i].freq = exp(log(btab[0].freq) + i*(log(btab[nbeep-1].freq)-log(btab[0].freq))/(nbeep-1));
				btab[i].amp = btab[0].amp + i*(btab[nbeep-1].amp-btab[0].amp)/(nbeep-1);
			}
		}
	}
	splen=0;
	for (i=0;i<nbeep;i++) {
		btab[i].size = dbtolin(btab[i].amp);
		btab[i].slen = (int)(btime * srate);
		btab[i].rlen = (int)(bramp * srate);
		btab[i].glen = (int)((i==nbeep-1) ? 0 : bgap * srate);
		splen += btab[i].slen + btab[i].glen;
		if (verbose)
			printf("Tone %2d: Freq %7.1f Amp %5.1f Duration %5.3f Gap %5.3f\n",
				i+1,btab[i].freq,btab[i].amp,btime,bgap);
	}

	/* build sine look-up table */
	makesinetable();

	/* get buffer */
	if ((sp=(short *)calloc(splen,sizeof(short)))==NULL)
		error("could not get memory");

	/* make the tones */
	pos=0;
	for (i=0;i<nbeep;i++)
		pos += makebeep(&sp[pos],&btab[i]);

	/* replay or save */
	if (!dosave) {
		if (dac_open(NULL) >= 0) {
			dac_playback(sp,splen,srate,16,1,1);
			dac_close(0);
		}
		else
			error("could not open replay");
	}
	else {
		sfsheader(&spitem,SP_TYPE,0,1,2,1.0/srate,0.0,1,0,0);
		if (docontrol)
			sprintf(spitem.history,"%s(cfile=%s,srate=%g,dur=%g,gap=%g,ramp=%g)",
				PROGNAME,
				cfilename,srate,btime,bgap,bramp);
		else
			sprintf(spitem.history,"%s(fstart=%g,fend=%g,astart=%g,aend=%g,num=%d,srate=%g,dur=%g,gap=%g,ramp=%g)",
				PROGNAME,
				fstart,fend,astart,aend,nbeep,
				srate,btime,bgap,bramp);

		putitem(sfilename,&spitem,splen,sp);
	}
	free(sp);
	free(btab);
	exit(0);
}
