/* compand -- perform companding on speech signals */

/* M.A.Huckvale - University College London */

/* version 1.0 - March 1996 */

#define PROGNAME "compand"
#define PROGVERS "1.1"
char	*progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH COMPAND SFS1 UCL
.SH NAME
compand -- compand (compress dynamic range) speech signals
.SH SYNOPSIS
.B compand
(-r dbrange) (-c mu-value) (-i item) file
.SH DESCRIPTION
.I compand
is a program to demonstrate the effects of dynamic range
compression on speech signals.  The normal dynamic range of a
16-bit signal (90dB) can be compressed by non-linear amplitude
mapping either (i) to a smaller dynamic range, or (ii) to a
mu-law compression function: outmag = log(1+mu*inpmag)/log(1+mu).
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select input item.
.TP 11
.BI -r dbrange
Select dynamic range compression.  Default 90dB.  Maximum 90dB.
.BI -c mu-value
Select mu-law compression.  Mild compression mu=10. Strong compression
mu=100.
.SH INPUT ITEMS
.IP 1.xx 11
Any speech item.
.SH OUTPUT ITEMS
.IP SPEECH
Companded signal.
.SH HISTORY
.IP dbrange=
Output dynamic range.
.IP mu=
Mu-law compression value.
.SH VERSION/AUTHOR
1.1 - Mark Huckvale.
*/
/*--------------------------------------------------------------------------*/

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"			/* header structures */

/* manifest constants */
#define MAXRANGE	90
#define BUFSIZE		4096

/* global data */
struct item_header spitem;	/* input item header data */
struct item_header opitem;	/* output item header */
short	buff[BUFSIZE];

/* parameters */
double	oprange=90;		/* output range */
double	iprange;		/* input range */
int	ipmag;			/* input magnitude */
int	mucomp=0;		/* select mu-law compression */
double	mu=1.0;			/* mu-law value */
short	*ampmap;		/* amplitude map table */

/* build amplitude map */
void makemap(ipr,opr)
double ipr;double opr;
{
	int	i;
	double	ival,oval;

	if ((ampmap=(short *)calloc(32769,sizeof(short)))==NULL)
		error("could not get memory");

	ampmap[0]=0;
	if (mucomp) {
		for (i=1;i<=32768;i++) {
			ampmap[i] = (short)(ipmag*log(1.0+mu*(double)i/ipmag)/log(1.0+mu));
		}
	}
	else {
		for (i=1;i<=32768;i++) {
			ival = 20.0*log10((double)i);
			oval = (MAXRANGE-oprange) + ival*opr/ipr;
			ampmap[i] = (short)pow(10.0,oval/20.0);
		}
	}
}

/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
	/* local variables */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument */
	int		errflg=0;	/* option error flag */
	int		c;		/* option char */
	int		it;
	char		*ty;
	char		*sptype="0";
	char		filename[SFSMAXFILENAME];
					/* database file name */
	int		fid,ofid;
	int		i,j,len;
	short		smax=0;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:r:c:")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Compand waveform V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* item spec */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					sptype = ty;
				else
					error("unsuitable item specification %s",optarg);
			}
			else
				error("illegal item specification %s",optarg);
			break;
		case 'r' :	/* db range */
			oprange = atof(optarg);
			mucomp=0;
			break;
		case 'c' :	/* mu-law compression */
			mu = atof(optarg);
			mucomp=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-r dbrange) (-c mu-value) file\n",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open file */
	if ((fid=sfsopen(filename,"w",NULL)) < 0) {
		if (fid==-1)
			error("unable to find file '%s'",filename);
		else
			error("access error on '%s'",filename);
	}

	/* locate input item */
	if (!sfsitem(fid,SP_TYPE,sptype,&spitem))
		error("unable to find input item in '%s'",filename);

	/* create output item header */
	sfsheader(&opitem,spitem.datatype,spitem.floating,
			spitem.datasize,spitem.framesize,
			spitem.frameduration,spitem.offset,
			spitem.windowsize,spitem.overlap,spitem.lxsync);
	if (mucomp)
		sprintf(opitem.history,"%s(%d.%02d;mu=%g)",
			PROGNAME,
			spitem.datatype,spitem.subtype,mu);
	else
		sprintf(opitem.history,"%s(%d.%02d;dbrange=%g)",
			PROGNAME,
			spitem.datatype,spitem.subtype,oprange);

	/* open output channel */
	if ((ofid=sfschannel(filename,&opitem)) < 0)
		error("unable to open output file",NULL);

	/* first pass to get input range */
	for (i=0;(len=sfsread(fid,i,BUFSIZE,buff)) > 0;i+=len) {
		for (j=0;j<len;j++) {
			if (buff[j] > smax) smax = buff[j];
			if (buff[j] < -smax) smax = -buff[j];
		}
	}
	ipmag = smax;
	iprange = 20.0*log10((double)smax);

	/* build amplitude map */
	makemap(iprange,oprange);

	/* process file */
	for (i=0;(len=sfsread(fid,i,BUFSIZE,buff)) > 0;i+=len) {

		/* map amplitude */
		for (j=0;j<len;j++) {
			if (buff[j] < 0)
				buff[j] = -ampmap[-buff[j]];
			else
				buff[j] = ampmap[buff[j]];
		}

		/* write out result of processing */
		if (sfswrite(ofid,len,buff) != len)
			error("write error on output file",NULL);
	}

	/* update file */
	if (!sfsupdate(filename))
		error("update error on %s",filename);

	/* ... that's all folks */
	free(ampmap);
	exit(0);
}

