/* Esform - interactive LPC spectrum display program */

/* M.A.Huckvale - University College London */

/* version 1.0 - October 1996 */

#define PROGNAME "Esform"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH ESFORM UCL1 UCL SFS
.SH NAME
Esform - interactive LPC spectrum display
.SH SYNOPSIS
.B Esform
(-I) (-p) (-s starttime|-S startsamp) (-e endtime|-E endsamp)
(-l lefttime|-L leftsamp) (-r righttime|-R rightsamp) (-t print_title) (-i item|-g item) (-12)
(-n ncoeff) sfsfile
.SH DESCRIPTION
.I Esform
is a program to display LPC spectrum cross-sections of regions of speech waveforms.
The program displays the input waveform at the top of the screen, and an LPC spectrum
at the bottom.  A single cursor or two cursors may be placed on the waveform, and
the LPC spectrum at the cursor or between the two cursors is calculated and displayed.
When one cursor is used, a window of 20ms is calculated; a hamming window is always
applied to the selected part of the signal.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i item
Specify input Speech item for waveform display.  Default: last speech item.
.TP 11
.BI -g item
Specify input Speech item for spectrographic display.  Default: last speech item.
.TP 11
.B -p
Direct output to printer.  The program does not enter interactive mode.  The
screen image is recreated for the printer using other command-line options.
.TP 11
.BI -s starttime
Specify start time for spectrum in seconds.  Default 0.0;
.TP 11
.BI -S startsamp
Specify start time for spectrum in samples.
.TP 11
.BI -e endtime
Specify end time for spectrum in seconds.  Default: end.
.TP 11
.BI -E endsamp
Specify end time for spectrum in samples.
.TP 11
.BI -l lefttime
Specify time for left cursor in seconds.  Default 0.0;
.TP 11
.BI -L leftsamp
Specify time for left cursor in samples.
.TP 11
.BI -e righttime
Specify time for right cursor in seconds.  Default: end.
.TP 11
.BI -E rightsamp
Specify time for right cursor in samples.
.TP 11
.BI -t print_title
Specify a title to appear on the printout.  If no title is given then the
date and time is printed instead.
.TP 11
.B -12
Identify 'unmarked' waveforms as 12-bits.
.TP 11
.BI -n ncoeff
Specify # coefficients to use for LPC analysis.  Default: 2+samprate/1000.
.SH "MAIN MENU"
.IP zoom 11
Zoom into selected region of the signal.
.IP up 11
Zoom out back to previous region of the signal.
.IP left 11
Scroll to view the immediately previous time.
.IP right 11
Scroll to view the immediately later time.
.IP top 11
Display the whole waveform.
.IP print 11
Prints display.  Requests a print title before printing in background.
.IP quit 11
Exits program.
.SH "MOUSE COMANDS"
.SS "ON WAVEFORM"
.IP left 11
Positions left cursor.
.IP middle 11
Replays waveform between cursors.
.IP right 11
Positions right cursor.
.SS "ON SPECTRUM"
.IP any 11
Calculates and draws spectrum from waveform between current cursors.
.SH VERSION/AUTHOR
.IP 1.0
Mark Huckvale
*/
/*--------------------------------------------------------------------------*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <malloc.h>
#include <time.h>
#include "sfs.h"
#include "dig.h"
#include "digdata.h"
#include "digitem.h"
#include "dgraph.h"
#include "complex.h"
#include "lsys.h"
#include "lpca.h"
#include "root.h"

#ifdef __STDC__
void 	esmenu_display(int,char **,char **,int);
void 	esmenu_redisplay(void);
int 	esmenu_check(float,float,int);
int 	esmouse_checkp(int,int,int);
#else
void 	esmenu_display();
void 	esmenu_redisplay();
int 	esmenu_check();
int 	esmouse_checkp();
#endif

/* graphics configuration */
#define Y0	0.0		/* bottom of screen */
#define Y1	0.05		/* bottom of spectrum */
#define Y2	0.6		/* top of spectrum */
#define Y3	0.61		/* bottom of waveform */
float	Y4;			/* top of waveform */
float	Y5;			/* top of timescale */

#define X0	0.0		/* left hand side of screen */
#define X1	0.10		/* left hand size of spectrum */
#define X2	0.80		/* right hand side of spectrum */
#define X4	0.85		/* formant freq text */
#define X3	1.0		/* right hand size of screen */
int	NSPECTPOINT;

#define inarea(x,y,xl,yb,xr,yt) (((x) >= (xl)) && ((x) < (xr)) && ((y) >= (yb)) && ((y) < (yt)))

float	charwidth,charheight;		/* character width and height in screen units */

/* configuration */
#define WIDE_SPECT_SIZE		0.005
#define NARROW_SPECT_SIZE	0.035
double	DEFAULT_SPECT_SIZE=WIDE_SPECT_SIZE;
#define MAXCOEFF	50	/* max # LPC coefficients */
#define	PREEMP		0.95	/* pre-emphasis coeff */
#define MAXWINDOW	2205	/* maximum analysis window size = 50ms @ 44.1kHz */
#define FREQRESPSIZE	1000

/* mouse commands */
#define LEFT	0
#define RIGHT	1
#define OFF	0
#define ON	1

#define MIN(x,y)	(((x)<(y))?(x):(y))
#define MAX(x,y)	(((x)>(y))?(x):(y))

/* global data */
int	laser=0;			/* laser - one shot to printer */
char	laser_title[80];		/* title for laser output */
int	bits12=0;			/* 12-bit signals */
int	lpcoeffs=0;			/* # LPC coeffs */
int	dospect=0;			/* display spectrogram */
int	donarrow=0;			/* narrow band */
char	*gargv0;

/* SFS data */
char	sfsfilename[SFSMAXFILENAME];		/* SFS file */
struct item_header	item;		/* input item header */
short			*sp;		/* input waveform */

/* menu and menu keys */
char	*menu_labels[]={
	"Quit", "Hard", "Play", "Clear", "Right", "Left", "Top", "Up", "Zoom", "Form",
};
char	*menu_keys[]={
	"Qq\033", "Hh", "Pp", "Cc", "Rr", "Ll", "Tt", "Uu", "Zz", "Ff",
};
	
/* commands */
#define COM_NULL	0
#define COM_QUIT	1
#define COM_ZOOM_IN	2
#define COM_ZOOM_OUT	3
#define COM_SCROLL_LEFT	4
#define COM_SCROLL_RIGHT	5
#define COM_RESTART	6
#define COM_REPLAY	7
#define COM_PRINT	8
#define COM_LEFT_CURS	9
#define COM_RIGHT_CURS	10
#define COM_SPECTRUM	11
#define COM_CLEAR	12
#define COM_FORMAT	13
#define COM_REFRESH	14

/* position stack */
#define ZOOM_DEPTH	10
struct {
	double	stime;
	double	etime;
} zoom[ZOOM_DEPTH];
int	zsp;
double	starttime = -1,endtime = -1;
int	startsamp = -1,endsamp = -1;
double	lefttime = -1,righttime = -1;
int	leftsamp = -1,rightsamp = -1;
double	ltime,rtime;

double mylog10(val)
double val;
{
	if (val < 1.0e-5)
		return(-5);
	else
		return(log10(val));
}

char *logname()
{
	static char *UNKNOWN="unknown";
	char *p,*getenv();

	if ((p=getenv("LOGNAME"))==NULL)
		return(UNKNOWN);
	else
		return(p);
}

void clear_top_line()
{
	int	oldmode;
	digflush();
	oldmode = diglinemode(0);
	digfillbox(0,0.0,Y5,1.0,1.0);
	digline(24,0.0,Y5,1.0,Y5);
	digflush();
	diglinemode(oldmode);
}

void hardcopy()
{
	char	comstr[256];
	float	x,y;
	char	b;

	clear_top_line();
	digprompt("Enter title for print: ");

	diggetmouse(laser_title,&b,&x,&y);

	if (laser_title[0] && (laser_title[0]!='\n') && (laser_title[0]!='\r')) {
		clear_top_line();
		digprompt("Printing ..");
#ifdef WIN32
		sprintf(comstr,"start /min %s -p -%c%d.%02d -S%d -E%d -L%d -R%d -t \"%s\" \"%s\"",
#else
		sprintf(comstr,"%s -p -%c%d.%02d -S%d -E%d -L%d -R%d -t \"%s\" %s &",
#endif
				gargv0,
				(dospect)?'g':'i',
				item.datatype,item.subtype,
				(int)(0.5+zoom[zsp].stime/item.frameduration),
				(int)(0.5+zoom[zsp].etime/item.frameduration),
				(int)(0.5+ltime/item.frameduration),
				(int)(0.5+rtime/item.frameduration),
				laser_title,sfsfilename);
		system(comstr);
		sleep(2);
	}

	clear_top_line();
}

/* draw a waveform */
void drawwaveform(xl,xr,yb,yt,tyb,tyt,item,wave,title)
float	xl,xr;		/* x-coords of box */
float	yb,yt;		/* y-coords of waveform */
float	tyb,tyt;	/* y-coords of timescale */
struct item_header *item;
short	*wave;
char	*title;
{
	int	flags;
	char	messg[80];
	
	digscale(1.0,1.0,0);

	/* display timescale */
	if (tyt > tyb)
		digitemtime(23,xl,tyb,xr,tyt,
			zoom[zsp].stime,zoom[zsp].etime,1);

	flags=DIGITEMBOX | DIGITEMTITLE | DIGITEMLABEL;
	digitab.title=title;
	if (dospect && donarrow)
		strcpy(messg,"Narrow-band");
	else if (dospect)
		strcpy(messg,"Wide-band");
	else
		strcpy(messg,"Waveform");
	digitab.label = messg;
	
	if (dospect) {
		if (donarrow)
			digitemFFT(242420,xl,yb,xr,yt,item,wave,
				zoom[zsp].stime,zoom[zsp].etime,flags|DIGITEMFFTNARROW);
		else
			digitemFFT(242420,xl,yb,xr,yt,item,wave,
				zoom[zsp].stime,zoom[zsp].etime,flags);
	}
	else
		digitemSP(232320,xl,yb,xr,yt,item,wave,
			zoom[zsp].stime,zoom[zsp].etime,flags);

	digflush();
}

/* spectral data */
float	*fsp;
int	fsize;
float	spectrum[FREQRESPSIZE];
float	fraxis[2];
float	logmaxmag=25.0;
int	colour=23;
COMPLEX	poly[MAXPOLY];
COMPLEX root[MAXPOLY];
float	fmfreq[MAXPOLY];
float	fmband[MAXPOLY];
int	npeak;

#define FREQLIMIT	100	/* Hz limit for a formant freq */
#define BANDLIMIT	3000	/* Hz limit for a formant bandwidth */
#define NFORMANT	7

void findroots(double *lpcoeff,int degree,double sdur)
{
	int	i,j;
	double	twopi,omega,rsq;
	double	ftemp,btemp;

	/* copy LPC coefficients into polynomial */
	poly[0].re = 1.0; poly[0].im = 0.0;
	for (i=1;i<=degree;i++) {
		poly[i].re = lpcoeff[i];
		poly[i].im = 0.0;
	}

	/* do root-solving using DSP library */
	memset(root,0,sizeof(root));
	AllRoots(poly,degree,root);

	/* convert roots to pole positions */
	twopi = 8.0*atan(1.0);
	npeak=0;
	for (i=0;i<=degree;i++) {
/* printf("(%g,%g) ",root[i].re,root[i].im); */
		if (root[i].im > 1.0E-10) {
			omega = atan2(root[i].im,root[i].re);
	 		fmfreq[npeak] = fabs(omega/(twopi*sdur));
	 		rsq = root[i].re*root[i].re+root[i].im*root[i].im;
			fmband[npeak] = fabs(log(rsq)/(2*twopi*sdur));
/* printf("raw freq=%g band=%g\n",fmfreq[npeak],fmband[npeak]); */
			if ((fmfreq[npeak] > FREQLIMIT) && (fmband[npeak] < BANDLIMIT)) npeak++;
		}
	}

	/* sort pole positions in terms of bandwidth */
	for (i=1;i<npeak;i++) {
		j = i;
		ftemp = fmfreq[i];
		btemp = fmband[i];
		while ((j>0) && (btemp < fmband[j-1])) {
			fmfreq[j]=fmfreq[j-1];
			fmband[j]=fmband[j-1];
			j--;
		}
		fmfreq[j]=ftemp;
		fmband[j]=btemp;
	}

	/* sort first NFORMANT pole positions in terms of frequency */
	for (i=1;(i<NFORMANT) && (i<npeak);i++) {
		j = i;
		ftemp = fmfreq[i];
		btemp = fmband[i];
		while ((j>0) && (ftemp < fmfreq[j-1])) {
			fmfreq[j]=fmfreq[j-1];
			fmband[j]=fmband[j-1];
			j--;
		}
		fmfreq[j]=ftemp;
		fmband[j]=btemp;
	}
	npeak = i;

/*	for (i=0;i<npeak;i++) printf("F%d=%.1fHz\n",i+1,fmfreq[i]); */
}

void dolpc(sp,N,fresp)
short	*sp;		/* input speech signal */
int 	N;		/* # samples */
float	*fresp;		/* frequency response squared magnitude */
{
	int		i;
	float		*xp;
	float		hc;
	LTIState	ltis;
	double		pe;
	double		mag;

	if (N > fsize) {
		if (fsize==0)
			fsp = calloc(N,sizeof(float));
		else
			fsp = realloc(fsp,N*sizeof(float));
		fsize = N;
	}
	
	/* put speech data into floating point buffer */
	hc = 8.0*atan(1.0)/(N-2);
	xp = fsp;
	sp++;
	srand(0);
	for (i=1;i<N;i++,sp++,xp++) {
		/* start with low-level random noise - to help stabilise analysis */
		*xp = (float)(rand()&0x7FFF) / 32768.0 - 0.5;
		/* add in signal */
		*xp += (*sp - PREEMP*(*(sp-1))) * (0.54 - (0.46 * cos(hc * (i-1))));
	}

	/* do autocorrelation analysis */
	LPCAutocorrelation(fsp,N-1,lpcoeffs,&ltis,&pe);

	/* return spectrum */
	for (i=0;i<FREQRESPSIZE;i++) {
		mag = 20.0*mylog10(LTISystemResponse(&ltis,0.5*(double)i/(double)FREQRESPSIZE));
		if (mag > logmaxmag) logmaxmag = mag;
		*fresp++ = mag;
	}

	findroots(ltis.b,lpcoeffs,item.frameduration);
}

/* calculate spectrum */
void calcspectrum(sp,sdur,stime,etime)
short	*sp;
double	sdur;
double	stime;
double	etime;
{
	int	ssamp,scount;
	double	tmp;

	if (etime < stime) {
		tmp = stime;
		stime = etime;
		etime = tmp;
	}
	ssamp = (int)(0.5+stime/sdur);
	scount = (int)(0.5+(etime-stime)/sdur);
	if (scount <= lpcoeffs) scount=lpcoeffs+1;

	dolpc(sp+ssamp,scount,spectrum);
}

/* draw spectrum axes */
void drawspectrumaxes(srate,xl,yb,xr,yt)
double	srate;
float	xl,yb,xr,yt;
{
	float	y[2];
	float	xresize;
	float	yresize;

	/* clear area of graph */
	digscale(1.0,1.0,0);
	digfillbox(0,xl,yb,xr,yt);

	/* change scaling */
	xresize = (1.0)/(xr-xl);	/* get half srate across x-axis */
	yresize = 1.0/(yt-yb);		/* get 50dB down y-axis */
	digscale(xresize,yresize,0);
	digorigin(xl*xresize,yb*yresize);

	/* do axes */
	fraxis[0] = 0.0;
	fraxis[1] = srate/2;
	y[0] = logmaxmag-57.0;
	y[1] = logmaxmag+3;

	dgraph((short *)fraxis,(short *)y,2,DGfloat+DGlowax+DGgrid,DGfloat+DGlowax,DGaxonly,23242423,
		NULL,"Frequency (Hz)","Amplitude (dB)");

	colour=23;
}

/* draw spectrum curve */
void drawspectrum(srate,xl,yb,xr,yt)
double	srate;
float	xl,yb,xr,yt;
{
	float		xresize;
	float		yresize;

	xresize = (1.0)/(xr-xl);
	yresize = 1.0/(yt-yb);
	digscale(xresize,yresize,0);
	digorigin(xl*xresize,yb*yresize);

	dgraph((short *)fraxis,(short *)spectrum,FREQRESPSIZE,DGfloat+DGsame+DGnext,
			DGfloat+DGsame,DGline,colour,NULL,NULL,NULL);
	colour++;
	if (colour >= 38) colour=23;

	digflush();

}

/* draw formant frequencies */
void drawformant(xl,yb,xr,yt)
float	xl;
float	yb;
float	xr;
float	yt;
{
	int	i;
	char	buf[80];

	digscale(1.0,1.0,0);
	digorigin(0.0,0.0);
	yt -= 5*charheight;
	digfillbox(0,xl,yb,xr,yt);
	yt -= 1.5*charheight;
	digtext(20,xl,yt,"Peaks:");
	yt -= 1.5*charheight;
	for (i=0;i<npeak;i++) {
		sprintf(buf,"F%d = %4d Hz",i+1,(int)fmfreq[i]);
		digtext(20,xl,yt,buf);
		yt -= 1.5*charheight;
	}
}

#define CURSCOL 23
int getcommand()
{
	int	com=COM_NULL;
	float	xm,ym;
	static int obut=0;
	int	but;
	int	ch;
	float	xl;
	float	x;
	double	ctim;

	digscale(1.0,1.0,0);
	
	digitab.ixoff = AXCHAR * digdata.chwidth + 3;
	xl = X0 + digitab.ixoff/digdata.xscale;

	/* get scale factor */
	digitab.scalex = (X3-xl)/(zoom[zsp].etime-zoom[zsp].stime);

	ctim = zoom[zsp].stime;
	do {
		esmenu_redisplay();
		
		do {
			do {
				ch=diggetmousech(&but,&xm,&ym);
			} while ((ch==DIG_MOUSE) && (but==obut));

			obut=but;

		} while ((ch==DIG_MOUSE) && (but == 0));

		if (ch=='+')
			com = COM_REPLAY;
		else if (ch==DIG_REDRAW)
			com = COM_REFRESH;
		else if ((com = esmenu_check(xm,ym,ch))>=0) {
			switch (com) {
			case 9:	com=COM_FORMAT; break;
			case 8:	com=COM_ZOOM_IN; break;
			case 7: com=COM_ZOOM_OUT; break;
			case 6: com=COM_RESTART; break;
			case 5: com=COM_SCROLL_LEFT; break;
			case 4: com=COM_SCROLL_RIGHT; break;
			case 3: com=COM_CLEAR; break;
			case 2: com=COM_REPLAY; break;
			case 1: com=COM_PRINT; break;
			case 0: com=COM_QUIT; break; 
			default: com=COM_NULL; break;
			}
		}
		else if (but==4) {
			if (inarea(xm,ym,X1,Y1,X2,Y2))
				com = COM_SPECTRUM;
			else if (inarea(xm,ym,xl,Y3,X3,Y4)) {
				com = COM_LEFT_CURS;
				ctim = zoom[zsp].stime + (xm-xl)/digitab.scalex;
				digflush();
				diglinemode(1);
				if (ltime > zoom[zsp].stime) {
					x = xl + (ltime-zoom[zsp].stime)*digitab.scalex;
					digline(CURSCOL,x,Y3,x,Y4);
					digflush();
				}
				if (ltime==ctim)
					ctim = zoom[zsp].stime;
				else {
					x = xl + (ctim-zoom[zsp].stime)*digitab.scalex;
					digline(CURSCOL,x,Y3,x,Y4);
					digflush();
				}
				ltime = ctim;
				digflush();
				diglinemode(0);
			}
			else {
				putchar('\007');
				fflush(stdout);
			}
		}
		else if (but==2) {
			if (inarea(xm,ym,X0,Y3,X3,Y4)) {
				com = COM_REPLAY;
			}
			else {
				putchar('\007');
				fflush(stdout);
			}
		}
		else if (but==1) {
			if (inarea(xm,ym,X1,Y1,X2,Y2))
				com = COM_CLEAR;
			else if (inarea(xm,ym,xl,Y3,X3,Y4)) {
				com = COM_RIGHT_CURS;
				ctim = zoom[zsp].stime + (xm-xl)/digitab.scalex;
				digflush();
				diglinemode(1);
				if (rtime < zoom[zsp].etime) {
					x = xl + (rtime-zoom[zsp].stime)*digitab.scalex;
					digline(CURSCOL,x,Y3,x,Y4);
					digflush();
				}
				if (rtime==ctim)
					ctim = zoom[zsp].etime;
				else {
					x = xl + (ctim-zoom[zsp].stime)*digitab.scalex;
					digline(CURSCOL,x,Y3,x,Y4);
					digflush();
				}
				rtime = ctim;
				digflush();
				diglinemode(0);
			}
			else {
				putchar('\007');
				fflush(stdout);
			}
		}
	} while (com==COM_NULL);

	return(com);
}

#ifdef _MSC_VER
char *ttyname()
{
	static char * str="console";
	return(str);
}
#endif

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */

	/* SFS file access */
	int		ipitem=SP_TYPE;
	char		*iptype="0";

	/* printing */
	char	messg[256],*logname(),*ttyname();
	time_t	tim;
	char	timbuf[32];
	float	x,xl;

	/* control */
	int	com;
	int	doplay;
	double	ctim;

	gargv0 = argv[0];
	
	/* decode switches */
	while ( (c = getopt(argc,argv,"Ipi:g:s:S:e:E:l:L:r:R:t:1:n:N")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Interactive smoothed-spectrum display V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'p' :	/* one shot to printer */
			laser++;
			break;
		case 'i' :	/* input item selection */
			if (itspec(optarg,&ipitem,&iptype)==0) {
				if ((ipitem==SP_TYPE) || (ipitem==LX_TYPE))
					/* OK */;
				else
					error("input item must be waveform");
			}
			else
				error("input selection error");
			break;
		case 'g' :	/* input item selection */
			if (itspec(optarg,&ipitem,&iptype)==0) {
				if ((ipitem==SP_TYPE) || (ipitem==LX_TYPE))
					/* OK */;
				else
					error("input item must be waveform");
			}
			else
				error("input selection error");
			dospect=1;
			break;
		case 't' :	/* print title */
			strncpy(laser_title,optarg,80);
			break;
		case 's' :	/* start time */
			starttime = atof(optarg);
			break;
		case 'e' :	/* end time */
			starttime = atof(optarg);
			break;
		case 'S' :	/* start sample */
			startsamp = atoi(optarg);
			break;
		case 'E' :	/* end sample */
			endsamp = atoi(optarg);
			break;
		case 'l' :	/* left cursor time */
			lefttime = atof(optarg);
			break;
		case 'r' :	/* right cursor time */
			righttime = atof(optarg);
			break;
		case 'L' :	/* left cursor sample */
			leftsamp = atoi(optarg);
			break;
		case 'R' :	/* right cursor sample */
			rightsamp = atoi(optarg);
			break;
		case '1' :	/* 12-bit waveform */
			if (strcmp(optarg,"2")==0)
				bits12=1;
			else
				error("unknown switch -1%s",optarg);
			break;
		case 'n' :
			lpcoeffs = atoi(optarg);
			if ((lpcoeffs < 2) || (lpcoeffs > MAXCOEFF))
				error("number coeffs between 2 and %d",MAXCOEFF);
			break;
		case 'N' :	/* narrow spectrogram */
			donarrow=1;
			dospect=1;
			DEFAULT_SPECT_SIZE = NARROW_SPECT_SIZE;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc < 2))
		error("usage: %s (-I) (-p) (-s starttime|-S startsamp) (-e endtime|-E endsamp) (-l leftctime|-L leftsamp) (-r righttime|-R rightsamp) (-t title) (-i item) (-12) (-n ncoeff) sfsfile",PROGNAME);

	/* get SFS filename */
	if (optind < argc)
		strcpy(sfsfilename,sfsfile(argv[optind]));
	else
		error("no input file specified");

	/* load input speech */
	getitem(sfsfilename,ipitem,iptype,&item,(void **)&sp);
	if (lpcoeffs == 0)
		lpcoeffs = (int)(2 + 1.0/(1000*item.frameduration));

	/* open graphics */
	digstart((char)((laser)?DIG_DEFAULT_PRINTER:DIG_DEFAULT_TERM),NULL,1);
	digscale(1.0,1.0,0);
	digclearscreen();
	charwidth = digdata.chwidth/digdata.xscale;
	charheight = digdata.chheight/digdata.yscale;
	NSPECTPOINT = (2*digdata.nhoriz)/3;
	
	/* initialise Graphics positions */
	Y5 = 1.0 - 2*charheight;	/* top of timescale */
	Y4 = Y5 - 2*charheight;		/* top of waveform */

	/* initialise position stack */
	if ((startsamp >= 0) && (startsamp < item.numframes))
		zoom[0].stime = startsamp*item.frameduration;
	else if ((starttime >= 0) && (starttime < item.numframes*item.frameduration))
		zoom[0].stime = starttime;
	else
		zoom[0].stime = 0.0;
	if ((endsamp >= 0) && (endsamp < item.numframes))
		zoom[0].etime = endsamp*item.frameduration;
	else if ((endtime >= 0) && (endtime < item.numframes*item.frameduration))
		zoom[0].etime = endtime;
	else
		zoom[0].etime = item.numframes*item.frameduration;
	zoom[1] = zoom[0];
	zsp = 1;

	/* sort out cursor positions */
	if ((leftsamp >= zoom[0].stime/item.frameduration) &&
 	    (leftsamp < zoom[0].etime/item.frameduration))
		ltime = leftsamp*item.frameduration;
	else if ((lefttime >= zoom[0].stime) && (lefttime < zoom[0].etime))
		ltime = lefttime;
	else
		ltime = zoom[0].stime;
	if ((rightsamp >= zoom[0].stime/item.frameduration) &&
	    (rightsamp < zoom[0].etime/item.frameduration))
		rtime = rightsamp*item.frameduration;
	else if ((righttime >= zoom[0].stime) && (righttime < zoom[0].etime))
		rtime = righttime;
	else
		rtime = zoom[0].etime;

	/* draw input waveform and timescale */
	drawwaveform(X0,X3,Y3,Y4,Y4,Y5,&item,sp,sfsfilename);

	/* if print-mode, do title and stop here */
	if (laser) {
		digscale(1.0,1.0,0);
	
		digitab.ixoff = AXCHAR * digdata.chwidth + 3;
		xl = X0 + digitab.ixoff/digdata.xscale;

		/* get scale factor */
		digitab.scalex = (X3-xl)/(zoom[0].etime-zoom[0].stime);

		/* draw cursors */
		if (ltime > zoom[0].stime) {
			x = xl + (ltime-zoom[0].stime)*digitab.scalex;
			digline(94,x,Y3,x,Y4);
		}
		if (rtime < zoom[0].etime) {
			x = xl + (rtime-zoom[0].stime)*digitab.scalex;
			digline(94,x,Y3,x,Y4);
		}

		/* display spectrum */
		if ((ltime > zoom[0].stime) && (rtime < zoom[0].etime))
			calcspectrum(sp,item.frameduration,ltime,rtime);
		else if (ltime > zoom[0].stime)
			calcspectrum(sp,item.frameduration,ltime-DEFAULT_SPECT_SIZE/2,ltime+DEFAULT_SPECT_SIZE/2);
		else if (rtime < zoom[0].etime)
			calcspectrum(sp,item.frameduration,rtime-DEFAULT_SPECT_SIZE/2,rtime+DEFAULT_SPECT_SIZE/2);
		else
			calcspectrum(sp,item.frameduration,zoom[0].stime,zoom[0].etime);
		
		/* display spectrum axes */
		drawspectrumaxes(1.0/item.frameduration,X1,Y1,X2,Y2);
		drawspectrum(1.0/item.frameduration,X1,Y1,X2,Y2);
		drawformant(X4,Y1,X3,Y2);

		tim = time((time_t *)0);
		strcpy(timbuf,ctime(&tim));
		timbuf[19]='\0';
		if (*laser_title)
			sprintf(messg,"LPC Spectrum.  User: %s  Terminal: %s  Title: %s",logname(),ttyname(1),laser_title);
		else
			sprintf(messg,"LPC Spectrum.  User: %s  Terminal: %s  Date: %s",logname(),ttyname(1),timbuf);

		digscale(1.0,1.0,0);
		digtext(24,0.0,1.0-charheight,messg);
		digbox(24,0.0,0.0,1.0,Y5);
		digquit(0);
		exit(0);
	}

	/* display spectrum axes */
	drawspectrumaxes(1.0/item.frameduration,X1,Y1,X2,Y2);
	
	/* initialise mouse */
	diginitmouse(0.5,(Y3+Y4)/2);

	/* initialise DAC */
	if (dac_open(NULL) >= 0)
		doplay=1;
	else
		doplay=0;

	/* display menu */
	esmenu_display(21,menu_labels,menu_keys,sizeof(menu_labels)/sizeof(char *));
	
	/* command loop */
	while ((com=getcommand()) != COM_QUIT) switch (com) {

	case COM_LEFT_CURS:
		break;
	case COM_RIGHT_CURS:
		break;
	case COM_REFRESH:
		digscale(1.0,1.0,0);
		digclearscreen();
		charwidth = digdata.chwidth/digdata.xscale;
		charheight = digdata.chheight/digdata.yscale;
		NSPECTPOINT = (2*digdata.nhoriz)/3;
	
		/* initialise Graphics positions */
		Y5 = 1.0 - 2*charheight;	/* top of timescale */
		Y4 = Y5 - 2*charheight;		/* top of waveform */

		/* draw input waveform and timescale */
		drawwaveform(X0,X3,Y3,Y4,Y4,Y5,&item,sp,sfsfilename);

		/* display spectrum axes */
		drawspectrumaxes(1.0/item.frameduration,X1,Y1,X2,Y2);
	
		/* display menu */
		esmenu_display(21,menu_labels,menu_keys,sizeof(menu_labels)/sizeof(char *));
 
		ltime = zoom[zsp].stime;
		rtime = zoom[zsp].etime;

		break;

	case COM_ZOOM_IN:
		if (zsp < ZOOM_DEPTH) zsp++;
		if (rtime < ltime) {
			ctim = rtime;
			rtime = ltime;
			ltime = ctim;
		}
		if (rtime < (ltime + 10*item.frameduration))
			rtime = ltime + 10*item.frameduration;
		zoom[zsp].stime = ltime;
		zoom[zsp].etime = rtime;
 		drawwaveform(X0,X3,Y3,Y4,Y4,Y5,&item,sp,sfsfilename);
		break;

	case COM_ZOOM_OUT:
		if (zsp <= 1) {
			zoom[1] = zoom[0];
			zsp = 1;
		}
		else
			zsp--;
		ltime = zoom[zsp].stime;
		rtime = zoom[zsp].etime;
 		drawwaveform(X0,X3,Y3,Y4,Y4,Y5,&item,sp,sfsfilename);
		break;

	case COM_SCROLL_LEFT:
		ctim = zoom[zsp].etime;
		if (zoom[zsp].etime != rtime)
			zoom[zsp].etime = rtime;
		else
			zoom[zsp].etime = zoom[zsp].stime;
		zoom[zsp].stime = zoom[zsp].etime - (ctim - zoom[zsp].stime);
		if (zoom[zsp].stime < 0) {
			zoom[zsp].etime += -zoom[zsp].stime;
			zoom[zsp].stime = 0.0;
		}
		ltime = zoom[zsp].stime;
		rtime = zoom[zsp].etime;
 		drawwaveform(X0,X3,Y3,Y4,Y4,Y5,&item,sp,sfsfilename);
		break;

	case COM_SCROLL_RIGHT:
		ctim = zoom[zsp].stime;
		if (zoom[zsp].stime != ltime)
			zoom[zsp].stime = ltime;
		else
			zoom[zsp].stime = zoom[zsp].etime;
		zoom[zsp].etime = zoom[zsp].stime + (zoom[zsp].etime - ctim);
		if (zoom[zsp].etime > item.numframes*item.frameduration) {
			zoom[zsp].stime -= zoom[zsp].etime - item.numframes*item.frameduration;
			zoom[zsp].etime = item.numframes*item.frameduration;
		}
		ltime = zoom[zsp].stime;
		rtime = zoom[zsp].etime;
 		drawwaveform(X0,X3,Y3,Y4,Y4,Y5,&item,sp,sfsfilename);
		break;

	case COM_RESTART:
		zoom[1] = zoom[0];
		zsp=1;
		ltime = zoom[zsp].stime;
		rtime = zoom[zsp].etime;
 		drawwaveform(X0,X3,Y3,Y4,Y4,Y5,&item,sp,sfsfilename);
		break;

	case COM_SPECTRUM:
		/* calculate spectrum */
		if ((ltime > zoom[zsp].stime) && (rtime < zoom[zsp].etime)) {
			/* calculate spectrum */
			calcspectrum(sp,item.frameduration,ltime,rtime);
			drawspectrum(1.0/item.frameduration,X1,Y1,X2,Y2);
			drawformant(X4,Y1,X3,Y2);
		}
		else if (ltime > zoom[zsp].stime) {
			/* calculate spectrum */
			calcspectrum(sp,item.frameduration,ltime-DEFAULT_SPECT_SIZE/2,ltime+DEFAULT_SPECT_SIZE/2);
			drawspectrum(1.0/item.frameduration,X1,Y1,X2,Y2);
			drawformant(X4,Y1,X3,Y2);
		}
		else if (rtime < zoom[zsp].etime) {
			/* calculate spectrum */
			calcspectrum(sp,item.frameduration,rtime-DEFAULT_SPECT_SIZE/2,rtime+DEFAULT_SPECT_SIZE/2);
			drawspectrum(1.0/item.frameduration,X1,Y1,X2,Y2);
			drawformant(X4,Y1,X3,Y2);
		}
		else {
/*			clear_top_line();
			digprompt("No cursors set");
			sleep(1);
			clear_top_line();
*/
			calcspectrum(sp,item.frameduration,zoom[zsp].stime,zoom[zsp].etime);
			drawspectrum(1.0/item.frameduration,X1,Y1,X2,Y2);
			drawformant(X4,Y1,X3,Y2);
		}
		break;

	case COM_REPLAY:	/* replay input waveform */
		startsamp = (int)(ltime/item.frameduration);
		endsamp = (int)(rtime/item.frameduration);
		if (doplay && (endsamp > startsamp))
			dac_playback(sp+startsamp,endsamp-startsamp,1.0/item.frameduration,
				(int)param(item.params,"bits",(bits12)?12.0:16.0),1,1);
		break;

	case COM_PRINT:		/* do hard copy of current setting */
		hardcopy();
		break;

	case COM_CLEAR:
		drawspectrumaxes(1.0/item.frameduration,X1,Y1,X2,Y2);
		break;

	case COM_FORMAT:
		if (dospect==0) {
			dospect=1;
			DEFAULT_SPECT_SIZE = WIDE_SPECT_SIZE;
		}
		else if (donarrow==0) {
			donarrow=1;
			DEFAULT_SPECT_SIZE = NARROW_SPECT_SIZE;
		}
		else {
			dospect=0;
			donarrow=0;
			DEFAULT_SPECT_SIZE = WIDE_SPECT_SIZE;
		}
		ltime = zoom[zsp].stime;
		rtime = zoom[zsp].etime;
 		drawwaveform(X0,X3,Y3,Y4,Y4,Y5,&item,sp,sfsfilename);
		break;

	}

	if (doplay) dac_close(0);
	digkillmouse();
	digclearscreen();
	digquit(0);
	exit(0);
}


