/* addnoise - adjust signal-to-noise ratio in file */

/* Mark Huckvale - Univserity College London */

/* version 1.0 - December 2002 */

#define PROGNAME "addnoise"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH ADDNOISE SFS1 UCL
.SH NAME
addnoise - add noise to speech signal
.SH SYNOPSIS
.B addnoise
(-w|-p) (-a dblevel|-r dblevel) (-n) (-t threshdb) file
.SH DESCRIPTION
.I addnoise
is a program to add noise to a signal and to control the absolute
signal level or signal-to-noise ratio.  Options control whether white
noise or pink noise is added and whether the final result is
normalised to a standard level.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i item
Select input item number.
.TP 11
.B -w
Select white noise (default).
.TP 11
.B -p
Select pink-noise.
.TP 11
.BI -a dblevel
Specify level of noise in absolute terms with respect to maximum
level signal.
.TP 11
.BI -r dblevel
Specify level of noise in relative terms with respect to the
level of the input signal.
.TP 11
.B -n
Specify that the resulting signal should be normalised to -18dB
absolute.
.TP 11
.BI -t threshold
Specifies the relative signal level to be considered silence and
ignored in SNR calculations.  Default is 50dB.
.SH INPUT ITEMS
.IP SP
Input speech signal.
.SH OUTPUT ITEMS
.IP SP
Signal with added noise
.SH HISTORY
.IP type
Noise type.
.IP abs
Absolute noise level.
.IP SNR
Signal to noise ratio.
.IP norm
Normalised.
.IP thresh
Silence threshold.
.SH VERSION/AUTHOR
.IP 1.0
Mark Huckvale
.SH SEE ALSO
testsig, level, combine
.SH BUGS
*/
/*--------------------------------------------------------------------------*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"

/* manifest constants */
#define WINTIME	0.05
#define SQRT2 1.414213562
#define MAXSAMPLE 32767.0
#define RMS0DB (MAXSAMPLE/SQRT2)
#define rmsLevel(a) ( ((a)<1) ? (-96.0) : (20.*log10((a)/RMS0DB)) )
#define NORMLEVEL -18.0

/* global variables */
struct item_header	spitem;
struct item_header	opitem;
short		*sp;			/* signal */
float		*fnp;			/* noise */
float		*fsp;			/* combination */
double		noiselevel;
double		signallevel;
int			winsize;		/* 50ms in samples */

/* operation */
int		noisetype=0;
int		doabs;
double	alevel=0;
int		dorel;
double	rlevel=0;
double	slevel=50;
int		donorm=0;

/* generate white noise */
void genwhite(int len)
{
	double x,y,r;
	int	i;

	fnp = (float *)calloc(len,sizeof(float));
	for (i=0;i<len;i++) {
		/* get a random co-ordinate inside the unit circle */
		do {
			x = (rand()%10000)/5000.0 - 1.0;
			y = (rand()%10000)/5000.0 - 1.0;
			r = (x*x)+(y*y);
		} while ((r == 0) || (r >= 1.0));
		/* transform into a normal distribution (Box-Muller transform) */
		fnp[i] = (float)(x * sqrt(-2.0*log(r)/r) * 10000);
	}
}

/* generate pink noise */
void genpink(int len)
{
	int	i;
	double b0=0,b1=0,b2=0,b3=0,b4=0,b5=0,b6=0;
	double	white,pink;

	fnp = (float *)calloc(len,sizeof(float));
	for (i=0;i<len;i++) {
		/* algorithm from
			http://shoko.calarts.edu/~glmrboy/musicdsp/sourcecode/pink.txt
		*/
		white = ((rand()&0x00007FFF)-0x4000)/10.0;
		b0 = 0.99886 * b0 + white * 0.0555179;
		b1 = 0.99332 * b1 + white * 0.0750759;
		b2 = 0.96900 * b2 + white * 0.1538520;
		b3 = 0.86650 * b3 + white * 0.3104856;
		b4 = 0.55000 * b4 + white * 0.5329522;
		b5 = -0.7616 * b5 - white * 0.0168980;
		pink = b0 + b1 + b2 + b3 + b4 + b5 + b6 + white * 0.5362;
		b6 = white * 0.115926;
		fnp[i] = (float)pink;
	}
}

/* get the level of a signal wrt RMS 0dB */
double getlevel(float *sig,int len)
{
	int	i,j;
	double	sum,size,max;
	double	min;
	int		cnt;
	double	total;

	/* get the maximum level */
	i=0;
	max=0;
	while (i < len) {
		sum=0;
		for (j=0;(j<winsize)&&(i<len);j++,i++)
			sum += sig[i] * sig[i];
		size = sqrt(sum/j)/RMS0DB;
		if (size > max) max=size;
	}

	/* calculate threshold level */
	min = max * (pow(10.0,-slevel/20));

	/* get the average above threshold */
	i=0;
	total=0;
	cnt=0;
	while (i < len) {
		sum=0;
		for (j=0;(j<winsize)&&(i<len);j++,i++)
			sum += sig[i] * sig[i];
		size = sqrt(sum/j)/RMS0DB;
		if (size > min) {
			total += sum;
			cnt += j;
		}
	}

	return(20.0*log10(sqrt(total/cnt)/RMS0DB));
}

/* combine two signals, adjusting level */
void combine(float *s1,float *s2,double dbadjust,int len)
{
	int	i;
	double	scale;

	scale = pow(10.0,dbadjust/20);

	for (i=0;i<len;i++)
		s1[i] = (float)(s1[i] + s2[i] * scale);
}

/* adjust signal level */
void adjust(float *sig,double dbadjust,int len)
{
	int	i;
	double	scale;

	scale = pow(10.0,dbadjust/20);

	for (i=0;i<len;i++)
		sig[i] = (float)(sig[i] * scale);
}

/* main program */
int main(int argc,char *argv[])
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item selection */
	char	*ty;		/* item sub type */
	char	*sptype="0";
	/* file variables */
	char	filename[SFSMAXFILENAME]; /* SFS data file name */

	int		i;
	int		overload;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:wpa:r:nt:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: add noise to signal V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					sptype = ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'w':
			noisetype=0;
			break;
		case 'p':
			noisetype=1;
			break;
		case 'a':
			alevel = atof(optarg);
			doabs=1;
			dorel=0;
			break;
		case 'r':
			rlevel = atof(optarg);
			dorel=1;
			doabs=0;
			break;
		case 'n':
			donorm=1;
			break;
		case 't':
			slevel = atof(optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-w|-p) (-a level|-r level) (-n) (-t level) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* load speech signal */
	getitem(filename,SP_TYPE,sptype,&spitem,&sp);
	fsp = (float *)calloc(spitem.numframes,sizeof(float));
	for (i=0;i<spitem.numframes;i++) fsp[i] = sp[i];
	winsize = (int)(WINTIME / spitem.frameduration);

	/* generate noise signal */
	if (noisetype==0)
		genwhite(spitem.numframes);
	else
		genpink(spitem.numframes);

	/* get the current levels */
	signallevel = getlevel(fsp,spitem.numframes);
	printf("input level  = %5.1fdB\n",signallevel);
	noiselevel  = getlevel(fnp,spitem.numframes);

	/* combine speech and noise */
	if (doabs) {
		printf("noise level  = %5.1fdB\n",noiselevel+alevel-noiselevel);
		combine(fsp,fnp,alevel - noiselevel,spitem.numframes);
	}
	else {
		printf("noise level  = %5.1fdB\n",noiselevel+signallevel - noiselevel + rlevel);
		combine(fsp,fnp,signallevel - noiselevel + rlevel,spitem.numframes);
	}

	/* normalise */
	if (donorm) {
		signallevel = getlevel(fsp,spitem.numframes);
		adjust(fsp,NORMLEVEL - signallevel,spitem.numframes);
	}

	/* report output level */
	signallevel = getlevel(fsp,spitem.numframes);
	printf("output level = %5.1fdB\n",signallevel);

	/* create output header */
	sfsheader(&opitem,SP_TYPE,0,2,1,spitem.frameduration,spitem.offset,1,0,0);
	if (doabs)
		sprintf(opitem.history,"%s(%d.%02d;type=%s,abs=%g,thresh=%g%s)",
			PROGNAME,spitem.datatype,spitem.subtype,
			(noisetype==0)?"white":"pink",
			alevel,slevel,
			(donorm)?",norm":"");
	else
		sprintf(opitem.history,"%s(%d.%02d;type=%s,rel=%g,thresh=%g%s)",
			PROGNAME,spitem.datatype,spitem.subtype,
			(noisetype==0)?"white":"pink",
			rlevel,slevel,
			(donorm)?",norm":"");

	/* convert */
	overload=0;
	for (i=0;i<spitem.numframes;i++) {
		if (fsp[i] < -32767) {
			sp[i] = -32767;
			overload=1;
		}
		else if (fsp[i] > 32767) {
			sp[i] = 32767;
			overload=1;
		}
		else
			sp[i] = (short)fsp[i];
	}
	if (overload) {
		fprintf(stderr,"output signal is overloaded, level=%.1fdB",signallevel);
		error("output signal is overloaded");
	}

	/* save */
	putitem(filename,&opitem,spitem.numframes,sp);

	/* that's all folks */
	exit(0);

}
