/* HQtx -- high performance tx from lx */

/* m.a. huckvale - march 1987 */

/* version 1.0s
	- SFS version
*/
/* version 1.1 - January 1988
	- filter impossibly high freqs
	- add 16kHz filter
*/
/* version 2.0 - July 2000
	- resurrect old code
	- add DSP filter design
*/
/* version 2.1a - February 2001
	- add a threshold on gradient amplitude
*/

/* version 2.2 - November 2001
	- increase max Fx to 1000Hz
	- modify prototype generation thresholding
	- modify filtering to eliminate VLF
*/

#undef IAG
#undef IAG2

#define PROGNAME "HQtx"
#define PROGVERS "2.5"
char *progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH HQTX SFS1 UCL
.SH NAME
HQtx - high performance conversion of Lx to Tx
.SH SYNOPSIS
.B HQtx
(-i item) (-n) file
.SH DESCRIPTION
.I HQtx
is a program to obtain excitation periods from a laryngograph signal.  It
exploits the characteristic shape of the lx waveform during voiced
excitation.  It utilises two derived functions: the sample-by-sample
differential, and the instantaneous gradient.  Tx points are extracted
from these functions using an automatic threshold determination procedure.
.I HQtx
automatically inverts the lx waveform if it is determined to be upside-down.
This can be cancelled by the "-n" flag.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.B -n
Do not automatically correct for waveform inversion.
.TP 11
.BI -i item
Select input item number.
.SH INPUT ITEMS
.IP 2.xx 11
Any excitation item.
.SH OUTPUT ITEMS
.IP 3 11
High performance tx.
.SH VERSION/AUTHOR
2.2 - Mark Huckvale.
.SH SEE ALSO
tx, txgen
.SH BUGS
Loads whole waveform into memory.
*/
/*--------------------------------------------------------------------------*/

/* global declarations */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "sfs.h"		/* header structures */
#include "filter.h"

#define DETECT	7.0		/* energy detection threshold multiplier */
#define MINNEGGRADIENT -50		/* smallest acceptable gradient of pulse */
#define MINPOSGRADIENT 100		/* smallest acceptable gradient of pulse */
#define MIN(x,y)	(( (x) < (y) ) ? (x) : (y))
#define MAX(x,y)	(( (x) > (y) ) ? (x) : (y))

/* global data */
struct item_header 	lxitem;		/* item header data */
short			*lx;		/* lx data buffer */
struct item_header 	txitem;		/* item header data */
int			lxinv=1;	/* auto lx inversion ON */
int			lxwasinv=0;	/* lx was inverted flag */

/* prototype tx structure table */
struct proto_rec {
	int	valley;			/* lx valley position */
	int	peak;			/* lx peak position */
	int	edge;			/* lx edge position */
	int	amplitude;		/* amp at peak */
	int	ampvalley;		/* amp at valley */
	int	height;			/* differentiated height */
	double	energy;			/* lx energy over window */
} *proto;
int prsize;
int	prcnt=0;			/* index into proto */

/* check LX right-way up - inverting if necessary */
void checklxpolarity(short *lx,int numf)
{
	int		i;
	int		d1=0,d2=0,d3=0;
	double	nsum=0,psum=0;
	int		ncnt=1,pcnt=1;

	/* scan for peaks and valleys in differentiated lx */
	d1 = lx[1]-lx[0];
	d2 = lx[2]-lx[1];
	for (i=2;i<numf-1;i++) {
		d3 = lx[i+1] - lx[i];
		if ((d2>5)&&(d1<d2-5)&&(d3<d2-5)) {
			/* got a peak */
			psum += d2*d2;
			pcnt++;
		}
		else if ((d2<-5)&&(d1>d2+5)&&(d3>d2+5)) {
			/* got a valley */
			nsum += d2*d2;
			ncnt++;
		}

		d1 = d2;
		d2 = d3;
	}

	nsum /= ncnt;		/* mean -ve value */
	psum /= pcnt;		/* mean +ve value */

/* fprintf(stderr,"nsum=%g,ncnt=%d,psum=%g,pcnt=%d\n",nsum,ncnt,psum,pcnt); */

	/* invert if required */
	if (lxinv && (nsum > psum)) {
		if (ttytest()) printf("%s: Lx waveform inverted       \n",PROGNAME);
		for (i=0;i<numf;i++) lx[i] = -lx[i];
		lxwasinv++;
	}
}

/* calculate instantaneous gradient */
void gradient(short *lx,int numf,double srate)
{
	int	i,j,width;
	float	*x,xsum,ysum,xysum,xsumsq;
	float	grad;
	float	hold[500];

	width = (int)(srate * 0.0002);
	if (width < 8) width=8;
	if (width > 500) width=500;

	/* init first samples */
	x = hold;
	for (i=0;i<width-1;i++) *x++ = *lx++;

	/* init constants */
	xsum = 0;
	xsumsq = 0;
	for (j=0;j<width;j++) {
		xsum += j;
		xsumsq += (float)j*j;
	}

	/* calculate gradient */
	for (i=width-1;i<numf;i++) {
		hold[width-1] = *lx;
		ysum = 0;
		xysum = 0;
		x = hold;
		for (j=0;j<width;j++,x++) {
			ysum += *x;
			xysum += j * *x;
		}
		grad = (width*xsumsq-xsum*xsum);
		if (grad != 0)
			grad = (width*xysum - ysum*xsum)/grad;

		/* scaled gradient per ms */
		j = (int)(grad * srate * 0.001)/4;
		if (j > 32767)
			*lx++ = 32767;
		else if (j < -32768)
			*lx++ = -32768;
		else
			*lx++ = j;

		/* shift along to next sample */
		x=hold+1;
		for (j=1;j<width;j++,x++) *(x-1) = *x;
	}
}

/* estimate number of prototypes */
int estnumproto(short *lx,int numf)
{
	int	i;
	int	count=0;

	/* pass over gradient function looking for zero crossings */
	numf--;
	for (i=0;i<numf;i++) {
		if ((*lx >= 0) && (*(lx+1) < 0)) count++;
		lx++;
	}
	return(count);
}

/* process gradient function to generate tx prototypes */
void progen(short *lx,int numf,double srate)
{
	int	i;
	int	valley=0;
	int	max=0;
	int min=0;
	int ms20 = (int)(0.5 + srate * 0.02);

	/* pass over gradient function looking for zero crossings */
	prcnt=0;
	numf--;
	for (i=0;i<numf;i++) {
#ifdef IAG2
printf("%.4f lx[%d]=%d valley=%d min=%d max=%d\n",i*lxitem.frameduration,i,lx[i],valley,min,max);
#endif
		if ((valley==0)&&(lx[i] <= MINNEGGRADIENT) && (lx[i+1] > MINNEGGRADIENT)) {
			/* positive going zero-crossing */
			proto[prcnt].valley = i;
			valley=1;
			max=0;
			min=lx[i];
		}
		else if ((valley>0)&&(lx[i] >= MINNEGGRADIENT) && (lx[i+1] < MINNEGGRADIENT)) {
			/* negative going zero-crossing */
			if (min < -500) min=-500;
			if ((valley>ms20)||(max > -3*min/2)) {
				proto[prcnt].peak = i;
				proto[prcnt].edge = i;
				proto[prcnt].amplitude = max;
				proto[prcnt].ampvalley = min;
				prcnt++;
				if (prcnt==prsize)
					error("Internal table full");
				valley=0;
			}
		}
		else if ((valley>0)&&(lx[i] < min)) {
			/* locate start at minimum gradient */
			proto[prcnt].valley = i;
			min = lx[i];
		}
		if (lx[i] > max) max = lx[i];
		if (valley > 0) valley++;
	}

}

/* calculate differential */
void differ(short *lx,int numf)
{
	int	i;
	short	*s,*d;

	/* generate differential */
	s=lx+1;
	d=lx;
	for (i=1;i<numf;i++,s++,d++) {
		if (lxwasinv)
			*d = *d - *s;
		else
			*d = *s - *d;
	}
	lx[numf-1]=0;

}

/* generate prototype tx edges */
void edgegen(short *lx)
{
	int	i,j,maxpos;
	short	*s,maxval;

	for (i=0;i<prcnt-1;i++) {
		maxpos=proto[i].valley;
		s = &lx[proto[i].valley];
		maxval = *s;
		for (j=proto[i].valley;j<=proto[i+1].valley;j++,s++) {
			if (*s > maxval) {
				maxval = *s;
				maxpos = j;
			}
		}
		proto[i].edge=maxpos;
		proto[i].height=maxval;
	}
}

/* integrate lx */
void integrate(short *lx,int numf,double srate)
{
	register int	i;
	register short	*s,*d;

	s=lx;
	d=lx+1;
	if (srate > 20000)
		for (i=1;i<numf;i++,s++,d++) *d = (short)(*s * 0.98 + *d);
	else
		for (i=1;i<numf;i++,s++,d++) *d = (short)(*s * 0.95 + *d);
}

/* filter lx waveform */
void filter(short *lx,int numf,double srate)
{
	FILTER	*flt;

/*	flt = filter_design(FILTER_BAND_PASS,4,20,2500.0,srate); */
	flt = filter_design(FILTER_LOW_PASS,8,2500.0,srate/2,srate);
	filter_signal(flt,lx,lx,numf);
	filter_free(flt);
}

/* calculate energy in lx */
double calcenergy(short *lx,int start,int finish)
{
	int	i;
	short	*s,*d;
	double	sumsq=0,val;
	double	dc=0;
	int		ms1;

	ms1 = (int)(0.5 + 0.001/lxitem.frameduration);

	/* check range */
	if (start < ms1) start = ms1;
	if (finish < 2*ms1) finish = 2*ms1;
	if (finish > lxitem.numframes-ms1-1) finish = lxitem.numframes-ms1-1;
	if (start >= finish) start=finish-1;

#if 0
/* get DC value around region */
	s = &lx[start-ms1];
	d = s+1;
	for (i=start-ms1;i<finish+ms1-1;i++,s++,d++) dc += (0.9 * *s) - *d ;
	dc /= (2*ms1+finish-start-1);
#endif

	/* sumsquare samples */
	s = &lx[start-ms1];
	d = s+1;
	for (i=start-ms1;i<finish+ms1-1;i++,s++,d++) {
		val = (0.9 * *s) - *d /* - dc */;
		sumsq += val * val;
	}
	return(sumsq/(2*ms1+finish-start-1));
}

/* delete a prototype by merging with neighbour */
void delproto(int i)
{
	int	l,n;
#if 0
	l = MAX(i-1,0);
	n = MIN(i+1,prcnt-1);
	/* choose shorter neighbour */
	if ((proto[i].edge-proto[l].edge) < (proto[n].edge-proto[i].edge)) {
		/* merge with last */
		if (proto[i].height > proto[l].height) {
			proto[l].height = proto[i].height;
			proto[l].edge = proto[i].edge;
		}
		proto[l].peak = proto[i].peak;
		proto[l].energy = MAX(proto[l].energy,proto[i].energy);
		proto[l].amplitude = MAX(proto[l].amplitude,proto[i].amplitude);
		proto[l].ampvalley = MIN(proto[l].ampvalley,proto[i].ampvalley);
	}
	else {
		/* merge with next */
		if (proto[i].height > proto[n].height) {
			proto[n].height = proto[i].height;
			proto[n].edge = proto[i].edge;
		}
		proto[n].valley = proto[i].valley;
		proto[n].energy = MAX(proto[n].energy,proto[i].energy);
		proto[n].amplitude = MAX(proto[n].amplitude,proto[i].amplitude);
		proto[n].ampvalley = MIN(proto[n].ampvalley,proto[i].ampvalley);
	}
#endif
	proto[i].energy=0;
}

/* get energies at excitation points */
void threshold(short *lx,double srate,double thresh,int fid)
{
	int		i,j;
	int		lasttx,ms03,ms1,ms10,val,p1,p2,p3;
	int		change,first;

	/* get 1 ms worth of samples */
	ms03 = (int)(0.5 + srate * 0.0003);
	ms1 = (int)(0.5 + srate * 0.001);
	ms10 = (int)(0.5 + srate * 0.01);

	/* reset tx count */
	lasttx=0;

	/* get energies at cross-overs */
	for (i=0;i<prcnt;i++) {
		proto[i].energy=calcenergy(lx,proto[i].valley,proto[i].peak);
	}

	/* loop around, cleaning up list of pulses */
	first=1;
	do {
		change=0;


		/* filter out spurious excitations */
		for (i=0;i<prcnt-1;i++) {
			if (i==0)
				p1 = proto[i].edge;
			else
				p1 = proto[i].edge - proto[i-1].edge;
			p2 = proto[i+1].edge - proto[i].edge;
			if (i==prcnt-2)
				p3 = 0;
			else
				p3 = proto[i+2].edge - proto[i+1].edge;
/* printf("i=%d p1=%d p2=%d p3=%d\n",i,p1,p2,p3); */

			/* remove lonesome pulses */
			if ((p1 > 2*ms10) &&
			    (p2 > 2*ms10)) {
				/* zap this period */
				delproto(i);
				change++;
#ifdef IAG2
				printf("%.4f is lonesome pulse\n",proto[i].edge*lxitem.frameduration);
#endif
			}
			/* remove lonesome periods */
			else if ((p1 > 2*ms10) &&
			    (p3 > 2*ms10)) {
				/* zap this period */
				delproto(i);
				change++;
#ifdef IAG2
				printf("%.4f is lonesome period\n",proto[i].edge*lxitem.frameduration);
#endif
			}
			/* remove secondary excitations */
			else if ((i>0) && (p1 < ms10) && (p2 < ms10) &&
			    (3*proto[i].height < proto[i-1].height) &&
			    (3*proto[i].height < proto[i+1].height)) {
				/* zap this period for real */
				delproto(i);
				change++;
#ifdef IAG2
				printf("%.4f is secondary (heights=%d,%d,%d)\n",
					proto[i].edge*lxitem.frameduration,
					proto[i-1].height,proto[i].height,proto[i+1].height);
#endif
			}

			/* remove secondary excitations part 2 */
			else if ((i>0) && (p1 < ms10) && (p2 < ms10) &&
			    (proto[i].amplitude < 1000) &&
			    (2*proto[i].amplitude < proto[i-1].amplitude) &&
			    (2*proto[i].amplitude < proto[i+1].amplitude)) {
				/* zap this period for real */
				delproto(i);
				change++;
#ifdef IAG2
				printf("%.4f is secondary (amplitudes=%d,%d,%d)\n",
					proto[i].edge*lxitem.frameduration,
					proto[i-1].amplitude,proto[i].amplitude,proto[i+1].amplitude);
#endif
			}
			/* merge short ones with neighbours */
			else if ((i>0) && !first && (fabs(proto[i].edge-proto[i-1].edge) < ms03)) {
				/* merge with neighbour */
				delproto(i);
				change++;
#ifdef IAG2
				printf("%.4f is merged (prev at %.4f)\n",
					proto[i].edge*lxitem.frameduration,
					proto[i-1].edge*lxitem.frameduration);
#endif
			}
		}

		/* filter out low energy peaks */
		j=0;
		for (i=0;i<prcnt;i++) {
			if ((proto[i].amplitude > -3*proto[i].ampvalley/2) && (proto[i].energy > thresh)) {
				/* looks OK */
				proto[j]=proto[i];
				j++;
			}
			else {
				/* remove it completely */
				change++;
#ifdef IAG2
				printf("%.4f is zapped (amp=%d val=%d en=%g thr=%g)\n",
					proto[i].edge*lxitem.frameduration,
					proto[i].amplitude,proto[i].ampvalley,proto[i].energy,thresh);
#endif
			}
		}
#ifdef IAG2
printf("%d from %d remain after thresholding\n",j,prcnt);
#endif
		prcnt=j;
		first=0;

	} while (change>0);

	/* copy remaining periods out to tx */
	for (i=0;i<prcnt;i++) {
		if ((proto[i].energy > 0) &&
		    ((val=proto[i].edge-lasttx) >= ms1)) {
			sfswrite(fid,1,&val);
			lasttx=proto[i].edge;
		}
		else
			proto[i].valley=0;
	}
	lasttx = lxitem.numframes - lasttx;
	if (lasttx > 0) {
		sfswrite(fid,1,&lasttx);
	}
}

/* list proto tx */
void prolst(char *filename)
{
	FILE	*op;
	int	i;

	op=fopen(filename,"w");
	for (i=0;i<prcnt;i++) {
		if (proto[i].valley)
			fprintf(op,"edge=%.4f. valley=%.4f, peak=%.4f, height=%d, energy=%g, amplitude=%d/%d\n",
					proto[i].edge*lxitem.frameduration,
					proto[i].valley*lxitem.frameduration,
					proto[i].peak*lxitem.frameduration,
					proto[i].height,proto[i].energy,proto[i].amplitude,proto[i].ampvalley);
	}
	fclose(op);
}

/* save part-processed lx to file */
void savelx(short *lx,struct item_header *lxitem,char *title,char *filename)
{
	struct item_header item;

	/* place processing details in item header */
	sfsheader(&item,lxitem->datatype,lxitem->floating,
			lxitem->datasize,lxitem->framesize,
			lxitem->frameduration,lxitem->offset,
			lxitem->windowsize,lxitem->overlap,
			lxitem->lxsync);
	sprintf(item.history,"%s(%d.%02d;component=%s)",PROGNAME,lxitem->datatype,lxitem->subtype,title);

	/* store it */
	putitem(filename,&item,lxitem->numframes,lx);

}

/* main program */
void main(int argc,char *argv[])
{
	/* local variables */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument */
	int		errflg=0;	/* option error flag */
	int		c;		/* option char */
	int		it;
	char		*ty;

	char		*lxtype = "*";	/* input lx sub-type */
	char		lxnum[5];	/* discovered lx sub-type */
	char		filename[SFSMAXFILENAME];	/* database file name */
	int		fid;
	double		srate;
	int		i,ms20;
	double		en,enmin,thresh;	/* lx background energy */

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:n")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: High-performance tx from lx V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* item spec */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == LX_TYPE)
					lxtype=ty;
				else
					error("unsuitable item specification %s",optarg);
			}
			else
				error("illegal item specification %s",optarg);
			break;
		case 'n' :	/* auto lx inversion */
			lxinv=0;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-n) dbase_file\n",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* check file ok to write */
	if ((fid=sfsopen(filename,"w",NULL)) < 0)
		error("access error on %s",filename);
	sfsclose(fid);

	/* load lx item */
	getitem(filename,LX_TYPE,lxtype,&lxitem,&lx);
	srate = 1.0/lxitem.frameduration;

	/* check polarity */
	checklxpolarity(lx,lxitem.numframes);

	/* calculate gradient function */
	if (ttytest()) { printf("Calculating gradient ...\r"); fflush(stdout); }
	gradient(lx,lxitem.numframes,srate);
#ifdef IAG2
	savelx(lx,&lxitem,"gradient",filename);
#endif

	/* generate prototype tx */
	if (ttytest()) { printf("Generating prototypes ...\r"); fflush(stdout); }
	prsize = estnumproto(lx,lxitem.numframes);
	proto = (struct proto_rec *)calloc(prsize,sizeof(struct proto_rec));
	if (proto==NULL)
		error("could not get memory buffer");
	progen(lx,lxitem.numframes,srate);
#ifdef IAG2
	prolst("HQtx.pr");
#endif

	/* calculate differential */
	free(lx);
	sprintf(lxnum,"%d",lxitem.subtype);
	getitem(filename,LX_TYPE,lxnum,&lxitem,&lx);
	differ(lx,lxitem.numframes);
#ifdef IAG2
	savelx(lx,&lxitem,"differential",filename);
#endif

	/* fill in edge positions in prototypes */
	edgegen(lx);
#ifdef IAG2
	prolst("HQtx.pr2");
#endif

	/* get lx back by weak integration */
	integrate(lx,lxitem.numframes,srate);

	/* filter lx */
	if (ttytest()) { printf("Filtering ...             \r"); fflush(stdout); }
	filter(lx,lxitem.numframes,srate);
#ifdef IAG2
	savelx(lx,&lxitem,"filtered",filename);
#endif

	/* get detection threshold */
	ms20 = (int)(0.5+srate*0.020);
	enmin = 1E10;
	for (i=0;i<lxitem.numframes-ms20;i+=ms20) {
		en = calcenergy(lx,i,i+ms20);
		if (en < enmin) enmin = en;
	}
	thresh = DETECT * enmin;
#ifdef IAG2
printf("enmin=%g thresh=%g\n",enmin,thresh);
#endif

#ifdef IAG2
	/* get energies at cross-overs */
	for (i=0;i<prcnt;i++) {
		proto[i].energy=calcenergy(lx,proto[i].valley,proto[i].peak);
	}
	prolst("HQtx.pr3");
#endif

	/* create tx header */
	sfsheader(&txitem,TX_TYPE,0,4,1,lxitem.frameduration,lxitem.offset,1,0,0);
	if (lxwasinv)
		sprintf(txitem.history,"%s(%d.%02d;lx=inverted)",
			PROGNAME,lxitem.datatype,lxitem.subtype);
	else
		sprintf(txitem.history,"%s(%d.%02d)",
			PROGNAME,lxitem.datatype,lxitem.subtype);

	/* open output channel */
	if ((fid = sfschannel(filename,&txitem)) < 0)
		error("cannot open temporary file",NULL);

	/* threshold tx prototypes using energies at excitation points */
	if (ttytest()) { printf("Locating pitch periods ...\r"); fflush(stdout); }
	threshold(lx,srate,thresh/2,fid);

#ifdef IAG2
	prolst("HQtx.pr4");
#endif
	/* and update */
	if (!sfsupdate(filename))
		error("update error on %s",filename);

	if (ttytest()) { printf("                           \r"); fflush(stdout); }

	/* ... that's all folks */
	exit(0);
}


