/*--------------------------------------------------------------------------

LEVEL.C  - plundered from prep.c   

Andrew Simpson - University College London

--------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
/**MAN
.TH LEVEL SFS1 UCL
.SH NAME
level - adjust signal level to fixed mean energy
.SH SYNOPSIS
.B level
(-I) (-i item) (-a absolute level|-r relative level) (-t silence_threshold) file
.SH DESCRIPTION
.I level
is a program to adjust the mean energy of a signal by a relative amount
or to a fixed level.  Silent regions in the signal are ignored in the
energy calculation.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i item
Select input item number.
.TP 11
.BI -a absolute_level
Set level required in absolute dB relative to the maximum level possible
for 16-bit data (i.e. 0dB = sinusoid at -32767..+32767).  Values are
always negative.
.TP 11
.BI -r relative level
Change level by a relative number of dB.
.TP 11
.BI -t threshold
Set threshold for identifying silence.  Default 70dB.
.SH INPUT ITEMS
.IP SP
Speech signal
.SH OUTPUT ITEMS
.IP SP
Adjusted level.
.SH VERSION/AUTHOR
.IP 1.0
Andrew Simpson
*/
/*--------------------------------------------------------------------------*/

#define PROGNAME "level"
#define PROGVERS "1.0"
char	*progname=PROGNAME;

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"   

 /* level constants */
#define SQRT2 1.414213562
#define MAXSAMPLE 32767.0
#define RMS0DB (MAXSAMPLE/SQRT2)
#define rmsLevel(a) ( ((a)<1) ? (-96.0) : (20.*log10((a)/RMS0DB)) )
#define FRAMEDURATION 0.01
#define LEVELTHRESHOLDDB -70.0
#define RELATIVE 1
#define ABSOLUTE 0
#define TRUE -1
#define FALSE 0


 /* manifest constants */
#define BUFSIZE	4096

 /* global data */
struct item_header item;	/* input item header data */
struct item_header newitem;	/* output item header */
short	buff[BUFSIZE];

/* my own version of exp10 */
#define exp10(x) pow(10.0,x)

/* scale samples within a buffer */
void scale (buff,len,scalingFactor)
short *buff;
int len;
double scalingFactor;

{
  register int 	i;
  double scaledSample;
  
  for (i=0;i<len;i++,buff++)
    {
      if (fabs(scaledSample=(scalingFactor*(float)*buff)) > MAXSAMPLE)
	{
	  error("signal clipped when scaling (%.1f)",scaledSample);
	  *buff=(short)MAXSAMPLE;
	}
      else
	*buff=(short)scaledSample;
    }
}

   char	filename[SFSMAXFILENAME];
   char adjustment[128];

 /* main program */
void main (argc,argv)
int argc;
char *argv[];

{
   /* local variables */
   extern int	optind;		/* option index */
   extern char	*optarg;	/* option argument */
   int c,errflg=0;		/* option char */
   int it,initem=SP_TYPE;
   char	*ty,*intype="0";
   int	fid,ofid=-1;
   int	i,len,frameSamples;
   int level=TRUE,setlevel=FALSE,sample;
   int totalFrames,framesAboveThreshold,newFile=FALSE,count;
   short *ptr;
   double levelThresholdDB=LEVELTHRESHOLDDB,rmsSample;
   double sumOfSquares,totalSumOfSquares,adjustmentDB;
   int adjustmentType=ABSOLUTE;
   double rmsLevelDB,desiredLevelDB,sampleScalingFactor=1.0;

   /* decode switches */
   while ( (c = getopt(argc,argv,"Ii:t:a:r:")) != EOF )
     switch (c)
       {
       case 'I' :	/* Identify */
	 fprintf(stderr,"%s: Speech level calculation/setting V%s\n",PROGNAME,PROGVERS);
	 exit(0);
	 break;

       case 'i' :	/* item spec */
	 if (itspec(optarg,&it,&ty) == 0)
	   {
	     if ((it == SP_TYPE) || (it == LX_TYPE))
	       {
		 initem = it;
		 intype = ty;
	       }
	   else
	     error("unsuitable item specification %s",optarg);
	   }
	 else
	   error("illegal item specification %s",optarg);
	 break;

       case 'a' : /* set level */
	 setlevel=TRUE;
	 adjustmentType=ABSOLUTE;	
	 strcpy(adjustment,optarg);
	 sscanf(optarg,"%lf",&desiredLevelDB);
	 newFile=TRUE;
	 break;

       case 'r' : /* set level relative */
	 setlevel=TRUE;
	 adjustmentType=RELATIVE;
	 sscanf(optarg,"%lf",&adjustmentDB);
	 strcpy(adjustment,optarg);
	 newFile=TRUE;
	 break;

       case 't' : /* set level calculation threshold */
	 sscanf(optarg,"%lf",&levelThresholdDB);
	 if (levelThresholdDB > 0)
	 	levelThresholdDB = -levelThresholdDB;
	 break;

       case '?' :	/* unknown */
	 errflg++;
       }
   if (errflg || (argc<2))
     error("usage: %s (-I) (-a absolute-level|-r relative-level)  (-t silence-threshold) file",PROGNAME);

   /* get filename */
   if (optind < argc)
     strcpy(filename,sfsfile(argv[optind]));
   else
     error("no data file specified",NULL);
   
   /* open file */
   if ((fid=sfsopen(filename,"w",NULL)) < 0) {
     if (fid==-1)
       error("unable to find file '%s'",filename);
     else
       error("access error on '%s'",filename);
   }
   
   /* locate input item */
   if (!sfsitem(fid,initem,intype,&item))
     error("unable to find input item in '%s'",filename);
   
   frameSamples=(int)((1.0/item.frameduration)*FRAMEDURATION);

   if (newFile)
     {
       /* create output item header */
       sfsheader(&newitem,item.datatype,item.floating,
		 item.datasize,item.framesize,
		 item.frameduration,item.offset,
		 item.windowsize,item.overlap,item.lxsync);
	if (adjustmentType==ABSOLUTE) {
	       sprintf(newitem.history,"%s(%d.%02d;abs=%gdB,silence=%gdB)",
			PROGNAME,
		       item.datatype,item.subtype,
			desiredLevelDB,levelThresholdDB);
	}
	else {
	       sprintf(newitem.history,"%s(%d.%02d;rel=%gdB)",
			PROGNAME,
		       item.datatype,item.subtype,
			adjustmentDB);
	}
       
       /* open output channel */
	    if ((ofid=sfschannel(filename,&newitem)) < 0)
	      error("unable to open output file",NULL);
     }
   
   if (level || setlevel)
     {
       /* if needed, calculate the rms level of speech above threshold */
       
       totalFrames=0;
       framesAboveThreshold=0;
       totalSumOfSquares=0.;
       
       for (i=0;(len=sfsread(fid,i,BUFSIZE,buff))>0;i+=len)
	 {
	   for (sample=0;sample<len;sample+=frameSamples,totalFrames++)
	     {
	       /* calculate the RMS value of this frame */
	       sumOfSquares=0.;

	       for (count=0,ptr=&buff[sample]; 
		    count<frameSamples; 
		    count++,ptr++)
		 sumOfSquares+=((float)*ptr * (float)*ptr);

	       rmsSample=sqrt(sumOfSquares/(float)frameSamples);
	       
	       /* include this frame if it exceeds the threshold */
	       if (rmsLevel(rmsSample) >= levelThresholdDB)
		 {
		   totalSumOfSquares+=sumOfSquares;
		   framesAboveThreshold++;
		 }
	     }
	 }
       
       /* calculate overall RMS level */
       rmsSample=sqrt(totalSumOfSquares/(framesAboveThreshold*frameSamples));
       rmsLevelDB=rmsLevel(rmsSample);
       
       if (!setlevel)
	 {
	   /* print mean rms level of those frames above threshold */
	   if (framesAboveThreshold>0)
	     printf("%s : %d.%.2d : RMS level %.2fdB (%.0f%% > %.2fdB silence threshold)\n",
		    filename,
		    item.datatype,item.subtype,
		    rmsLevelDB,(framesAboveThreshold/(float)totalFrames)*100.,
		    levelThresholdDB);
	   else
	     printf("No frames above the %.2fdB threshold\n",levelThresholdDB);
	 }
       
       if (setlevel)
	 {
	   if (adjustmentType==ABSOLUTE)
	     adjustmentDB=(desiredLevelDB-rmsLevelDB);
	   sampleScalingFactor=exp10(adjustmentDB/(float)20.);
	 }
     }
   
   /* process file */
   for (i=0;(len=sfsread(fid,i,BUFSIZE,buff)) > 0;i+=len)
     {
       if (setlevel) scale (buff,len, sampleScalingFactor);
       
       if (newFile)
	 {
	   /* write out result of processing */
	   if (sfswrite(ofid,len,buff) != len)
	     error("write error on output file",NULL);
	 }
     }
   
   if (newFile)
     {
       /* update file */
       if (!sfsupdate(filename))
	 error("update error on %s",filename);
     }
	
   /* ... that's all folks */
   exit(0);

}

