/* txstat - calculate Tx statistics tracks */

/* M.A.Huckvale - University College London */

/* version 1.0 - August 2001 */

#define PROGNAME "txstat"
#define PROGVERS "1.0"
char	*progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH TXSTAT SFS1 UCL SFS
.SH NAME
txstat -- calculate Tx statistics of jitter and shimmer
.SH SYNOPSIS
.B txstat
(-i item) (-j|-J jittersize) (-s|-S shimmersize) (-f framerate) file
.SH DESCRIPTION
.I txstat
is a program to calculate some statistics of a tx item and to
report the results as track items. Output is
on 5ms frames (200 frames/sec) but may be changed by a command line
option. Jitter is a measure of pitch pertubation on a cycle to cycle
basis.  The duration of each pitch period T is compared to the average
of the N surrounding pitch periods and scaled according to
100.0*abs(1-T/averageT).  By default N=5.
Shimmer is a measure of amplitude perturbation on a cycle
to cycle basis using an equivalent formula based on the
maximum signal amplitude over the pitch period A compared
to the average of the N surrounding pitch periods:
100.0*abs(1-A/averageA).
.PP
Default operation is to calculate both jitter and shimmer.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version number.
.TP 11
.BI -i item
Select input item.
.TP 11
.B -j
Calculate jitter track.
.TP 11
.BI -J jittersize
Change number of pitch cycles used in computation.  Default 5.
.TP 11
.B -s
Calculate shimmer track.
.TP 11
.BI -S shimmersize
Change number of pitch cycles used in computation.  Default 5.
.TP 11
.BI -f sfreq
Select sampling frequency of track items. Default 200.
.SH INPUT ITEMS
.IP SP.xx 11
Speech waveform used for shimmer measurement.
.IP TX.xx 11
Any excitation period data item.
.SH OUTPUT ITEMS
.IP TR 11
Jitter track.
.IP TR 11
Shimmer track.
.SH VERSION/AUTHOR
.IP 1.0 11
Mark Huckvale
.SH BUGS
Tracks are sampled, not averaged.
*/
/*--------------------------------------------------------------------------*/

/* standard definitions */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"		/* database structures */

/* input file variables */
char		filename[SFSMAXFILENAME]; /* sfs file name */
struct item_header	spitem;	/* item header for speech data */
short	*sp;
struct item_header txitem;	/* item header for tx data */
int		*tx;

/* output file variables */
int		ffreq = 200;		/* output frame rate */
double	fdur;				/* output frame duration */
int		txlimit;			/* largest interval that is voiced */
int		dojitter=0,doshimmer=0;
int		jsize=5;			/* jitter window size */
int		ssize=5;			/* shimmer window size */
struct item_header jitem;	/* item header for jitter track */
double	*jtab;				/* cycle by cycle jitter values */
struct item_header sitem;	/* item header for shimmer track */
double	*stab;				/* cycle by cycle shimmer values */

/* calculate jitter */
void calcjitter()
{
	int	i,j;
	double	sum;

	for (i=jsize/2;i<txitem.numframes-jsize/2-1;i++) {
		for (sum=0,j=-jsize/2;j<=jsize/2;j++) {
			if (tx[i+j] > txlimit)
				sum += tx[i];
			else
				sum += tx[i+j];
		}
		jtab[i] = 100.0*fabs(1-jsize*tx[i]/sum);
	}
	for (i=0;i<jsize/2;i++)
		jtab[i] = jtab[jsize/2];
	for (i=txitem.numframes-jsize/2-1;i<txitem.numframes;i++)
		jtab[i]=jtab[txitem.numframes-jsize/2-2];
}

/* calculate the speech signal amplitude */
int	peak2peak(int tidx)
{
	int	tx1=0,tx2=0;
	int	sp1,sp2;
	int	min=0,max=1;
	int	i;

	for (i=0;i<tidx;i++) tx1 += tx[i];
	tx2 = tx1 + tx[tidx];
	sp1 = (int)(((txitem.offset + tx1*txitem.frameduration)-spitem.offset)/spitem.frameduration);
	sp2 = (int)(((txitem.offset + tx2*txitem.frameduration)-spitem.offset)/spitem.frameduration);

	if (sp1 >= spitem.numframes) sp1=spitem.numframes-1;
	if (sp2 >= spitem.numframes) sp2=spitem.numframes-1;
	for (i=sp1;i<sp2;i++) {
		if (sp[i] < min) min=sp[i];
		if (sp[i] > max) max=sp[i];
	}
	return (max-min);
}

/* calculate shimmer */
void calcshimmer()
{
	int	i,j;
	double	sum;

	for (i=ssize/2;i<txitem.numframes-ssize/2-1;i++) {
		for (sum=0,j=-ssize/2;j<=ssize/2;j++) {
			if (tx[i+j] > txlimit)
				sum += peak2peak(i);
			else
				sum += peak2peak(i+j);
		}
		stab[i] = 100.0*fabs(1-ssize*peak2peak(i)/sum);
	}
	for (i=0;i<ssize/2;i++)
		stab[i] = stab[ssize/2];
	for (i=txitem.numframes-ssize/2-1;i<txitem.numframes;i++)
		stab[i]=stab[txitem.numframes-ssize/2-2];
}

/* output a track */
void outtrack(int fid,double *tab,int ntab)
{
	int		i;
	double	t=0,txtime=0;
	float	val;

	for (i=0; i < ntab; i++) {

		/* get time at end of tx period */
		txtime += tx[i]*txitem.frameduration;

		/* get value to write */
		if (tx[i] > txlimit)
			val = 0;
		else
			val = (float)tab[i];

		/* write value to file */
		while (t <= txtime) {

			if (sfswrite(fid,1,&val) != 1)
				error("write error on temporary file",NULL);

			t += fdur;
		}
	}
}

/* main program */
void main(int argc,char **argv)
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item type selection */
	char		*ty;		/* item match selection */
	char		*sptype="0";
	char		*txtype="0";

	/* output file variables */
	int		jfid;		/* output file descriptor */
	int		sfid;		/* output file descriptor */

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:f:jJ:sS:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Tx statistics V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					sptype=ty;
				else if (it == TX_TYPE)
					txtype=ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'f' :	/* output sampling freq */
			if ((ffreq=atoi(optarg)) <= 0)
				error("illegal sampling frequency: %s",optarg);
			break;
		case 'j':
			dojitter=1;
			break;
		case 'J':
			jsize = atoi(optarg);
			jsize = jsize | 1;
			if (jsize < 3) error("unsuitable jitter size");
			dojitter=1;
			break;
		case 's':
			doshimmer=1;
			break;
		case 'S':
			ssize = atoi(optarg);
			ssize = ssize | 1;
			if (ssize < 3) error("unsuitable shimmer size");
			doshimmer=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-f frame rate) (-j|-J jittersize) (-s|-S shimmersize) file",PROGNAME);
	if ((dojitter==0)&&(doshimmer==0))
		dojitter = doshimmer = 1;

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no file specified",NULL);

	/* load Tx data */
	getitem(filename,TX_TYPE,txtype,&txitem,&tx);
	if (doshimmer) getitem(filename,SP_TYPE,sptype,&spitem,&sp);
	fdur = 1.0/(double)ffreq;
	txlimit = (int)(0.025/txitem.frameduration);

	/* create output item headers and channels */
	if (dojitter) {
		/* make header */
		sfsheader(&jitem,TR_TYPE,1,4,1,fdur,txitem.offset,1,0,0);
		sprintf(jitem.history,"%s(%d.%02d;jitter,size=%d,rate=%d)",
			PROGNAME,txitem.datatype,txitem.subtype,jsize,ffreq);

		/* open output channel */
		if ((jfid=sfschannel(filename,&jitem)) < 0)
			error("cannot open output channel",NULL);

		/* get buffer */
		jtab = calloc(txitem.numframes,sizeof(double));

	}
	if (doshimmer) {
		sfsheader(&sitem,TR_TYPE,1,4,1,fdur,txitem.offset,1,0,0);
		sprintf(sitem.history,"%s(%d.%02d,%d.%02d;shimmer,size=%d,rate=%d)",
			PROGNAME,spitem.datatype,spitem.subtype,
			txitem.datatype,txitem.subtype,ssize,ffreq);

		/* open output channel */
		if ((sfid=sfschannel(filename,&sitem)) < 0)
			error("cannot open output channel",NULL);

		/* get buffer */
		stab = calloc(txitem.numframes,sizeof(double));
	}

	/* calculate the cycle by cycle values */
	if (dojitter) calcjitter();
	if (doshimmer) calcshimmer();

	/* output the tracks */
	if (dojitter) outtrack(jfid,jtab,txitem.numframes);
	if (doshimmer) outtrack(sfid,stab,txitem.numframes);

	/* update data file */
	if (!sfsupdate(filename))
		error("update error on %s",filename);

	/* that's all folks */
	if (jtab) free(jtab);
	if (stab) free(stab);
	if (tx) free(tx);
	if (sp) free(sp);
	exit(0);
}


