/* wordplay -- play words out of an annotated file for checking */

/* M.A.Huckvale - University College London */

/* version 1.0 - June 1994 */

#define PROGNAME "wordplay"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH WORDPLAY SFS1 UCL
.SH NAME
wordplay -- play word annotated regions of a speech item
.SH SYNOPSIS
.B wordplay
(-n wnum) (-s wnum) (-e wnum) (-i item) (-12) file
.SH DESCRIPTION
.I wordplay
is a program to replay annotated regions of a speech item.  It assumes that individual
words are annotated at the start and end.  If an annotation is labelled simply '/', then
it is skipped.  See
.I replay
for further information about DAC control.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i item
Select input item number.
.TP 11
.BI -n wnum
Replay word number wnum only.  Default: play all words.
.TP 11
.BI -s wnum
Replay starting at word wnum.  Default: word 1.
.TP 11
.BI -e enum
Finish replaying at word wnum.  Default: last word.
.TP 11
.B -12
Scale all unmarked speech items up from 12 bits.
.SH INPUT ITEMS
.IP SP
Any speech item.
.IP AN
Word annotations.
.SH VERSION/AUTHOR
.IP 1.0
Mark Huckvale
.SH SEE ALSO
replay
*/
/*--------------------------------------------------------------------------*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <malloc.h>
#include "sfs.h"

/* global data */
struct item_header	spitem;
short			*sp;
int			spsize;
double			sprate;
int			spbits=16;
struct item_header	anitem;
struct an_rec		*antab;

/* operational modes */
int	wnum = -1;	/* replay all words */
int	snum = 1;
int	fnum = 1000000;
int	doscale12=0;	/* scale from 12 bits */

/* do replay */
void dorep(fid,stime,etime)
int	fid;
double	stime,etime;
{
	int	ssamp,nsamp;

	ssamp = (int)((stime-spitem.offset)/spitem.frameduration);
	nsamp = (int)((etime-spitem.offset)/spitem.frameduration - ssamp);

	if (ssamp < 0) ssamp = 0;
	if (nsamp > (spitem.numframes - ssamp)) nsamp = spitem.numframes-ssamp;
	if (nsamp <= 0) return;
	
	if (spsize==0) {
		if ((sp = (short *)calloc(nsamp,sizeof(short)))==NULL)
			error("could not get buffer");
		spsize = nsamp;
	}
	else if (nsamp > spsize) {
		if ((sp = (short *)realloc(sp,nsamp*sizeof(short)))==NULL)
			error("could not get buffer");
		spsize = nsamp;
	}

	if (sfsread(fid,ssamp,nsamp,sp)!=nsamp)
		error("read error on speech item");

	dac_playback(sp,nsamp,sprate,spbits,1,1);
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item selection */
	char		*ty;		/* item sub type */
	char		*sptype="0";
	char		*antype="0";
	/* file variables */
	char		filename[SFSMAXFILENAME]; /* SFS data file name */
	int		fid;		/* input file descriptor */
	int		i,wcnt;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:n:s:e:1:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Replay words in file V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					sptype = ty;
				else if (it == AN_TYPE)
					antype = ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case '1' :	/* 12-16 bits conversion */
			if (*optarg=='2')
				doscale12++;
			else
				error("bad '-1' option, -12 intended?");
			break;
		case 'n' :	/* replay one word */
			wnum = atoi(optarg);
			break;
		case 's' :	/* start word # */
			snum = atoi(optarg);
			break;
		case 'e' :	/* end word # */
			fnum = atoi(optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-n wnum) (-s wnum) (-e wnum) (-12) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open file */
	if ((fid=sfsopen(filename,"r",NULL)) < 0)
		error("access error on '%s'",filename);

	/* find annotations */
	if (!sfsitem(fid,AN_TYPE,antype,&anitem))
		error("could not find annotation item in '%s'",filename);

	/* load annotations */
	if ((antab = (struct an_rec *)sfsbuffer(&anitem,anitem.numframes))==NULL)
		error("could not get buffer");
	if (sfsread(fid,0,anitem.numframes,antab)!=anitem.numframes)
		error("read error on '%s'",filename);

	/* locate speech */
	if (!sfsitem(fid,SP_TYPE,sptype,&spitem))
		error("could not find speech item in '%s'",filename);

	sprate = 1.0/spitem.frameduration;
	spbits = (int)param(spitem.params,"bits",(doscale12)?12.0:16.0);
	
	/* open output device */
	if (dac_open(NULL)<0)
		error("failed to open replay device");

	/* replay */
	if (wnum > 0) snum = fnum = wnum;
	wcnt=1;
	for (i=0;i<anitem.numframes-1;i++) {
		if (strcmp(antab[i].label,"/")) {
			if ((wcnt >= snum) && (wcnt <= fnum)) {
				printf("%d. %s",wcnt,antab[i].label);
				fflush(stdout);
				dorep(fid,antab[i].posn*anitem.frameduration+anitem.offset,
					  antab[i+1].posn*anitem.frameduration+anitem.offset);
				printf("\n");
			}
			wcnt++;
		}
	}

	/* that's all folks ! */
	dac_close(0);
	sfsclose(fid);
	exit(0);
}

