// butter.cpp -- Butterworth filter design
//
// C++ (c) 1996 Mark Huckvale University College London

#include "tools.h"
#include "ltisys.h"
#include "ltisysch.h"
#include "butter.h"

//	ButterworthPoles()	calculate positions of z-plane
//				poles to Butterworth formula
//	ButterworthLowPass()	build low-pass recursive filter
//				to Butterworth design
//	ButterworthHighPass()	build high-pass recursive filter
//	ButterworthBandPass()	build band-pass recursive filter

// position nsection poles for Butterworth Low-pass
ComplexWaveform ButterworthPoles(
	double freq,		// cut-off frequency (fraction)
	int nsection		// number of 2nd order sections reqd
	)			// returns array of pole positions
				//   on positive imaginary axis only
{
	// get array of complex values
	ComplexWaveform poles(nsection,1);

	// calculate angles
	double w = PI * freq;
	double tanw = sin(w)/cos(w);

	// calculate +im pole position for each section
	for (int m=nsection;m<2*nsection;m++) {
		// Butterworth formula adapted to z-plane
		double ang = (2*m + 1) * PI / (4*nsection);
		double d = 1 - 2*tanw*cos(ang) + tanw*tanw;
		poles[m-nsection+1] =
			Complex((1 - tanw*tanw)/d, 2 * tanw * sin(ang)/d);
	}

	return poles;
}

// build Butterworth low-pass filter
LTISystemChain ButterworthLowPass(
	double freq,			// cut-off frequency (fraction)
	int nsection			// number of 2nd order sections reqd
	)				// returns array of LTI systems
{
	// create empty system chain
	LTISystemChain lpfilt(nsection);

	// get pole positions
	ComplexWaveform pol = ButterworthPoles(freq,nsection);

	// convert each conjugate pole pair to
	//   difference equation
	for (int i=0;i<nsection;i++) {
		lpfilt[i] = LTISystem(2,2);
		// put in conjugate pole pair
		lpfilt[i].b[1] = -2.0 * pol[i+1].real();
		lpfilt[i].b[2] = pol[i+1].real() * pol[i+1].real() +
				 pol[i+1].imag() * pol[i+1].imag();
		// put 2 zeros at (-1,0)
		double tot = 4.0/(1 + lpfilt[i].b[1] + lpfilt[i].b[2]);
		lpfilt[i].a[0] = 1.0/tot;
		lpfilt[i].a[1] = 2.0/tot;
		lpfilt[i].a[2] = 1.0/tot;
	}

	return lpfilt;
}

// build Butterworth high-pass filter
LTISystemChain ButterworthHighPass(
	double freq,		// cut-off frequency (fraction)
	int nsection		// number of 2nd order sections reqd
	)			// returns array of LTI systems }
{
	// create empty system chain
	LTISystemChain hpfilt(nsection);

	// get pole positions for low-pass
	ComplexWaveform pol = ButterworthPoles(0.5-freq,nsection);

	// flip all the poles over to get high pass
	int i;
	for (i=1;i<=nsection;i++)
		pol[i] = Complex(-pol[i].real(),pol[i].imag());

	// convert each conjugate pole pair to
	//   difference equation
	for (i=0;i<nsection;i++) {
		hpfilt[i] = LTISystem(2,2);
		// put in conjugate pole pair
		hpfilt[i].b[1] = -2.0 * pol[i+1].real();
		hpfilt[i].b[2] = pol[i+1].real() * pol[i+1].real() +
				 pol[i+1].imag() * pol[i+1].imag();
		// put 2 zeros at (1,0)
		hpfilt[i].a[0] = 1.0;
		hpfilt[i].a[1] = -2.0;
		hpfilt[i].a[2] = 1.0;
		// normalise to unity gain
		double gain = mag(hpfilt[i].response(0.5));
		hpfilt[i].a[0] = hpfilt[i].a[0]/gain;
		hpfilt[i].a[1] = hpfilt[i].a[1]/gain;
		hpfilt[i].a[2] = hpfilt[i].a[2]/gain;
	}

	return hpfilt;
}

// build Butterworth band-pass filter
LTISystemChain ButterworthBandPass(
	double lofreq,		// low cut-off frequency (fraction)
	double hifreq,		// high cut-off frequency (fraction)
	int nsection		// number of 2nd order sections reqd
	)			// returns array of LTI systems
{
	// force even # sections
	if ((nsection%2)==1) nsection++;

	// create empty system chain
	LTISystemChain bpfilt(nsection);

	// get pole positions for low-pass of equivalent width
	ComplexWaveform pol = ButterworthPoles(hifreq-lofreq,nsection/2);

	// translate the poles to band-pass position
	ComplexWaveform bpol(nsection,1);
	double wlo = 2*PI*lofreq;
	double whi = 2*PI*hifreq;
	double ang = cos((whi+wlo)/2)/cos((whi-wlo)/2);
	int i;
	for (i=1;i<=nsection/2;i++) {
		Complex p1  = Complex(pol[i].real()+1,pol[i].imag());
		Complex tmp = sqrt(p1*p1*ang*ang*0.25-pol[i]);
		bpol[2*i-1] = (p1*ang*0.5) + tmp;
		bpol[2*i]   = (p1*ang*0.5) - tmp;
	}

	// convert each conjugate pole pair to
	//   difference equation
	for (i=0;i<nsection;i++) {
		bpfilt[i] = LTISystem(2,2);
		// put in conjugate pole pair
		bpfilt[i].b[1] = -2.0 * bpol[i+1].real();
		bpfilt[i].b[2] = bpol[i+1].real() * bpol[i+1].real() +
				 bpol[i+1].imag() * bpol[i+1].imag();
		// put zeros at (1,0) and (-1,0)
		bpfilt[i].a[0] = 1.0;
		bpfilt[i].a[1] = 0.0;
		bpfilt[i].a[2] = -1.0;
		// normalise to unity gain
		double gain = mag(bpfilt[i].response((hifreq+lofreq)/2));
		bpfilt[i].a[0] = bpfilt[i].a[0]/gain;
		bpfilt[i].a[1] = bpfilt[i].a[1]/gain;
		bpfilt[i].a[2] = bpfilt[i].a[2]/gain;
	}

	return bpfilt;
}
