// cfft.cpp -- implementation of complex-to-complex Fast Fourier Transform
//
// C++ (c) 1996 Mark Huckvale University College London

#include "tools.h"
#include "cfft.h"

// The iLog2 function returns the integer log based 2
int	iLog2(int x)
{
	int	p=0;
	int	y=1;
	while (y < x) {
		p++;
		y *= 2;
	}
	return p;
}

// The iPow2 function returns the integer power of 2
int	iPow2(int p)
{
	int	x=1;
	while (p > 0) {
		p--;
		x *= 2;
	}
	return x;
}

// The FFTBitReverseTable function returns a dynamically-allocated
// table of indexes for FFT in-place sample shuffling
int *	FFTBitReverseTable(int size)
{
	int	*idx = new int[size];

	// find # bits involved
	int	nbit = iLog2(size);

	// for each table entry
	for (int i=0;i<size;i++) {
		// store bit reversed index
		int a1 = i;
		int a2 = 0;
		for (int j=1;j<=nbit;j++) {
			a2 *= 2;
			if (a1 & 1) a2 |= 1;
			a1 /= 2;
		}
		idx[i] = a2;
	}
	return idx;
}

// The FFTButterfly function performs the key FFT cross-multiply
void FFTButterfly(Complex& upper,Complex& lower,const Complex& w)
{
	Complex temp = lower * w;
	lower = upper - temp;
	upper = upper + temp;
}

// The ComplexFFT function implements a fast complex to
// complex discrete fourier transform

Spectrum ComplexFFT(const ComplexWaveform& x)
{
	int	size = iPow2(iLog2(x.count()));
	Spectrum s(size,size/x.rate());

	// get bit reverse table
	int *amap = FFTBitReverseTable(size);

	// shuffle input data into spectrum
	int i;
	for (i = 0 ; i < size ; i++)
		s[amap[i]] = x[i+1];
		// uses bad index capability of x[]

	// do multiple butterfy passes over data
	//   with steps of 1,2,4,..,N
	for (int d = 1 ; d < size ; d *= 2) {
		// for each start position
		for (int j = 0 ; j < d ; j++) {
			Complex w = exp(Complex(0,-(PI*j)/d));
			// for each step
			for (int i = j ; i < size ; i += 2*d)
				// do butterfly
				FFTButterfly(s[i],s[i+d],w);
		}
	}

	// normalise
	for (i = 0 ; i < size ; i++)
		s[i] /= x.count();

	delete [] amap;

	return s;
}

// The ComplexIFFT function implements a fast complex to
// complex inverse discrete fourier transform

ComplexWaveform ComplexIFFT(const Spectrum& s)
{
	ComplexWaveform x(s.count(),s.count()/s.rate());

	// get bit reverse table
	int *amap = FFTBitReverseTable(x.count());

	// shuffle input data into waveform
	for (int i = 0 ; i < s.count() ; i++)
		x[i+1] = s[amap[i]];

	// do multiple butterfy passes over data
	//   with steps of 1,2,4,..,N
	for (int d = 1 ; d < s.count() ; d *= 2) {
		// for each start position
		for (int j = 0 ; j < d ; j++) {
			Complex w = exp(Complex(0,(PI*j)/d));
			// for each step
			for (int i = j+1 ; i <= s.count() ; i += 2*d)
				// do butterfly
				FFTButterfly(x[i],x[i+d],w);
		}
	}

	delete [] amap;

	return x;
}

