// complex.cc - implementation for complex arithmetic class
//
// (c) 1996 Mark Huckvale University College London

#include <iostream.h>
#include <stdlib.h>
#include "complex.h"

// update operators
Complex& Complex::operator+= (const Complex& c)
{
	re += c.re;
	im += c.im;
	return *this;
}
Complex& Complex::operator-= (const Complex& c)
{
	re -= c.real();
	im -= c.imag();
	return *this;
}
Complex& Complex::operator*= (const Complex& c)
{
	double tre = re * c.real() - im * c.imag();
	im = re * c.imag() + im * c.real();
	re = tre;
	return *this;
}
Complex& Complex::operator/= (const Complex& c)
{
	double	denom =  c.real() * c.real() + c.imag() * c.imag();
	if (denom==0) {
		cerr << "Complex: division by zero.\n";
		exit(1);
	}
	double tre = (re * c.real() + im * c.imag())/denom;
	im = (im * c.real() - re * c.imag())/denom;
	re = tre;
	return *this;
}

// arithmetic operators
Complex operator + (const Complex& c1,const Complex& c2)
{
	return Complex(c1.real()+c2.real(),c1.imag()+c2.imag());
}
Complex operator - (const Complex& c1,const Complex& c2)
{
	return Complex(c1.real()-c2.real(),c1.imag()-c2.imag());
}
Complex operator * (const Complex& c1,const Complex& c2)
{
	return Complex( c1.real() * c2.real() - c1.imag() * c2.imag(),
			c1.real() * c2.imag() + c1.imag() * c2.real() );
}
Complex operator / (const Complex& c1,const Complex& c2)
{
	double	denom =  c2.real() * c2.real() + c2.imag() * c2.imag();
	if (denom==0) {
		cerr << "Complex: division by zero.\n";
		exit(1);
	}
	return Complex( (c1.real() * c2.real() + c1.imag() * c2.imag()) / denom,
			(c1.imag() * c2.real() - c1.real() * c2.imag()) / denom );
}

// complex functions
double	mag(const Complex& c)
{
	return sqrt(c.real()*c.real() + c.imag()*c.imag());
}
double	arg(const Complex& c)
{
	if ((c.imag()==0) && (c.real()==0))
		return 0.0;
	else
		return atan2(c.imag(),c.real());
}
Complex	exp(const Complex& c)
{
	double r = exp(c.real());
	return Complex(r * cos(c.imag()),r * sin(c.imag()));
}
Complex	sqrt(const Complex& c)
{
	double	r = sqrt(c.real()*c.real()+c.imag()*c.imag());
	if (r==0)
		return Complex(0.0,0.0);
	else {
		double w = sqrt((fabs(c.real())+r)*0.5);
		double d = c.imag()/(2.0*w);
		if (c.real() > 0)
			return Complex(w,d);
		else if (c.imag() >= 0.0)
			return Complex(d,w);
		else
			return Complex(-d,-w);
	}
}

// iostream functions
istream& operator >> (istream& ip, Complex& c)
{
	double	r,i;
	char	ch;
	ip >> ws;	// skip whitespace
	ip.get(ch);
	if (ch == '(') {
		//  accept real,imag
		ip >> r;
		ip >> ws;
		ip.get(ch);
		if (ch == ',') {
			ip >> i;
			ip >> ws;
			ip.get(ch);
		}
		else
			i = 0;
		if (ch != ')')
			ip.clear(ios::failbit);
	}
	else {
		// accept a pure real
		ip.putback(ch);
		ip >> r;
		i = 0;
	}
	c = Complex(r,i);
	return ip;
}

ostream& operator << (ostream& op, const Complex& c)
{
	return op << "(" << c.real() << "," << c.imag() << ")";
}

#ifdef EMO
void main()
{
	Complex	a(0.0,0.0);

	cout.setf(ios::fixed);

	cout << "Enter a complex number (r,i) : ";
	cout.flush();

	cin >> a;

	cout << "a=" << a << "\n";

	Complex b(-1,0);
	cout << "Square root of -1 is : " << sqrt(b) << "\n";
	
	a = Complex(1.0,2.0);
	b = Complex(3.0,4.0);
	cout << a << "+" << b << "=" << a+b << "\n";
	cout << a << "-" << b << "=" << a-b << "\n";
	cout << a << "*" << b << "=" << a*b << "\n";
	cout << a << "/" << b << "=" << a/b << "\n";
	Complex c = a/b;
	cout << c << "*" << b << "=" << c*b << "\n";
	Complex d = sqrt(c);
	cout << "sqrt(" << c << ")=" << d << "\n";
	cout << d << "*" << d << "=" << d*d << "\n";
	cout << "magnitude(" << b << ")=" << mag(b) << "\n";
	cout << "4*argument(" << Complex(4,4) << ")=" << 4*arg(Complex(4,4)) << "\n";
	cout << "exp(" << b << ")=" << exp(b) << "\n";
	b = Complex(0,3.14159);
	cout << "exp(" << b << ")=" << exp(b) << "\n";
	cout << "4.0*" << b << "=" << 4.0*b << "\n";

	const Complex w(sqrt(0.5),sqrt(0.5));
	Complex e=w;
	for (int i=0;i<8;i++) {
		cout << i << "\t" << e << "\t" << 180.0*arg(e)/3.14159 << "\n";
		e = e * w;
	}

	a = Complex(1,2);
	b = Complex(3,4);
	cout << a << "+=" << b << "=";
	a += b;
	cout << a << "\n";
	a = Complex(1,2);
	b = Complex(3,4);
	cout << a << "-=" << b << "=";
	a -= b;
	cout << a << "\n";
	a = Complex(1,2);
	b = Complex(3,4);
	cout << a << "*=" << b << "=";
	a *= b;
	cout << a << "\n";
	a = Complex(1,2);
	b = Complex(3,4);
	cout << a << "/=" << b << "=";
	a /= b;
	cout << a << "\n";
}
#endif
