/* dspc -- driver program for C++ compilation with DSP classes */

/* Mark Huckvale - University College London */

/* version 1.0 - October 1996 */

#define PROGNAME "dspc"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH DSPC SFS1 UCL
.SH NAME
DSPC - Digital Signal Processing Classes
.SH SYNOPSIS
.B dspc
[ options ] file.cpp
.SH DESCRIPTION
.I dspc
is a program to compile a C++ program that uses the SFS digital
signal processing class library.  It uses the system C++ compiler
which must be first installed and operational.  The compile line
to link with the appropriate SFS libraries is created automatically.
Compile errors are folded back into the source as comments.
.PP
Documentation is still under construction for the class library.
In the meantime, study the include files and examples.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.B "other options"
All other options are passed to the C++ compiler.
.SH FILES
.IP $(SFSBASE)\include\dspc
DSP classes include files
.IP $(SFSBASE)\lib\dspc.lib
DSP classes library
.SH VERSION/AUTHOR
.IP 1.0
Mark Huckvale
.SH SEE ALSO
spc
*/
/*--------------------------------------------------------------------------*/


#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <fcntl.h>
#if (defined(DJGPP1) || defined(DJGPP2))
#include <process.h>
#endif
#include "sfs.h"

/* global variables */
char	filename[80];
FILE	*ip;
FILE	*op;
FILE	*ep;
int	lno;
int	errcnt;
int	conly=0;

/* 'C' compiler command line */
char	*cargv[50];
int	cargc=0;

/* compiler command - has sfsbase() as arguments */
char 	*comparg[8]={
#ifdef DJGPP2
	"redir -eo",
#endif
#ifdef SUN4
	"g++ -Werror -Wall -I%s/include",
#elif defined(MSC32)
	"cl -Zp -W3 -I\"%s\\include\" -MD -D_POSIX_ -D__STDC__ -DWIN32",
	"-I\"%s\\include\\dspc\"",
#else
	"gcc -Werror -Wall -I%s/include",
	"-I%s/lang/dspc",
#endif
#if defined(MSC32)
#else
	"-L%s/lib",
#endif
};

/* extended linkage - from makefile (use SPCLINK for extra libs) */
char	*xtdlink[8]={
#ifdef MSC32
	"\"%s\\lib\\dspc.lib\"",
	"\"%s\\lib\\dig.lib\"",
	"\"%s\\lib\\dsp.lib\"",
	"\"%s\\lib\\sfs.lib\"",
#else
	"-ldspc -ldig -ldsp -lsfs",
#endif
#ifdef SPCLINK
	SPCLINK,
#endif
#ifdef MSC32
	"-link -subsystem:console",
#else
	"-lm",
#endif
};

/* Executable extensions */
char *exextn={
#if defined(DJGPP1) || defined(DJGPP2) || defined(MSC32)
	".exe"
#else
	""
#endif
};

/* system call buffer */
char systemcall[1024];

/* add string components to carg array */
void addstring(char *str)
{
	char	*p,*strsave();

	p = strtok(str," ");
	while (p && *p) {
		cargv[cargc++] = strsave(p);
		p = strtok(NULL," ");
	}
}

/* print a file to standard output */
void printfile(fname)
char *fname;
{
	FILE	*ip;
	int	c;

	if ((ip=fopen(fname,"r"))==NULL) {
		fprintf(stderr,"could not open '%s'",fname);
		fclose(ip);
		return;
	}

	while ((c=getc(ip))!=EOF)
		putchar(c);

	fclose(ip);
}

/* print a file to standard output */
int errcomments(fname)
char *fname;
{
	FILE	*ip;
	char	iline[256];

	if ((ip=fopen(fname,"r"))==NULL)
		return(0);

	while (fgets(iline,256,ip)) {
		if (strncmp(iline,"//$ERR",6)==0) {
			fclose(ip);
			return(1);
		}
	}

	fclose(ip);
	return(0);
}

char *strsave(str)
char *str;
{
	char *p = malloc(strlen(str)+1);
	strcpy(p,str);
	return(p);
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	int		optind;
	char		basename[80];
	char		extname[16];
	char		pfilename[80];
	char		ofilename[80];
	char		efilename[80];
	char		xfilename[80]; 
	char		*p;
	int		i;
	char		tempbuf[256];
#if (defined(DJGPP1) || defined(DJGPP2))
	int		oldout,fid;
#endif

	xfilename[0]='\0';

	/* initialise C command line */
	for (i=0;comparg[i];i++) {
		sprintf(tempbuf,comparg[i],sfsbase());
		addstring(tempbuf);
	}

	/* scan compiler agruments */
	for (optind=1;optind<argc;optind++) {
		if (argv[optind][0]=='-') {
			/* a switch of some kind */
			switch (argv[optind][1]) {
			case 'I':	/* identify */
				fprintf(stderr,"%s: DSP Classes C++ Compiler Vs%s\n",
					PROGNAME,PROGVERS);
				exit(0);
			default:
				/* pass any other flags over to C command line */
				cargv[cargc++] = strsave(argv[optind]);
				break;
			}
		}
		else {
			/* a filename */
			/* get base part and extension */
			strcpy(basename,argv[optind]);
			strcpy(extname,"");
			if ((p=strrchr(basename,'.'))!=NULL) {
				*p = '\0';
				strcpy(extname,p+1);
			}

			/* do action */
			if (strcmp(extname,"cpp")==0) {
				/* C++ */
				strcpy(pfilename,argv[optind]);
				strcpy(ofilename,basename);
				strcat(ofilename,".o");
				strcpy(efilename,basename);
				strcat(efilename,".err");
				if (!xfilename[0]) {
					strcpy(xfilename,basename);
					strcat(xfilename,exextn);
				}
				cargv[cargc++] = strsave(pfilename);
			}
			else {
				/* pass anything else on to 'C' compiler */
				cargv[cargc++] = strsave(argv[optind]);
			}
		}
	}

	/* put in mandatory link components */
	for (i=0;xtdlink[i];i++) {
		sprintf(tempbuf,xtdlink[i],sfsbase());
		addstring(tempbuf);
	}

	/* put output name */
#ifdef MSC32
	sprintf(tempbuf,"/OUT:%s",xfilename);
	addstring(tempbuf);
#else
	cargv[cargc++] = "-o";
#if DJGPP1
	sprintf(outfilename,"%s.out",basename);
	cargv[cargc++] = outfilename;
#else
	cargv[cargc++] = xfilename;
#endif
#endif

	/* OK call 'C' compiler with stdout > spc.err */
#if (defined(DJGPP1) || defined(DJGPP2))
#ifdef SHOW
	for (i=0;i<cargc;i++) printf("%s ",cargv[i]);
	printf("\n");
#endif
	unlink("dspc.err");
	fid=open("dspc.err",O_CREAT|O_WRONLY,0664);
	oldout = dup(1);
	dup2(fid,1);
	i = spawnvp(P_WAIT,cargv[0],cargv);
	dup2(oldout,1);
	close(fid);
#else
	strcpy(systemcall,cargv[0]);
	for (i=1;i<cargc;i++) {
		strcat(systemcall," ");
		strcat(systemcall,cargv[i]);
	}
	strcat(systemcall," >dspc.err");
#ifndef MSC32
	strcat(systemcall," 2>&1");
#endif

#ifdef SHOW
puts(systemcall); 
#endif
	i=system(systemcall);
#endif

	/* check for compiler errors */
	if (i) {
		printfile("dspc.err");
		sprintf(systemcall,"dspcerr %s dspc.err",pfilename);
#ifdef SHOW
puts(systemcall); 
#endif
		system(systemcall);
		unlink("dspc.err");
		exit(1);
	}

#ifdef DJGPP1
	/* create protected mode .EXE */
	sprintf(systemcall,"coff2exe %s",outfilename);
#ifdef SHOW
puts(systemcall);
#endif
	system(systemcall);
	unlink(outfilename);
#endif

	/* all is well */
	fprintf(stderr,"No errors\n");
	if (errcomments(pfilename)) {
		sprintf(systemcall,"dspcerr %s",pfilename);
#ifdef SHOW
puts(systemcall); 
#endif
		system(systemcall);
	}
	unlink("dspc.err");
	unlink(efilename);
	unlink(ofilename);
	exit(0);
}

