// waveplot.cpp -- implementation for WaveShort plotting
//
// (c) 1996 Mark Huckvale University College London

// This module supports plotting of simple XY graphs for
// the dynamic WaveShort classes

#include <iostream.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>

#include "waveshor.h"
#include "wavedoub.h"
#include "wavecomp.h"
#include "graph.h"

#define MAXSCREENWIDTH	3000	// 10" @ 300dpi

// plot WaveShort graph
//
// takes a potentially long WaveShort and compresses to a small
// number of doubles for plotting.
void WaveShort::plot(Graph& gr,int gno,const char *title,const char *xt,const char *yt) const
{

	gr.setgtitle(title);
	gr.setxtitle(xt);
	gr.setytitle(yt);
	gr.settype(GRAPH_LINE);

	// squash samples into pixels
	int	nsq = 1 + (siz / MAXSCREENWIDTH);
	int	nwv = siz/nsq;

	// get suitable temporary buffer
	double 	*y = new double[2*nwv];

	// map block of samples to two double values
	int	min,max;
	int	pos=0;
	for (int i=0;i<nwv;i++) {
		min = 32768; max= -32768;
		for (int j=0;j<nsq;j++) {
			if (buf[pos] < min) min = buf[pos];
			if (buf[pos] > max) max = buf[pos];
			pos++;
		}
		y[2*i] = min;
		y[2*i+1] = max;
	}

	// adjust timescale according to squashing
	static double x[2];
	x[0]=0.0;
	x[1]=tau * nsq / 2;
	gr.setxdata(GRAPH_X_INC,x);

	// call plot routine from graph class
	gr.plot(y,2*nwv,gno);

	delete y;
}

void WaveDouble::plot(Graph& gr,int gno,const char *title,const char *xt,const char *yt) const
{
	// set axis labels
	gr.setgtitle(title);
	gr.setxtitle(xt);
	gr.setytitle(yt);
	gr.settype(GRAPH_LINE);

	// set timescale
	static double x[2];
	x[0]=0.0;
	x[1]=tau;
	gr.setxdata(GRAPH_X_INC,x);

	// call plot routine from graph class
	gr.plot(buf,siz,gno);
}

void WaveComplex::plot(Graph& gr,int gno,const char *title,const char *xt,const char *yt) const
{
	magplot(gr,gno,title,xt,yt);
}

void WaveComplex::realplot(Graph& gr,int gno,const char *title,const char *xt,const char *yt) const
{
	// call plot routine from WaveDouble class
	WaveDouble wv = realtoWaveDouble();
	wv.plot(gr,gno,title,xt,yt);
}
void WaveComplex::imagplot(Graph& gr,int gno,const char *title,const char *xt,const char *yt) const
{
	// call plot routine from WaveDouble class
	WaveDouble wv = imagtoWaveDouble();
	wv.plot(gr,gno,title,xt,yt);
}
void WaveComplex::magplot(Graph& gr,int gno,const char *title,const char *xt,const char *yt) const
{
	// call plot routine from WaveDouble class
	WaveDouble wv = magtoWaveDouble();
	wv.plot(gr,gno,title,xt,yt);
}
void WaveComplex::argplot(Graph& gr,int gno,const char *title,const char *xt,const char *yt) const
{
	// call plot routine from WaveDouble class
	WaveDouble wv = argtoWaveDouble();
	wv.plot(gr,gno,title,xt,yt);
}
void WaveComplex::dbplot(Graph& gr,int gno,const char *title,const char *xt,const char *yt) const
{
	// call plot routine from WaveDouble class
	WaveDouble wv = magtoWaveDouble();
	// convert to decibels
	int off=wv.first();
	for (int i=0;i<wv.count();i++)
		if (wv[off+i] > 1.0E-5)
			wv[off+i] = 20.0*log10(wv[off+i]);
		else
			wv[off+i] = -100.0;
	wv.plot(gr,gno,title,xt,yt);
}

#ifdef EMO
void main()
{
	Graph g;
	g.settitle("Example WaveShort Graphs");

	WaveShort sine(1000);
	int i;
	for (i=0;i<1000;i++)
		sine[i] = (short)(20000 * sin(i*2.0*3.14159/250.0));
	sine.plot(g,1,"Short Integer","Time","Amp");

	g.wait();
	g.setsize(2,3);
	sine.plot(g,1,"Short Integer","Time","Amp");

	WaveDouble siner = sine.toWaveDouble();
	siner.plot(g,2,"Real","Time","Amp");

	WaveComplex sinec(10000);
	for (i=0;i<10000;i++) {
		sinec[i] = Complex(20000 * sin(i*2.0*3.14159/2500.0),
			           10000 * cos(i*2.0*3.14159/2500.0));
	}

	sinec.magplot(g,3,"Complex magnitude","Time","Magnitude");
	sinec.argplot(g,4,"Complex argument","Time","Argument");
	sinec.realplot(g,5,"Complex real","Time","Amplitude");
	sinec.imagplot(g,6,"Complex imaginary","Time","Amplitude");

}
#endif

