/* dprec -- simple dynamic programming whole-word template recogniser
	 -- with simple ecological template lexicon */

/* Mark Huckvale - University College London */

/* version 1.0 - November 1996 */
/* version 1.2 - January 1997
	- add k-nearest neighbour classification */
/* version 1.3 - February 1997
	- add feature-based voting */
	
#define PROGNAME "dprec"
#define PROGVERS "1.3"
char *progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH DPREC SFS1 UCL
.SH NAME
dprec - whole word template recogniser
.SH SYNOPSIS
.B dprec
(-l|-r|-p) (-d database) (-f num) (-i item) (-t token) (-a) (-c) (-s)
(-S top) (-n nearest) (-F features) file(s)
.SH DESCRIPTION
.I dprec
is an isolated word recognizer implemented using a dynamic programming algorithm.
It maintains a set of templates (the database) from one run to the next, so that it
can add templates when it fails to recognise a word correctly.  If a template
causes many mis recognitions then it can be forgotten from the database.
.PP
The 
.I dp
program operates using a Euclidean metric between vectors of a "Coefficient" (CO) data
set in the files (identified by the "-i item" switch).  To identify a token, the 'token'
field in the main header of the input file is used.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program name and version number.
.TP 11
.B -l
Learn mode.  If an input word is not recognised correctly, then add it to
the database of templates.
.TP 11
.B -r
Recognition mode.  Find the closest token to each input file using the
database of templates.  Default.
.TP 11
.B -p
Purge database of unused templates. (Non functional)
.TP 11
.BI -d database
Specify the file to use to collect templates.  This is just an SFS file.
Default: dprec.sfs.
.TP 11
.BI -f num
Cause templates that cause more than num misrecognitions to be forgotten.
.TP 11
.BI -i item
Select input item number.
.TP 11
.B -a
Save all templates seen during training.
.TP 11
.B -c
Clear all scores of recognition success/failure (see -f switch)
.TP 11
.B -s
Print scores of recognition success/failure for each template.
.TP 11
.BI -S topnum
Display list of topnum best matches and distances.
.TP 11
.BI -n num
Use a k nearest neighbour voting system with num neighbours.  Max number
of neighbours=100.
.TP 11
.BI -F featurefile
Perform voting by accumulating scores by features rather than by tokens.
'featurefile' contains a feature breakdown for each template token name.
Each line contains the token name and list of features separated by
spaces, e.g. "ba BILABIAL VOICED STOP".
.SH INPUT ITEMS
.IP CO.xx 11
Any Coefficient item.
.SH VERSION/AUTHOR
.IP 1.3 11
Mark Huckvale
.SH BUGS
Although the database of templates looks like an SFS file, in fact the
header token field is used to maintain the template match history, so
items may not be added or deleted by external programs.
*/
/*--------------------------------------------------------------------------*/

/* include files */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <malloc.h>
#include <math.h>	/* maths header */
#include "sfs.h"	/* database structures */

/* main header for database */
char	dbfilename[SFSMAXFILENAME]={"dbrec.sfs"};
struct main_header dbhead;

/* manifest constants */
#define MAXRECORD sizeof(dbhead.token)	/* max number of recorded templates */
#define MAXTEMPLATE 1000		/* max number of reference templates */
#define EDGE	10.0E6			/* large value for dp algorithm */

/* database of templates */
struct {
	char		*token;		/* token name */
	struct co_rec	*tco;		/* coefficient data */
	int		nco;		/* # coefficient frames */
	int		score;		/* template score */
	float		distance;	/* DP distance */
	int		new;
} dtable[MAXTEMPLATE];
int	ntemplate;
int	nactive;
int	ncoeff;
int	minidx;
float	mindist;
double	framedur;
int	forgetthresh=-100;

/* neighbours */
#define MAXNEIGHBOUR 101
struct neighbour_rec {
	int	tno;		/* template number */
	double	score;		/* score */
	double	ftotal;		/* feature total */
} ntable[MAXNEIGHBOUR];
int	ncount;
int	nlimit;

/* feature analysis */
char	featurefile[128];
#define MAXTOKEN	64
#define MAXFEATURE	16
struct feature_rec {
	char	*token;
	char	*feat[MAXFEATURE];
	int	nfeat;
} ftable[MAXTOKEN];
int fcount;

/* input file token */
char	tokname[128];
struct main_header rhead;
struct item_header ritem;
struct co_rec	   *rco;

/* dp matching buffers */
#define MAXTEMPLENGTH	1024
float	cudist1[MAXTEMPLENGTH];
float	cudist2[MAXTEMPLENGTH];

/* store string in dynamic memory */
char *strsave(char *str)
{
	char *p = malloc(strlen(str)+1);
	strcpy(p,str);
	return(p);
}

/* load templates into table */
int	loadtemplates()
{
	int	fid;
	struct item_header	item;

	ntemplate=0;
	if ((fid=sfsopen(dbfilename,"r",&dbhead))<0)
		return(1);
	
	while (sfsnextitem(fid,&item)) if (item.datatype==CO_TYPE) {
		if (ncoeff==0)
			ncoeff = SFSRECSIZE(&item);
		else if (ncoeff != SFSRECSIZE(&item))
			error("mismatch in CO record size");
		if (framedur==0) framedur=item.frameduration;
		dtable[ntemplate].tco = sfsbuffer(&item,item.numframes);
		if (dtable[ntemplate].tco==NULL)
			error("out of memory");
		sfsread(fid,0,item.numframes,dtable[ntemplate].tco);
		dtable[ntemplate].nco = item.numframes;
		if (ntemplate < MAXRECORD)
			dtable[ntemplate].score = (int)(signed char)dbhead.token[ntemplate];
		else
			dtable[ntemplate].score = 0;
		dtable[ntemplate].token = strsave(item.params);
		ntemplate++;
	}

	sfsclose(fid);
	return(0);
}

/* add a template to table */
int addtemplate(struct co_rec *tco,int nco,char *token)
{
	if (ntemplate >= MAXTEMPLATE) return(-1);
	dtable[ntemplate].token = strsave(token);
	dtable[ntemplate].tco = tco;
	dtable[ntemplate].nco = nco;
	dtable[ntemplate].score = 0;
	dtable[ntemplate].distance = 0;
	dtable[ntemplate].new = 1;
	ntemplate++;
	return(0);
}

/* update database from table */
int savetemplates(int purge)
{
	int	i;
	struct item_header item;
	int	ofid;
	int	cnt=0;

	/* rewrite header */
	for (i=0;i<ntemplate;i++) {
		if (i < MAXRECORD)
			dbhead.token[i] = (signed char)dtable[i].score;
		if (dtable[i].score >= 0) nactive++;
	}
	if (sfsopen(dbfilename,"h",&dbhead) != 0)
		error("write error on %s",dbfilename);

	/* check templates for any changes */	
	for (i=0;i<ntemplate;i++) if (dtable[i].new) {
		sfsheader(&item,CO_TYPE,1,4,(4*ncoeff+sfsstruct[CO_TYPE])/4,
			framedur,0.0,0,0,1);
		sprintf(item.history,"dprec(template=%d,token=%s)",i+1,dtable[i].token);
		strcpy(item.params,dtable[i].token);
		ofid = sfschannel(dbfilename,&item);
		sfswrite(ofid,dtable[i].nco,dtable[i].tco);
		cnt++;
		if (cnt > 10) {
			sfsupdate(dbfilename);
			cnt=0;
		}
	}
	if (cnt > 0)
		sfsupdate(dbfilename);
	return(0);
}

/* load feature analysis */
void loadfeatures(char *fname)
{
	FILE	*ip;
	char	iline[256];
	char	*p;

	if ((ip=fopen(fname,"r"))==NULL)
		error("could not open '%s'",fname);
	fcount=0;
	while (fgets(iline,256,ip)) {
		p = strtok(iline," \t\n");
		if (p && *p) {
			ftable[fcount].token = strsave(p);
			p = strtok(NULL," \t\n");
			ftable[fcount].nfeat=0;
			while (p && *p) {
				ftable[fcount].feat[ftable[fcount].nfeat++] =
					strsave(p);
				p = strtok(NULL," \t\n");
			}
			fcount++;
		}
	}
	fclose(ip);
}	

/* distance metric for coeff frames */
float metric(a,b)
struct co_rec *a,*b;
{
	register int i;
	register float sumsq=0,val,*ap,*bp;

	ap = &a->data[0];
	bp = &b->data[0];
	for (i=ncoeff;i;i--) {
		val = *ap++ - *bp++;
		sumsq += val*val;
	}
	return(sqrt(sumsq));
}

#define MINOF(x,y,z) (((x) < (y)) ? (((x) < (z)) ? (x) : (z)) : (((y) < (z)) ? (y) : (z)))

/* Dynamic Programming Metric */
float	dpm(struct co_rec *tco,int ntco,struct co_rec *rco,int nrco)
{
	register int	i,j;		/* loop counters */
	float		up,left,diag;	/* path values for three DP paths */
	float		*cuthis = cudist1;
	float		*culast = cudist2;
	float		*tmp;
	float		dist;
	
	/* initialisation */
	culast[0] = 0;
	for (i=1;i<=nrco;i++) culast[i] = EDGE;

	/* DP match */
	for (i=1;i<=ntco;i++) {
		cuthis[0] = EDGE;
		for (j=1;j<=nrco;j++) {
			/* get distance */
			dist = metric(&tco[i-1],&rco[j-1]);
			/* diagonal path */
			up   = culast[j];
			diag = culast[j-1];
			left = cuthis[j-1];
			/* store best path so far */
			cuthis[j] = dist + MINOF(diag,up,left);
		}
		/* swap over buffers */
		tmp=culast;
		culast=cuthis;
		cuthis=tmp;
	}

	/* return final distance */
	return culast[nrco-1] / (float)(ntco+nrco);
}

/* add a template score to neighbur table */
void	addscore(int tno,float score)
{
	int	i = ncount;
/*
fprintf(stderr,"%-16s score=%g\n",dtable[tno].token,score);
*/
	while ((i>0) && (score < ntable[i-1].score)) {
		ntable[i-1] = ntable[i];
		i--;
	}
	if (i < nlimit) {
		ntable[i].tno = tno;
		ntable[i].score = score;
	}
	ncount++;
	if (ncount > nlimit) ncount=nlimit;
}

/* collapse scores by token */
int	collapsescores()
{
	int	i,j;
	float	max;

	/* invert scores */
	for (i=0;i<ncount;i++)
		if (ntable[i].score==0)
			ntable[i].score = EDGE;
		else
			ntable[i].score = 1.0/ntable[i].score;

	/* collapse scores by template */
	for (i=0;i<ncount;i++) {
		for (j=i+1;j<ncount;j++)
			if (strcmp(dtable[ntable[i].tno].token,
				   dtable[ntable[j].tno].token)==0) {
				ntable[i].score += ntable[j].score;
				ntable[j].score=0;
			}
	}
	/* return best sum inverse score */
	j=0;
	max=ntable[0].score;
	for (i=0;i<ncount;i++) {
		if (ntable[i].score > max) {
			max = ntable[i].score;
			j = i;
		}
/*
fprintf(stderr,"%d. %-16s %g\n",i,dtable[ntable[i].tno].token,ntable[i].score);
*/
	}
	return(ntable[j].tno);
}
	
/* collapse scores by feature */
int	collapsescoresbyfeature()
{
	int	i,j,k,l,m,n;
	float	max;

	/* invert scores */
	for (i=0;i<ncount;i++) {
		if (ntable[i].score==0)
			ntable[i].score = EDGE;
		else
			ntable[i].score = 1.0/ntable[i].score;
		ntable[i].ftotal = 0;
	}

	/* for each of the top scores */
	for (i=0;i<ncount;i++) {
		/* find feature entry */
		for (j=0;j<fcount;j++)
			if (strcmp(ftable[j].token,dtable[ntable[i].tno].token)==0) break;
		if (j==fcount) continue;
		/* add scores to templates with matching features */
		for (k=0;k<ncount;k++) {
			/* find feature entry */
			for (l=0;l<fcount;l++)
				if (strcmp(ftable[l].token,dtable[ntable[k].tno].token)==0) break;
			if (l==fcount) continue;
			for (m=0;m<ftable[j].nfeat;m++)
				for (n=0;n<ftable[l].nfeat;n++)
					if (strcmp(ftable[j].feat[m],ftable[l].feat[n])==0)
						ntable[k].ftotal += ntable[i].score;
		}
	}

	/* return best sum inverse score */
	j=0;
	max=ntable[0].ftotal;
	for (i=0;i<ncount;i++) {
		if (ntable[i].ftotal > max) {
			max = ntable[i].ftotal;
			j = i;
		}

/*
fprintf(stderr,"%2d. %-16s %16g %16g\n",i,dtable[ntable[i].tno].token,
	ntable[i].score,ntable[i].ftotal);
*/
	}
	return(ntable[j].tno);
}
	
/* recognise a token */
int	recognise(char *fname,int fid,char *cotype,char *token)
{
	int	i;

	if (!sfsitem(fid,CO_TYPE,cotype,&ritem))
		error("cannot find input CO item in %s",fname);
	if (ncoeff==0)
		ncoeff = SFSRECSIZE(&ritem);
	else if (ncoeff != SFSRECSIZE(&ritem))
		error("mismatch in number of coefficients");
	if (framedur==0)
		framedur = ritem.frameduration;
	if ((rco=(struct co_rec *)sfsbuffer(&ritem,ritem.numframes))==NULL)
		error("out of memory");
	if (sfsread(fid,0,ritem.numframes,rco)!=ritem.numframes)
		error("read error on '%s'",fname);

	/* check if templates to match against */
	if (ntemplate==0) return(0);	/* fail */

	/* calculate matches to database */
	ncount=0;
	for (i=0;i<ntemplate;i++) {
		if (dtable[i].score > forgetthresh)
			dtable[i].distance = dpm(dtable[i].tco,dtable[i].nco,
						rco,ritem.numframes);
		else
			dtable[i].distance = EDGE;
		if (nlimit) addscore(i,dtable[i].distance);
	}

	/* find best */
	if (nlimit==0) {
		/* nearest neighbour */
		mindist = dtable[0].distance;
		minidx = 0;
		for (i=1;i<ntemplate;i++)
			if (dtable[i].distance < mindist) {
				mindist = dtable[i].distance;
				minidx = i;
			}
	}
	else if (fcount > 0) {
		minidx = collapsescoresbyfeature();
	}
	else {
		minidx = collapsescores();
	}

	/* check correct */
	if (strcmp(dtable[minidx].token,token)==0)
		return(1);
	else
		return(0);
}

/* check if word could have been recognised */
int isnewword(char *token)
{
	int	i;
	for (i=0;i<ntemplate;i++)
		if (strcmp(token,dtable[i].token)==0)
			return(0);
	return(1);
}

/* main program */
void main(argc, argv)
int argc ;
char *argv[] ;
{
	/* option decoding variables */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;
	char		*ty;
	char		*cotype="0";	/* default data type = last */

	/* control variables */
	int	dolearn=0;
	int	dopurge=0;
	int	doforget=0;
	int	dosaveall=0;
	int	doclear=0;
	int	dostatus=0;
	int	dofeatures=0;
	int	doscores=0;
	int	numscore=0;
	
	/* processing */
	int	fid;
	char	*p;
	int	ncor=0,ntot=0;
	int	i,j,idx;
	double	last,diff;

	/* decode switches */
	while ((c = getopt(argc,argv,"Ilrpd:f:t:i:acsn:F:S:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Dynamic programming recogniser V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'l':
			dolearn=1;
			dopurge=0;
			break;
		case 'r':
			dolearn=0;
			dopurge=0;
			break;
		case 'p':
			dolearn=0;
			dopurge=1;
			fprintf(stderr,"%s: purge not yet implemented\n",PROGNAME);
			break;
		case 'd':
			strcpy(dbfilename,optarg);
			break;
		case 'f':
			doforget=1;
			forgetthresh = -atoi(optarg);
			break;
		case 't':
			strcpy(tokname,optarg);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == CO_TYPE)
					cotype = ty;
				else 
					error("unsuitable item specifier %s",optarg);
		  	}
		  	else 
				error("illegal item specifier %s",optarg);
			break;
		case 'a' :	/* save all in training */
			dosaveall=1;
			break;
		case 'c' :	/* clear scores */
			doclear=1;
			break;
		case 's' :	/* print template status */
			dostatus=1;
			break;
		case 'S' :	/* print template scores */
			doscores=1;
			numscore = atoi(optarg);
			break;
		case 'n' :	/* k nearest neighbour voting */
			nlimit = atoi(optarg);
			if (nlimit < 0) nlimit=0;
			if (nlimit >= MAXNEIGHBOUR) nlimit=MAXNEIGHBOUR-1;
			break;
		case 'F' :	/* feature-based voting */
			strcpy(featurefile,optarg);
			dofeatures=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-l|-r|-p) (-d database) (-f forgetnum) (-i item) (-t token) (-i item) (-a) (-c) (-s) (-S) (-n nearest) (-F featurefile) file(s)",PROGNAME);

	/* load database */
	loadtemplates();
	if (doclear) for (i=0;i<ntemplate;i++) dtable[i].score=0;

	/* load features */
	if (dofeatures) loadfeatures(featurefile);
	
	/* process each input file in turn */
	for (;optind < argc;optind++) {
		if ((fid = sfsopen(argv[optind],"r",&rhead))<0)
			error("could not open '%s'",argv[optind]);
		if (tokname[0])
			p = tokname;
		else if (rhead.token[0])
			p = rhead.token;
		else
			p = "Unknown";
		if (!recognise(argv[optind],fid,cotype,p)) {
			/* fail */
			if (ntemplate)
				printf("%s %s\n",p,dtable[minidx].token);
			else
				printf("%s NULL\n",p);
			if (dolearn) {
				if (doforget && !isnewword(p))
					dtable[minidx].score--;
				addtemplate(rco,ritem.numframes,p);
			}
		}
		else {
			/* success */
			printf("%s %s\n",p,dtable[minidx].token);
			if (dolearn) {
				dtable[minidx].score++;
				if (dosaveall)
					addtemplate(rco,ritem.numframes,p);
			}
			ncor++;
		}
		if (doscores) {
			last = -1;
			for (i=0;(i<ntemplate) && (i<numscore);i++) {
				diff=1000;
				idx=0;
				for (j=0;j<ntemplate;j++) {
					if (dtable[j].distance > last) {
						if ((dtable[j].distance-last) < diff) {
							diff = dtable[j].distance-last;
							idx = j;
						}
					}
				}
				last = dtable[idx].distance;
				for (j=0;j<ntemplate;j++)
					if (dtable[j].distance==last)
						printf("%8.2f %s\n",dtable[j].distance,dtable[j].token);
			}
		}
		sfsclose(fid);
		ntot++;
	}

	/* save templates */
 	if (dolearn || dopurge || doclear) savetemplates(dopurge);
	if (dostatus) {
		for (i=0;i<ntemplate;i++)
			printf("%-16s%8d\n",dtable[i].token,dtable[i].score);
	}
	fprintf(stderr,"Correct %d/%d, Active Templates %d/%d.\n",
		ncor,ntot,nactive,ntemplate);
  
	/* that's all folks */
	exit(0);
}

