/* Esfilt - interactive filtering program */

/* M.A.Huckvale - University College London - November 1991 */

/* version 1.0 */
/* version 1.1 - August 1993
	- add -12 switch for 12-bit waveforms
*/

#define PROGNAME "Esfilt"
#define PROGVERS "1.1"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH ESFILT UCL1 UCL SFS
.SH NAME
Esfilt - interactive waveform filtering
.SH SYNOPSIS
.B Esfilt
(-I) (-p) (-f filtfreq) (-h|-l|-w|-n) (-t print_title) (-i item) (-12) sfsfile
.SH DESCRIPTION
.I Esfilt
is a program to demonstrate filtering of speech and other waveforms.
Four types of filter are built into the program, and can be specified by a
single frequency parameter: high-pass, low-pass, wide-band-pass (300Hz),
narrow-band-pass (45Hz).
The program displays the original waveform at the top of the screen, the
frequency response of the filter in the middle of the screen, and the output
waveform at the bottom of the screen.  A mouse click will select replay
of either waveform.  The selection of filter type and its frequency are
made using keyboard commands.  Options also allow printing the display
and saving the filtered waveform back to the SFS file.  The RMS amplitude
of the input and output waveforms are displayed in Volts, assuming an
ADC acquisition of 16-bits in the range -5 .. +5 V.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i item
Specify input Speech or Lx Waveform item.  Default: last speech item.
.TP 11
.B -p
Direct output to printer.  The program does not enter interactive mode.  The
screen image is recreated for the printer using filter specifications made
using other command-line options.
.TP 11
.BI -f filtfreq
Set up initial filter frequency.  Default 1000Hz.
.TP 11
.B -h
Specify initial filter type as High-Pass.
.TP 11
.B -l
Specify initial filter type as Low-Pass. Default.
.TP 11
.B -n
Specify initial filter type as Narrow-Band-Pass.
.TP 11
.B -w
Specify initial filter type as Wide-Band-Pass.
.TP 11
.BI -t print_title
Specify a title to appear on the printout.  If no title is given then the
date and time is printed instead.
.TP 11
.BI -12
Identify unmarked signals as 12-bits.
.SH "MAIN MENU"
.IP low-pass 11
Select low-pass filter, requests edge frequency and re-draws display.
.IP high-pass 11
Select high-pass filter, requests edge frequency and re-draws display.
.IP narrow-band 11
Select narrow-band-pass filter, requests centre frequency and re-draws display.
.IP wide-band 11
Select wide-band-pass filter, requests centre frequency and re-draws display.
.IP print 11
Prints display.  Requests a print title before printing in background.
.IP save 11
Saves currently filtered waveform to SFS file.
.IP quit 11
Exits program.
.SH "MOUSE COMANDS"
.IP middle 11
Replays waveform under mouse pointer.
.SH VERSION/AUTHOR
.IP 1.1
Mark Huckvale
*/
/*--------------------------------------------------------------------------*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <time.h>
#include <string.h>
#include <ctype.h>
#include "sfs.h"
#include "filter.h"
#include "dig.h"
#include "digdata.h"
#include "digitem.h"
#include "dgraph.h"

/* graphics configuration */
#define Y0	0.0		/* bottom of screen */
float	Y1;			/* top of timescale */
#define Y2	0.25		/* top of output waveform */
#define Y3	0.26		/* bottom of filter */
float	Y4;			/* filter title */
#define Y5	0.69		/* top of filter */
#define Y6	0.70		/* bottom of input waveform */
float	Y7;			/* bottom of timescale */
#define Y8	0.95		/* top of timescale */

#define X0	0.0		/* left hand side of screen */
#define X1	0.25		/* left hand size of filter */
float	X2;			/* filter title */
#define X3	0.75		/* right hand side of filter */
#define X4	1.0		/* right hand size of screen */
#define inarea(x,y,xl,yb,xr,yt) (((x) >= (xl)) && ((x) < (xr)) && ((y) >= (yb)) && ((y) < (yt)))

float	charwidth,charheight;		/* character width and height in screen units */

/* configuration */
#define LOW_PASS	0
#define HIGH_PASS	1
#define NARROW_BAND	2
#define WIDE_BAND	3
int	filttype=LOW_PASS;
double	filtfreq=1000.0;
char	*filtnames[4]={"low-pass","high-pass","narrow-band","wide-band"};
char	filtchar[4]={'l','h','n','w'};
char	filtitle[80];

#define MIN(x,y)	(((x)<(y))?(x):(y))
#define MAX(x,y)	(((x)>(y))?(x):(y))

/* global data */
int	laser=0;			/* laser - one shot to printer */
char	laser_title[80];		/* title for laser output */
double	getfiltfreq();
int	autoscale=1;
int	bits12=0;			/* 12-bit data default */

/* SFS data */
char	sfsfilename[SFSMAXFILENAME];		/* SFS file */
struct item_header	item;		/* input item header */
short			*sp;		/* input waveform */
struct item_header	opitem;		/* output item header */
short			*osp;		/* output waveform */

/* commands */
#define COM_NULL	0
#define COM_QUIT	1
#define COM_LOW_PASS	2
#define COM_HIGH_PASS	3
#define COM_NARROW_BAND	4
#define COM_WIDE_BAND	5
#define COM_REPLAY_IN	6
#define COM_REPLAY_OUT	7
#define COM_PRINT	8
#define COM_SAVE	9
#define COM_AUTO_SCALE	10
#define COM_FIX_SCALE	11

char *logname()
{
	static char *UNKNOWN="unknown";
	char *p,*getenv();

	if ((p=getenv("LOGNAME"))==NULL)
		return(UNKNOWN);
	else
		return(p);
}

void hardcopy()
{
	char	comstr[256];
	float	x,y;
	char	b;

	digprompt("Enter title for print: ");

	diggetmouse(laser_title,&b,&x,&y);

	if (laser_title[0]) {
		digprompt("Printing ..");
		sprintf(comstr,"Esfilt -p -i%d.%02d -f%g -%c -t \"%s\" %s &",
				item.datatype,item.subtype,
				filtfreq,
				filtchar[filttype],
				laser_title,sfsfilename);
		system(comstr);
		sleep(2);
	}
}

/* draw a waveform */
void drawwaveform(xl,xr,yb,yt,tyb,tyt,item,wave,title)
float	xl,xr;		/* x-coords of box */
float	yb,yt;		/* y-coords of waveform */
float	tyb,tyt;	/* y-coords of timescale */
struct item_header *item;
short	*wave;
char	*title;
{
	double	stime,etime;
	int	flags;
	float	sumsq;
	int	i;
	char	messg[80];
	static int	wavelo,wavehi;
	short	*ptr;
	
	digscale(1.0,1.0,0);

	stime = 0.0;
	etime = item->numframes*item->frameduration;
	
	/* display timescale */
	if (tyt > tyb)
		digitemtime(23,xl,tyb,xr,tyt,stime,etime,1);

	sumsq=0;
	for (i=0,ptr=wave;i<item->numframes;i++,ptr++) {
		if (*ptr > wavehi) wavehi = *ptr;
		if (*ptr < wavelo) wavelo = *ptr;
		sumsq += *ptr * *ptr;
	}

	sprintf(messg,"RMS Amplitude = %.3fV",sqrt(sumsq/item->numframes)/410.0);
	flags=DIGITEMBOX | DIGITEMLABEL | DIGITEMTITLE;
	digitab.title=title;
	digitab.label=messg;
	if (!autoscale) {
		flags |= DIGITEMFIX;
		digitab.lo = wavelo;
		digitab.hi = wavehi;
	}

	digitemSP(232320,xl,yb,xr,yt,item,wave,stime,etime,flags);

	if ((wavelo==0) && (wavehi==0)) {
		wavelo = digitab.lo;
		wavehi = digitab.hi;
	}

	digflush();
}

/* filter data */
#define NCOEFF 		4
#define NARROW_WIDTH	45.0
#define WIDE_WIDTH	300.0
FILTER	*flt;

/* calculate filter shape */
void calcfilter(ftype,ffreq,srate)
int	ftype;
double	ffreq;
double	srate;
{
	if (flt) filter_free(flt);
	
	switch (ftype) {
	case LOW_PASS:
		ffreq = MAX(ffreq,0.005*srate);
		ffreq = MIN(ffreq,0.495*srate);
		flt = filter_design(FILTER_LOW_PASS,2*NCOEFF,ffreq,ffreq,srate);
		sprintf(filtitle,"Low-pass at %dHz",(int)ffreq);
		break;
	case HIGH_PASS:
		ffreq = MAX(ffreq,0.005*srate);
		ffreq = MIN(ffreq,0.495*srate);
		flt = filter_design(FILTER_HIGH_PASS,2*NCOEFF,ffreq,ffreq,srate);
		sprintf(filtitle,"High-pass at %dHz",(int)ffreq);
		break;
	case NARROW_BAND:
		ffreq = MAX(ffreq,0.005*srate+NARROW_WIDTH/2);
		ffreq = MIN(ffreq,0.495*srate-NARROW_WIDTH/2);
		flt = filter_design(FILTER_BAND_PASS,NCOEFF,
			ffreq-NARROW_WIDTH/2,ffreq+NARROW_WIDTH/2,srate);
		sprintf(filtitle,"Narrow-band at %dHz",(int)ffreq);
		break;
	case WIDE_BAND:
		ffreq = MAX(ffreq,0.005*srate+WIDE_WIDTH/2);
		ffreq = MIN(ffreq,0.495*srate-WIDE_WIDTH/2);
		flt = filter_design(FILTER_BAND_PASS,NCOEFF,
			ffreq-WIDE_WIDTH/2,ffreq+WIDE_WIDTH/2,srate);
		sprintf(filtitle,"Wide-band at %dHz",(int)ffreq);
		break;
	}
	if (flt==NULL)
		error("failed to design filter");
}

/* do the filtering */
void dofilter(isp,osp,len)
short	*isp;	/* input signal */
short	*osp;	/* output signal */
int	len;	/* # samples */
{
	filter_clear(flt);
	filter_signal(flt,isp,osp,len);
}

/* draw filter axes */
void drawfilteraxes(srate,xl,yb,xr,yt)
double	srate;
float	xl,yb,xr,yt;
{
	float	x[2],y[2];
	float	xresize;
	float	yresize;

	xresize = (1.0)/(xr-xl);	/* get half srate across x-axis */
	yresize = 1.0/(yt-yb);		/* get 75dB down y-axis */
	digscale(xresize,yresize,0);
	digorigin(xl*xresize,yb*yresize);

	x[0] = 0;
	x[1] = 0.5*srate;
	y[0] = -75.0;
	y[1] = 0.0;
	dgraph((short *)x,(short *)y,2,DGfloat+DGlowax,DGfloat+DGlowax,DGaxonly,23242423,
		NULL,"Frequency (Hz)","Response (dB)");
}

#define NFILTPOINT	200

/* draw filter curve */
void drawfilter(srate,xl,yb,xr,yt)
double	srate;
float	xl,yb,xr,yt;
{
	static float	x[NFILTPOINT+1],y[NFILTPOINT+1];
	static int	once=0;
	float	xstep;
	int	i;
	float	xresize;
	float	yresize;

	xresize = (1.0)/(xr-xl);
	yresize = 1.0/(yt-yb);
	digscale(xresize,yresize,0);
	digorigin(xl*xresize,yb*yresize);
	
	if (once) {
		dgraph((short*)x,(short *)y,NFILTPOINT+1,DGfloat+DGsame,
			DGfloat+DGsame,DGline,0,NULL,NULL,NULL);
		x[0] = 0;
		x[1] = 0.5*srate;
		y[0] = -75.0;
		y[1] = 0.0;
		dgraph((short *)x,(short *)y,2,DGfloat+DGlowax,DGfloat+DGlowax,DGaxonly,23242423,
			NULL,NULL,NULL);
	}
	
	xstep = 0.5*srate/NFILTPOINT;
	for (i=0;i<=NFILTPOINT;i++) {
		x[i] = i * xstep;
		y[i] = filter_response(flt,x[i],srate);
		y[i] = 20.0*log10(MAX(y[i],1.0e-10));
	}
	dgraph((short *)x,(short *)y,NFILTPOINT+1,DGfloat+DGsame,
			DGfloat+DGsame,DGline,23,NULL,NULL,NULL);
	digflush();
	once=1;
}

/* get a filter frequency */
double getfiltfreq(prompt,max)
char	*prompt;
double	max;
{
	char	messg[80];
	float	x,y;
	char	b;
	double	freq=0,atof();

	do {
		digprompt(prompt);

		diggetmouse(messg,&b,&x,&y);

		if (*messg=='\n') {
			freq = -1;
			break;
		}
		else if (*messg)
			freq = atof(messg);

		if ((freq <= 0) || (freq >= max)) {
			putchar('\007');
			fflush(stdout);
		}
	} while ((freq <= 0) || (freq >= max));
	return(freq);
}

int stringcomp(s,t,l)
char	*s;
char	*t;
int	l;
{
	int	c=0;

	while ((l-- > 0) && ((c=(toupper(*t)-toupper(*s)))==0)) {
		s++;
		t++;
	}
	return(c);
}

/* command table */
struct command_rec {
	char	*comname;
	int	comval;
} comtab[]={
	/* main menu */
	{ "low-pass",	COM_LOW_PASS },
	{ "high-pass",	COM_HIGH_PASS },
	{ "narrow-band",	COM_NARROW_BAND },
	{ "wide-band",	COM_WIDE_BAND },
	{ "save",		COM_SAVE },
	{ "print",	COM_PRINT },
	{ "auto-scale",	COM_AUTO_SCALE },
	{ "fix-scale",	COM_FIX_SCALE },
	{ "quit",		COM_QUIT },
};
#define NUMCOMMAND (sizeof(comtab)/sizeof(struct command_rec))

int getcommand()
{
	int	i,l;
	int	com=COM_NULL;
	float	xm,ym;
	char	but;
	char	string[80];

	do {
		if (autoscale)
			digprompt("Low-pass, High-pass, Narrow-band, Wide-band, Fix-scale, Save, Print, Quit: ");
		else
			digprompt("Low-pass, High-pass, Narrow-band, Wide-band, Auto-scale, Save, Print, Quit: ");

		diggetmouse(string,&but,&xm,&ym);

		if (string[0]) {
			l = strlen(string);
			/* search commands table */
			for (i=0;i<NUMCOMMAND;i++)
				if (stringcomp(string,comtab[i].comname,l)==0) {
					com = comtab[i].comval;
					break;
				}
			if (com==COM_NULL) {
				putchar('\007');
				fflush(stdout);
			}
		}
		else if (but) {
			if (inarea(xm,ym,X0,Y0,X4,Y2)) {
				com = COM_REPLAY_OUT;
			}
			else if (inarea(xm,ym,X0,Y6,X4,Y8)) {
				com = COM_REPLAY_IN;
			}
			else {
				putchar('\007');
				fflush(stdout);
			}
		}

	} while (com==COM_NULL);

	return(com);
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */

	/* SFS file access */
	int		ipitem=SP_TYPE;
	char		*iptype="0";

	/* printing */
	char	messg[256],*logname(),*ttyname();
	time_t	tim;
	char	timbuf[32];

	/* control */
	int	com;
	int	doplay;
	
	/* decode switches */
	while ( (c = getopt(argc,argv,"Ipi:f:lhnwt:1:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Interactive filtering V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'p' :	/* one shot to printer */
			laser++;
			break;
		case 'i' :	/* input item selection */
			if (itspec(optarg,&ipitem,&iptype)==0) {
				if ((ipitem==SP_TYPE) || (ipitem==LX_TYPE))
					/* OK */;
				else
					error("input item must be waveform");
			}
			else
				error("input selection error");
			break;
		case 't' :	/* print title */
			strncpy(laser_title,optarg,80);
			break;
		case 'f' :	/* filter frequency */
			filtfreq = atof(optarg);
			if ((filtfreq <= 0) || (filtfreq > 1.0e6))
				error("bad filter frequency");
			break;
		case 'l' :	/* low-pass */
			filttype = LOW_PASS;
			break;
		case 'h' :	/* low-pass */
			filttype = HIGH_PASS;
			break;
		case 'n' :	/* low-pass */
			filttype = NARROW_BAND;
			break;
		case 'w' :	/* low-pass */
			filttype = WIDE_BAND;
			break;
		case '1' :	/* 12-bit waveform */
			if (strcmp(optarg,"2")==0)
				bits12=1;
			else
				error("unknown switch -1%s",optarg);
			break;
 		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc < 2))
		error("usage: %s (-I) (-p) (-t title) (-f filtfreq) (-l|-h|-n|-w) (-i item) (-12) sfsfile",PROGNAME);


	/* get SFS filename */
	if (optind < argc)
		strcpy(sfsfilename,sfsfile(argv[optind]));
	else
		error("no input file specified");

	/* load input speech */
	getitem(sfsfilename,ipitem,iptype,&item,(void **)&sp);

	/* generate buffer for output waveform */
	if ((osp=(short *)sfsbuffer(&item,item.numframes))==NULL)
		error("could not get waveform buffer");

	/* open graphics */
	digstart((laser)?DIG_DEFAULT_PRINTER:DIG_DEFAULT_TERM,NULL,1);
	digscale(1.0,1.0,0);
	digclearscreen();
	charwidth = digdata.chwidth/digdata.xscale;
	charheight = digdata.chheight/digdata.yscale;

	/* initialise Graphics positions */
	Y1 = Y0 + 2*charheight;		/* top of timescale */
	Y4 = Y5 - 2*charheight;		/* filter X axis */
	Y7 = Y8 - 2*charheight;		/* bottom of timescale */
	X2 = X1 + 4*charwidth;		/* filter Y axis */

	/* draw input waveform and timescale */
	drawwaveform(X0,X4,Y6,Y7,Y7,Y8,&item,sp,sfsfilename);

	/* calculate filter */
	calcfilter(filttype,filtfreq,1.0/item.frameduration);

	/* display filter axes */
	drawfilteraxes(1.0/item.frameduration,X1,Y3,X3,Y5);
	
	/* display filter */
	drawfilter(1.0/item.frameduration,X1,Y3,X3,Y5);
	
	/* filter waveform */
	dofilter(sp,osp,item.numframes);

	/* draw output waveform and timescale */
	drawwaveform(X0,X4,Y1,Y2,Y0,Y1,&item,osp,filtitle);
	
	/* if print-mode, do title and stop here */
	if (laser) {
		tim = time((time_t *)0);
		strcpy(timbuf,ctime(&tim));
		timbuf[19]='\0';
		if (*laser_title)
			sprintf(messg,"Interactive Filtering.  User: %s  Terminal: %s  Title: %s",logname(),ttyname(1),laser_title);
		else
			sprintf(messg,"Interactive Filtering.  User: %s  Terminal: %s  Date: %s",logname(),ttyname(1),timbuf);

		digtext(24,0.0,1.0-charheight,messg);
		digbox(24,0.0,0.0,1.0,Y8);
		digquit(0);
		exit(0);
	}

	/* initialise mouse */
	diginitmouse(0.5,0.5);

	/* initialise DAC */
	if (dac_open(NULL) >= 0)
		doplay=1;
	else
		doplay=0;

	/* command loop */
	while ((com=getcommand()) != COM_QUIT) switch (com) {

	case COM_LOW_PASS:
		if ((filtfreq=getfiltfreq("Low-pass edge frequency: ",0.5/item.frameduration)) < 0) break;
		filttype = LOW_PASS;
		goto dofilt;
	case COM_HIGH_PASS:
		if ((filtfreq=getfiltfreq("High-pass edge frequency: ",0.5/item.frameduration)) < 0) break;
		filttype = HIGH_PASS;
		goto dofilt;
	case COM_NARROW_BAND:
		if ((filtfreq=getfiltfreq("Narrow band-pass centre frequency: ",0.5/item.frameduration)) < 0) break;
		filttype = NARROW_BAND;
		goto dofilt;
	case COM_WIDE_BAND:
		if ((filtfreq=getfiltfreq("Wide band-pass centre frequency: ",0.5/item.frameduration)) < 0) break;
		filttype = WIDE_BAND;
		goto dofilt;
	dofilt:
		digprompt("Filtering ..");

		/* calculate filter */
		calcfilter(filttype,filtfreq,1.0/item.frameduration);

		/* display filter */
		drawfilter(1.0/item.frameduration,X1,Y3,X3,Y5);

		/* filter waveform */
		dofilter(sp,osp,item.numframes);

		/* draw output waveform without timescale */
		drawwaveform(X0,X4,Y1,Y2,Y0,Y0,&item,osp,filtitle);
	
		break;

	case COM_REPLAY_IN:	/* replay input waveform */
		digprompt("Replaying ..");
		if (doplay)
			dac_playback(sp,item.numframes,1.0/item.frameduration,
				(int)param(item.params,"bits",(bits12)?12.0:16.0),1,1);
		break;

	case COM_REPLAY_OUT:	/* replay output waveform */
		digprompt("Replaying ..");
		if (doplay)
			dac_playback(osp,item.numframes,1.0/item.frameduration,
				(int)param(item.params,"bits",(bits12)?12.0:16.0),1,1);
		break;

	case COM_PRINT:		/* do hard copy of current setting */
		hardcopy();
		break;

	case COM_SAVE:		/* save filtered waveform to file */
		digprompt("Saving ..");
		sfsheader(&opitem,item.datatype,
				item.floating,
				item.datasize,
				item.framesize,
				item.frameduration,
				item.offset,
				item.windowsize,
				item.overlap,
				item.lxsync);
		sprintf(opitem.history,"%s(%d.%02d;filttype=%s,filtfreq=%g)",
			PROGNAME,
			item.datatype,item.subtype,
			filtnames[filttype],
			filtfreq);
		putitem(sfsfilename,&opitem,item.numframes,osp);
		break;
	case COM_AUTO_SCALE:		/* autoscale filtered waveform */
		autoscale = 1;
		goto dofilt;
		break;
	case COM_FIX_SCALE:		/* fix scale on filtered waveform */
		autoscale = 0;
		goto dofilt;
		break;
	}

	if (doplay) dac_close(0);	
	digkillmouse();
	digclearscreen();
	digquit(0);
	exit(0);
}


