/* nsegment -- segment a coefficient item into N regions of stable spectrum */

/* m.a.huckvale - july 1990 */

/* version 1.0 
	- from Bridle & Sedgwick, ICASSP 1977, p656 
*/

/*--------------------------------------------------------------------------*/
/**MAN
.TH NSEGMENT SFS1 UCL
.SH NAME
nsegment - segment coefficient data into N stationary regions
.SH SYNOPSIS
.B nsegment
(-i item) (-n numseg) (-c) file
.SH DESCRIPTION
.I nsegment
uses a dynamic programming procedure to determine the location of N-1 boundaries
in time of a coefficient item, such that if each region was replaced by its mean,
the squared distortion error would be at a minimum.  The algorithm is taken from
Bridle & Sedgwick, ICASSP 1977.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version.
.TP 11
.BI -i item
Select input item numbers.
.TP 11
.BI -n numseg
Specify the number of segments to divide the item into.  Default 2.
.TP 11
.B -c
Output the result as a coefficient item.  Default output an annotation item.
.SH INPUT ITEMS
.IP 11.xx 11
Any coefficient data.
.SH OUTPUT ITEMS
.IP AN 11
N-1 boundary locations.
.IP CO 11
(optional) reconstructed segmented mean spectra.
.SH VERSION/AUTHOR
1.0 - Mark Huckvale
.SH SEE ALSO
npoint
*/
/*--------------------------------------------------------------------------*/

/* program name and version */
#define	PROGNAME "nsegment"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>		/* standard i-o routines */
#include <unistd.h>
#include <stdlib.h>
#include <math.h>
#include <malloc.h>
#include "sfs.h"		/* database filing system structures */

/* global data */
struct item_header coitem;	/* filing system item header for input */
struct co_rec		*co;	/* data table */
struct item_header opitem;	/* filing system item header for output */
struct co_rec		*corec;	/* output CO record */
struct an_rec		*anrec;	/* output AN record */

/* global data */
float	*sum,*sumsq;		/* working storage */
float	**err;			/* segment error matrix */
float	**dist;			/* cumulative distance matrix */
short	**path;			/* DP path directions matrix */
short	*junc;			/* boundary frame locations */
int	numc;			/* # coefficients */
int	numf;			/* # frames */
int	nums=2;			/* number of segments */
int	coreq=0;		/* output CO required */

/* error function */
void	errfunc(start)
int	start;
{
	int	i,j,nf;
	float	*fp;
	float	toterr;

	/* inititalise working storage with first frame */
	fp = co[start].data;
	for (i=0;i<numc;i++,fp++) {
		sum[i] = *fp;
		sumsq[i] = *fp * *fp;
	}
	for (j=0;j<=start;j++)
		err[start][j] = 0;

	/* add in rest of frames */
	for (j=start+1;j<numf;j++) {

		/* accumulate working values */
		fp = co[j].data;
		for (i=0;i<numc;i++,fp++) {
			sum[i] += *fp;
			sumsq[i] += *fp * *fp;
		}

		/* determine sum of mean squared error */
		toterr = 0;
		nf = j-start+1;
		for (i=0;i<numc;i++)
			toterr += sumsq[i] - (sum[i]*sum[i])/nf;

		err[start][j] = toterr;
	}

}

/* Dynamic Programming */
void dpmatch()
{
	int	i,j,k;
	int	minidx;
	float	val,minval;

	/* initialise zeroth segmentation */
	for (j=0;j<numf;j++)
		dist[0][j] = err[0][j];

	/* over every segment count */
	for (i=1;i<nums;i++) {

		/* over every segmentation point */
		for (j=0;j<numf;j++) {

			minidx=0;
			minval=10E10;

			/* over every earlier segmentation point */
			for (k=0;k<j;k++) {
				val = dist[i-1][k] + err[k+1][j];
				if (val < minval) {
					minidx = k+1;
					minval = val;
				}
			}

			dist[i][j] = minval;
			path[i][j] = minidx;

		}
	}
}

/* traceback segmentation path */
void trace()
{
	int	i,idx;

	i=nums-1;
	idx=numf-1;
	while (i > 0) {
		idx = path[i][idx];
		junc[i] = idx;
		i--;
	}
	junc[0] = 0;
}

/* load mean vector for range of input */
void loadmeanvec(start,stop,vec)
int	start;
int	stop;
float	*vec;
{
	int	i,j;
	float	*fp,*vp;

	memset(vec,0,numc*sizeof(float));
	for (i=start;i<stop;i++) {
		fp = co[i].data;
		vp = vec;
		for (j=0;j<numc;j++)
			*vp++ += *fp++;
	}
	for (j=0;j<numc;j++)
		vec[j] /= (stop-start);
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	char		*cotype="0";	/* input co-types */
	int		c;		/* option switch */
	int		it;		/* item/sub-type specifiers */
	char		*ty;		/* sub-type match */
	/* file variables */
	char		filename[SFSMAXFILENAME];
	int		ofid;
	int		i;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:n:c")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Segment spectral data into N steady regions V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == CO_TYPE)
					cotype = ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'n' :	/* N regions */
			nums = atoi(optarg);
			if (nums <= 1)
				error("number of sections muist be > 1");
			break;
		case 'c' :	/* CO output */
			coreq++;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	/* check for option decoding error */
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-n numseg) (-c) file",PROGNAME);

	/* get data filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* read in data set */
	getitem(filename,CO_TYPE,cotype,&coitem,(void *)&co);

	/* check can write to data file */
	if ((ofid=sfsopen(filename,"w",NULL)) < 0)
		error("access error on %s",filename);
	sfsclose(ofid);

	/* get memory for calculations */
	numc = SFSRECSIZE(&coitem);
	if ((sum=(float *)calloc(numc,sizeof(float)))==NULL)
		error("could not get memory buffer");
	if ((sumsq=(float *)calloc(numc,sizeof(float)))==NULL)
		error("could not get memory buffer");
	numf = coitem.numframes;
	if ((err=(float **)calloc(numf,sizeof(float *)))==NULL)
		error("could not get memory buffer");
	if ((dist=(float **)calloc(nums,sizeof(float *)))==NULL)
		error("could not get memory buffer");
	if ((path=(short **)calloc(nums,sizeof(short *)))==NULL)
		error("could not get memory buffer");
	for (i=0;i<numf;i++)
		if ((err[i]=(float *)calloc(numf,sizeof(float)))==NULL)
			error("could not get memory buffer");
	for (i=0;i<nums;i++)
		if (((dist[i]=(float *)calloc(numf,sizeof(float)))==NULL) ||
		    ((path[i]=(short *)calloc(numf,sizeof(short)))==NULL))
			error("could not get memory buffer");
	if ((junc=(short *)calloc(nums,sizeof(short)))==NULL)
		error("could not get memory buffer");

	/* initialise error matrix */
	for (i=0;i<(numf-1);i++) {
		/* calculate squared error for smoothing from start i */
		errfunc(i);	
		if (ttytest()) {
			printf("\rFrame %d/%d",i+1,numf-1);
			fflush(stdout);
		}
	}
	if (ttytest()) printf("\r                    \r");

	/* do DP processing */
	dpmatch();

	/* traceback to get boundaries */
	trace();	

	if (coreq) {
		/* coefficient data required */

		/* make CO header */
		sfsheader(&opitem,CO_TYPE,coitem.floating,
				coitem.datasize,coitem.framesize,
				coitem.frameduration,coitem.offset,
				0,0,1);
		sprintf(opitem.history,"%s(%d.%02d;segments=%d)",
			PROGNAME,
			coitem.datatype,coitem.subtype,
			nums);

		/* open output channel */
		if ((ofid=sfschannel(filename,&opitem)) < 0)
			error("could not open output channel to %s",filename);

		/* get output record */
		if ((corec=(struct co_rec *)sfsbuffer(&opitem,1))==NULL)
			error("could not get memory buffer");

		for (i=0;i<nums;i++) {
			corec->posn = co[junc[i]].posn;
			if (i < (nums-1))
				corec->size = co[junc[i+1]].posn - corec->posn;
			else
				corec->size = co[numf-1].posn - corec->posn;
			corec->flag = co[junc[i]].flag;
			corec->mix = co[junc[i]].mix;
			corec->gain = co[junc[i]].gain;
			if (i < (nums-1))
				loadmeanvec(junc[i],junc[i+1],corec->data);
			else
				loadmeanvec(junc[i],numf,corec->data);
			sfswrite(ofid,1,corec);
		}
	}
	else {
		/* annotation data required */

		/* make AN header */
		sfsheader(&opitem,AN_TYPE,-1,1,-1,coitem.frameduration,coitem.offset,0,0,1);
		sprintf(opitem.history,"%s(%d.%02d;segments=%d)",
			PROGNAME,
			coitem.datatype,coitem.subtype,
			nums);

		/* open output channel */
		if ((ofid=sfschannel(filename,&opitem)) < 0)
			error("could not open output channel to %s",filename);

		/* get output record */
		if ((anrec=(struct an_rec *)sfsbuffer(&opitem,1))==NULL)
			error("could not get memory buffer");

		for (i=0;i<nums;i++) {
			anrec->posn = co[junc[i]].posn;
			if (i < (nums-1))
				anrec->size = co[junc[i+1]].posn - anrec->posn;
			else
				anrec->size = co[numf-1].posn - anrec->posn;
			sprintf(anrec->label,"seg%d",i+1);
			sfswrite(ofid,1,anrec);
		}
	}

	/* close output file and update */
	if (!sfsupdate(filename))
		error("update error on %s",filename);

	/* that's all folks */
	exit(0);
}

