/* sunin -- audio input on Sun DBRI interface into SFS format */

/* Mark Huckvale - University College London */

/* version 0.1 - July 1993 */

/*-------------------------------------------------------------------------*/
/**MAN
.TH SUNIN SFS1 UCL
.SH NAME
sunin - primitive audio input on Sun audio interface
.SH SYNOPSIS
.B sunin
(-l) (-f sfreq) (-t time) (-m|-M) file
.SH DESCRIPTION
.I sunin
performs simple audio acquisition through the Sun SPARC-10 DBRI audio
interface.  This supports 16-bit acquisition only.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.B -l
Two channel acquisition into Speech and Lx.  Default Speech only.
.BI -f sfreq
Select sampling rate.  Note that the SPARC-10 audio only supports the following frequencies:
8000, 9600, 11025, 16000, 18900, 22050, 32000, 37800, 44100, and 48000Hz.  Default 16000Hz.
.TP 11
.BI -t time
Select the number of seconds to acquire.  Default waits for user to press
[RETURN] to stop.
.TP 11
.B -m
Select microphone input.  Default: line input.
.TP 11
.B -M
Don't monitor recording through Sun speaker.  Default: monitor line input through
speaker.
.SH OUTPUT ITEMS
.IP SPEECH
Speech waveform
.IP LX
(Optional) Laryngograph waveform.
.SH HISTORY
.IP sfreq
Sampling frequency
.SH FILES
.IP /dev/audio
Sun audio device
.SH VERSION/AUTHOR
.IP 0.1
Mark Huckvale
.SH SEE ALSO
inwd
.SH BUGS
*/
/*--------------------------------------------------------------------------*/

#define PROGNAME "sunin"
#define PROGVERS "0.1"
char *progname=PROGNAME;

#include <stdio.h>
#include <errno.h>
#include <malloc.h>
#include <math.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <sys/signal.h>
#include <stropts.h>
#include <sys/ioctl.h>
#include <multimedia/libaudio.h>
#include <multimedia/audio_device.h>
#include "sfs.h"

/* global defines */
#define AUDIONAME	"/dev/audio"	/* Sun SPARC audio device */
#define BUFTIME		1.0		/* length of audio buffer in seconds */
#define FOREVER		3600.0		/* forever = 1 hour ! */
#define MIN(x,y) (((x)<(y))?(x):(y))

/* global variables */
struct item_header	spitem;		/* SFS structures */
struct item_header	lxitem;
#define BUFSIZE	1024
short	sbuf[BUFSIZE];			/* output buffer for Speech */
short	lbuf[BUFSIZE];			/* output buffer for Lx */
Audio_hdr	audev;			/* audio configuration */
Audio_hdr	repdev;			/* audio configuration */
#define CBUFLEN 8192			/* copy buffer size */
char	cbuf[CBUFLEN];

/* acquisition parameters */
int	nchan=1;			/* number of channels */
int	srate=16000;			/* default sampling rate */
double	atime=FOREVER;			/* acquisition time */
short	*buf;				/* acquisition buffer */
int	buflen;				/* length of buffer in bytes */
int	doabort=0;			/* ctrl/C flag */
int	port=AUDIO_LINE_IN;		/* input port */
int	oport=AUDIO_SPEAKER;
int	monitor=1;			/* monitor input */

void	intrupt();
int	kbhit();

/* main program */
main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */

	/* file variables */
	char		filename[SFSMAXFILENAME]; /* SFS data file name */
	int		fid;		/* input file descriptor */
	int		sfid,lfid;	/* output file descriptors */
	int		afd;		/* audio file descriptor */
	int		err;		/* error code */
	struct stat	st;
	int		totlen;
	int		len,len2;
	int		smin=0x10000;	/* waveform checks */
	int		smax=-0x10000;
	int		lmin=0x10000;
	int		lmax=-0x10000;
	register int	i;
	register short	*ptr;
	int		ofd;
	int		repover=0;		/* report overload */
	/* decode switches */
	while ( (c = getopt(argc,argv,"Ilf:t:mM")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Sun SPARC-10 DBRI audio acquisition V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'l' :	/* acquire lx */
			nchan = 2;
			break;
		case 'f' :	/* set sample rate */
			srate = atoi(optarg);
			if ((srate < 8000) || (srate > 48000))
				error("sampling rate out of range");
			break;
		case 't' :	/* set acquisition time */
			atime = atof(optarg);
			if ((atime <= 0.001) || (atime >= FOREVER))
				error("invalid acquisition time");
			break;
		case 'm' :	/* input from microphone */
			port = AUDIO_MICROPHONE;
			monitor=0;
			break;
		case 'M' :	/* don't monitor */
			monitor=0;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-l) (-f sfreq) (-t time) (-m|-M) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* check file suitable */
	if ((fid=sfsopen(filename,"w",NULL)) < 0)
		error("access error on '%s'",filename);
	else
		sfsclose(fid);

	/* Validate and open the audio device */
	if (stat(AUDIONAME, &st) < 0)
		error("could not open '%s'",AUDIONAME);
	if (!S_ISCHR(st.st_mode))
		error("'%s' is not an audio device",AUDIONAME);

	if ((afd=open(AUDIONAME, O_RDONLY)) < 0) {
		if (errno == EBUSY)
			error("'%s' is busy", AUDIONAME);
		else
			error("could not open '%s'",AUDIONAME);
	}
	if (audio_get_record_config(afd, &audev) != AUDIO_SUCCESS)
		error("could not get audio configuration");

	/* set up header for acquire */
	audev.sample_rate = sunnearfreq(srate);
	audev.samples_per_unit = 1;
	audev.bytes_per_unit = 2;
	audev.channels = nchan;
	audev.encoding = AUDIO_ENCODING_LINEAR;
	audev.data_size = atime*srate*nchan;

	/* set record characteristics */
	if (audio_set_record_config(afd,&audev)!=AUDIO_SUCCESS) {
		fprintf(stderr,"couldn't set audio param (%d,%d)\n",
				audev.sample_rate,audev.channels);
		error("abandoned");
	}
	audio_set_record_port(afd,&port);

	if (monitor) {
		/* set playback characteristics */
		if ((ofd=open(AUDIONAME, O_WRONLY)) < 0) {
			if (errno == EBUSY)
				error("'%s' is busy", AUDIONAME);
			else
				error("could not open '%s'",AUDIONAME);
		}
		if (audio_get_play_config(ofd, &repdev) != AUDIO_SUCCESS)
			error("could not get audio configuration");
		repdev.sample_rate      = audev.sample_rate;
		repdev.samples_per_unit = audev.samples_per_unit;
		repdev.bytes_per_unit   = audev.bytes_per_unit;
		repdev.channels         = audev.channels;
		repdev.encoding         = audev.encoding;
		repdev.data_size        = CBUFLEN;
		if (audio_set_play_config(ofd,&repdev)!=AUDIO_SUCCESS) {
			fprintf(stderr,"couldn't set audio param (%d,%d)\n",
					repdev.sample_rate,repdev.channels);
			error("abandoned");
		}
		oport = AUDIO_SPEAKER;
		audio_set_play_port(ofd,&oport);
	}

	/* get buffer for acquisition */
	buflen = BUFTIME*srate*nchan*2;
	if ((buf=(short *)calloc(buflen,1))==NULL)
		error("could not get acquisition buffer");

	/* open output channels to SFS file */
	if (nchan==1) {
		sfsheader(&spitem,SP_TYPE,0,2,1,1.0/(double)srate,0.0,1,0,0);
		sprintf(spitem.history,"%s(sfreq=%d)",PROGNAME,srate);
		sprintf(spitem.params,"bits=16");
		if ((sfid = sfschannel(filename,&spitem)) < 0)
			error("could not open output channel to '%s'",filename);
	}
	else {
		sfsheader(&spitem,SP_TYPE,0,2,1,1.0/(double)srate,0.0,1,0,0);
		sprintf(spitem.history,"%s/SP(sfreq=%d)",PROGNAME,srate);
		sprintf(spitem.params,"bits=16");
		if ((sfid = sfschannel(filename,&spitem)) < 0)
			error("could not open output channel to '%s'",filename);
		sfsheader(&lxitem,LX_TYPE,0,2,1,1.0/(double)srate,0.0,1,0,0);
		sprintf(lxitem.history,"%s/LX(sfreq=%d)",PROGNAME,srate);
		sprintf(lxitem.params,"bits=16");
		if ((lfid = sfschannel(filename,&lxitem)) < 0)
			error("could not open output channel to '%s'",filename);
	}

	/* set up interrupt handler */
	signal(SIGINT,intrupt);

	/* wait for GO */
	printf("Hit [RETURN] to start ->");
	fflush(stdout);
	if (monitor) {
		audio_flush_record(afd);
		while (!doabort && !kbhit()) {
			len = read(afd,cbuf,CBUFLEN);
			write(ofd,cbuf,len);
		}
		if (doabort) exit(1);
	}
	else {
		while ((c=getchar()) && (c!='\n') && (c!=EOF)) /* loop */;
	}
	audio_flush_record(afd);
	if (monitor) audio_flush_play(ofd);
	err=0;
	audio_set_record_error(afd,&err);

	/* prompt them to stop */
	if (atime==FOREVER) {
		printf("Hit [RETURN] to stop  ->");
		fflush(stdout);
	}

	/* loop until time is up or ctrl/C is hit */
	totlen = atime*srate*nchan*2;
	while (!doabort && (totlen>0)) {
		len = MIN(totlen,buflen);
		len = read(afd,buf,len);
		if (monitor) write(ofd,buf,len);
		totlen -= len;
		if (nchan==1) {
			len = len/2;
			for (i=0,ptr=buf;i<len;i++,ptr++)
				if (*ptr < smin)
					smin = *ptr;
				else if (*ptr > smax)
					smax = *ptr;
			if (sfswrite(sfid,len,buf)!=len)
				error("write error on output file");
			if (((smin==-32767) || (smax==32767)) && !repover) {
				fprintf(stderr,"\nOVERLOAD\n");
				repover++;
			}
		}
		else {
			len = len/4;
			ptr=buf;
			while (len > 0) {
				len2 = MIN(len,BUFSIZE);
				for (i=0;i<len2;i++) {
					if (*ptr < smin)
						smin = *ptr;
					else if (*ptr > smax)
						smax = *ptr;
					sbuf[i] = *ptr++;
					if (*ptr < lmin)
						lmin = *ptr;
					else if (*ptr > lmax)
						lmax = *ptr;
					lbuf[i] = *ptr++;
				}
				if (sfswrite(sfid,len2,sbuf)!=len2)
					error("write error on output file");
				if (sfswrite(lfid,len2,lbuf)!=len2)
					error("write error on output file");
				len -= len2;
			}
			if (((smin==-32768) || (smax==32767) || (lmin==-32768) || (lmax==32767)) && !repover) {
				fprintf(stderr,"\nOVERLOAD\n");
				repover++;
			}
		}
		if ((atime==FOREVER) && kbhit()) break;
	}
	audio_pause_record(afd);
	printf("Done.\n");
	
	/* and update SFS file */
	if (!sfsupdate(filename))
		error("update error on '%s'",filename);

	/* Check for error during record */
	if (audio_get_record_error(afd, (unsigned *)&err) != AUDIO_SUCCESS)
		error("error reading device status");
	else if (err)
		error("WARNING: data overflow occurred");
	else if (nchan==1)
		printf("Speech: %d..%d\n",smin,smax);
	else
		printf("Speech: %d..%d, Lx: %d..%d\n",smin,smax,lmin,lmax);

	/* close down audio */
	close(afd);
	if (monitor) close(ofd);

	/* that's all folks */
	exit(0);
}

void intrupt()
{
	fprintf(stderr,"Interrupted\n");
	doabort=1;
}

/* Sun SPARC-10 dbri playback frequency table */
static int	sunfrq[]={8000,9600,11025,16000,18900,
			22050,32000,37800,44100,48000,0};

/* get nearest sample rate */
static int	sunnearfreq(srate)
int	srate;
{
	int	i;
	int	min,idx;

	idx=0;
	min=abs(sunfrq[0]-srate);
	for (i=0;sunfrq[i];i++) {
		if (abs(sunfrq[i]-srate)<min) {
			min=abs(sunfrq[i]-srate);
			idx=i;
		}
	}
	return(sunfrq[idx]);
}

/* kbhit -- test if RETURN has been pressed on keyboard input */
int	kbhit()
{
	char	c;
	
	fcntl(0,F_SETFL,O_NDELAY);

	if (read(0,&c,1)!=1) c = 0;
	
	fcntl(0,F_SETFL,0);

	return(c=='\n');
}

