/* txan -- convert tx item to annotation item */

/* M.A.Huckvale - December 1986 */

/* version 1.1 - March 1988
	- SFS version
*/
/* version 1.2 - February 1992
	- add -v voicing decision only flag
*/

#define PROGNAME "txan"
#define PROGVERS "1.2s"
char	*progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH TXAN SFS1 UCL
.SH NAME
txan -- convert tx item to annotations
.SH SYNOPSIS
.B txan
(-i item) (-v) file
.SH DESCRIPTION
.I txan
is a program to copy the excitation periods found in a tx item into
a set of annotations.  Each excitation pulse is converted to an annotation
"p" at the location given in the tx.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version number.
.TP 11
.BI -i item
Select input item number.
.TP 11
.B -v
Produce one annotation per voiced/voiceless region only.
.SH INPUT ITEMS
.IP 3.xx 11
Any excitation period data item.
.SH OUTPUT ITEMS
.IP 5 11
Excitation annotations.
.SH VERSION/AUTHOR
1.2 - Mark Huckvale
*/
/*--------------------------------------------------------------------------*/

/* standard definitions */
#include "SFSCONFG.h"
#include <stdio.h>		/* standard io library */
#include <stdlib.h>
#include <unistd.h>
#include "sfs.h"		/* database structures */

/* global data */
struct item_header txitem;	/* item header for tx data */
int		*txbuff;	/* buffer for tx data */
struct item_header anitem;	/* item header for an data */

/* voice/voiceless decision */
#define MINVOICE 0.025		/* min voicing = 40Hz */

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item selections */
	char		*ty;
	char		*txtype = "0";	/* default sub-type = last */
	/* file variables */
	char		filename[SFSMAXFILENAME];	/* dbase file name */
	int		fid;
	/* data variables */
	int		i;
	struct an_rec	*anrec;		/* annotation record */
	int		dovoice=0;
	int		voicing=0;
	int		minvoice;
	
	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:v")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Convert Tx markers to annotations V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == TX_TYPE)
					txtype = ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'v' :	/* do voicing decision only */
			dovoice++;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-v) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* load tx data */
	getitem(filename,TX_TYPE,txtype,&txitem,(void *)&txbuff);

	/* create new item header */
	sfsheader(&anitem,AN_TYPE,-1,1,-1,
		txitem.frameduration,txitem.offset,0,0,1);
	if (dovoice)
		sprintf(anitem.history,"%s(%d.%02d;type=voicing)",PROGNAME,txitem.datatype,txitem.subtype);
	else
		sprintf(anitem.history,"%s(%d.%02d;type=pulses)",PROGNAME,txitem.datatype,txitem.subtype);
	strcpy(anitem.params,"");

	/* get annotation record */
	anrec=(struct an_rec *)sfsbuffer(&anitem,1);

	/* open output channel */
	if ((fid=sfschannel(filename,&anitem)) < 0)
		error("unable to open output channel to '%s'",filename);

	/* copy out tx data as annotations */
	if (dovoice) {
		minvoice = MINVOICE/txitem.frameduration;
		anrec->posn = 0;
		if (txbuff[0] > minvoice)
			voicing = 0;
		else
			voicing = 1;
		anrec->size=txbuff[0];
		for (i=1;i<txitem.numframes;i++) {
			if (voicing && (txbuff[i] > minvoice)) {
				strcpy(anrec->label,"V+");
				sfswrite(fid,1,anrec);
				voicing = 0;
				anrec->posn += anrec->size;
				anrec->size = 0;
			}
			else if (!voicing && (txbuff[i] <= minvoice)) {
				strcpy(anrec->label,"V-");
				sfswrite(fid,1,anrec);
				voicing = 1;
				anrec->posn += anrec->size;
				anrec->size = 0;
			}
			anrec->size += txbuff[i];
		}
		if (voicing) {
			strcpy(anrec->label,"V+");
			sfswrite(fid,1,anrec);
		}
		else {
			strcpy(anrec->label,"V-");
			sfswrite(fid,1,anrec);
		}
	}
	else {
		anrec->posn=txbuff[0];
		strcpy(anrec->label,"p");
		for (i=1;i<txitem.numframes;i++) {
			anrec->size = txbuff[i];
			sfswrite(fid,1,anrec);
			anrec->posn += txbuff[i];
		}
	}

	/* update original file */
	if (!sfsupdate(filename))
		error("update error on %s",filename);

	/* that's all folks ... */
	exit(0);
}
