/* txgen -- Lx to Tx with open/close markers */

/* GEC - David Pearce and Lynne Whittaker
	- converted by Mark Huckvale - University College London
	- with automatic parameter setting by default */

/* version 3.0 - May 1996 */

#define PROGNAME "txgen"
#define PROGVERS "3.0"
char *progname=PROGNAME;

/**MAN
.TH TXGEN 1 GEC SPAR
.SH NAME
txgen - generate open and close phase markers from lx
.SH SYNOPSIS
.B txgen
(-I) (-i lx_item) (-d delay) (-h threshold) (-l threshold) file
.SH DESCRIPTION
.I txgen 
takes a laryngograph item and generates tx items for the open and close phase
markers. The processing performed is as follows :-
.PD 0
.IP 1)
Low-pass filter the Lx waveform at 3000Hz.
.IP 2)
Differentiate the Lx waveform 
.IP 3)
Find maxima and minima of differentiated signal that lie above a threshold.
These are taken as the positions of closure and opening on the lx waveform
.IP 4)
The positions of larynx opening are determined as the next point in the lx waveform
at amplitude equal to that at closure.
.IP 5)
The markers may be corrected for the acoustic delay between lx and speech signals
.PD
.TP 11
.B Options
.TP
.B -I
Identify the program version and then exit.
.IP -i lx_item
item number and subtype for lx waveform.
.IP -f freq
Select low-pass filter cut-off frequency.  Default 3000Hz.
.TP 11
.B -F
Do not filter Lx.
.IP -h threshold
Specify high threshold.  Default 0.1 * maximum Lx difference.
.IP -l threshold
Specify low threshold.  Default 0.1 * minimum Lx difference.
.IP -w peakwidth
Specify minimum peak width of differential in samples.  Default 0.0001 of sample rate.
.IP -d delay
Acoustic delay in microseconds.  Default 0us.
.SH INPUT ITEMS
.IP 2.xx 11
Any laryngograph signal
.SH OUTPUT ITEMS
.IP 3.xx 11
tx closed phase markers
.IP 3.xx 11
tx open phase markers
.PD
.SH VERSION/AUTHOR
.IP 1.0
David Pearce and Lynn Whitaker
.IP 2.0
David Pearce
.IP 3.0
Mark Huckvale
*/

/*-------------------------------------------------------------------*/

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>		/* standard i-o routines */
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
#include "sfs.h"
#include "filter.h"

/* default parameters */
#define DEFPEAKTHRESH	0.1
#define DEFVALLTHRESH	0.1
#define DEFPEAKWIDTH	0.0001

/* global data */
struct item_header lxitem;	/* filing system item header */
struct item_header cpitem;	/* filing system item header for tx closes */
struct item_header opitem;	/* filing system item header for tx opens */

/* parameters */
int		dofilt=1;	/* do filtering */
double		filtfreq=3000.0;	/* low-pass filter cut-off */
int		hupper = -1; 	/* peak height */
int		hlower = -1;	/* valley height */
int		peakwidth = -1;	/* peak width */
double		delay=0;	/* acoustic delay */
char		filename[SFSMAXFILENAME];

/* buffers */
short		*lx;		/* input lx */
short		*difflx;	/* differential lx */
int		*opbuff;	/* open markers */
int		*cpbuff;	/* close markers */

/* low-pass filter */
FILTER	*lpf;

/* allow for delay between lx and speech */
/* acoustic propogation delay from glottis to the */
/* microphone is applied to lx */
/* delay is approx 0.7 ms */
void 	correctdel(open,close,n)
int	*open;
int	*close;
int	n;
{
	int	idelay;
	int	i;

	idelay = (int)(0.5+delay/lxitem.frameduration);
	for (i=0;i<n;i++) {
		close[i] = close[i] + idelay;
		open[i] =  open[i] + idelay;
	}
}

/* differentiate lx */
void	glottal_pulse(lxdata,difflx,length)
short 	*lxdata;
short 	*difflx;
int   	length;           /* size of lx waveform */
{
	int   i;

	difflx[0] = 0;
	for (i=1;i<length;i++) {
		difflx[i] = lxdata[i] - lxdata[i-1];
	}
}

int i;
int pnum; /* peak number */
int vnum; /* valley number */
int countup;
int countdown;

/* peak detection */
int	findpeak(np,sample,item)
short 	*sample;
struct item_header *item;
int    	*np;
{
	extern int i;
	extern int pnum;
	int reference;
	extern int countdown;
	extern int countup;
	countdown=0;
	countup=0;
	reference=i;

	while (1) {
		while (sample[i+1]>=sample[i]) {
			countup++;
			if (i+1 > item->numframes)
				return(0);
			else
				i++;
		}
		while (sample[i+1]<=sample[i]) {
			countdown++;
			if (i+1 > item->numframes)
				return(0);
			else
				i++;
		}
		if ((countdown>=peakwidth) && (countup>=peakwidth) &&
		    (sample[reference + countup]>hupper) && (i>200)){
			/* found peak */
			if (np) np[pnum]=(int)(reference + countup);
			pnum=pnum+1;
			return(1);
		}
		else {
			countup = countup + countdown;
			countdown = 0;
		}
	}
	/* end of peak detection */
}

/* valley detecting routine */
int	findvalley(nv,sample,item)
short 	*sample;
struct item_header *item;
short 	*nv;
{
	extern int i;
	extern int vnum;
	int reference;
	int total;
	int mark;
	extern int countup;
	extern int countdown;
	countup=0;
	total=0;
	mark=0;
	reference=i;

	while (1) {
		while (sample[i+1]<=sample[i]) {
 			countdown++;
			if (i+1 > item->numframes)
				return(0);
			else
				i++;
		} 
		while (sample[i+1]>=sample[i]) {
  			countup++;
			if (i+1 > item->numframes)
				return(0);
			else
				i++;
		}
		if ((countdown>=peakwidth) && (countup>=peakwidth) &&
		    (sample[reference]< -(hlower))) {
			/* found valley */
			if (nv) nv[vnum]=(int)reference;
			vnum=vnum+1;
			return(1);
		}
		else {
			while(1) {
				total = countdown;
				mark = reference;
				reference = reference + countup;
				countdown = 0;
				countup=0;
				while (sample[i+1]<=sample[i]) {
					countdown++;
					if (i+1 > item->numframes)
						return(0);
					else
						i++;
				} 
				while (sample[i+1]>=sample[i]) {
  					countup++;
					if (i+1 > item->numframes)
						return(0);
					else
						i++;
				}
				if (((total + countdown)>=peakwidth) && (countup>=peakwidth) &&
				    (sample[reference + countdown]< -(hlower))) {
					/* found valley */
					if (sample[reference + countdown] > sample[mark]) {
						if (nv) nv[vnum] = (int)mark;
						vnum=vnum+1;
						return(1);
					}
					else {
						if (nv) nv[vnum]=(int)(reference + countdown);
						vnum=vnum+1;
						return(1);
					}
				}
				reference = reference + countdown;
				total = total + countdown;
			}
		}
	}
	/* end of valley detection */
}


/* peak detection */
int peak1(np,sample,item)
short 	*sample;
struct item_header *item;
int 	*np;
{
	extern int i;
	extern int pnum;
	int reference;
	extern int countdown;
	extern int countup;
	int total;
	int mark;

	mark=0;
	total=0;
	countdown=0;
	reference=i;

	while (1) {
		while (sample[i+1]>=sample[i]) {
			countup++;
			if (i+1 > item->numframes)
				return(0);
			else
				i++;
		}
		while (sample[i+1]<=sample[i]) {
			countdown++;
			if (i+1 > item->numframes)
				return(0);
			else
				i++;
		}
		if ((countdown>=peakwidth) && (countup>=peakwidth) &&
		    (sample[reference]>hupper)) {
			/* found peak */
			if (np) np[pnum]=(int)reference;
			pnum=pnum+1;
			return(1);
		}
		else {
			while (1) {
				total = countup;
				mark = reference;
				reference = reference + countdown;
				countup = 0;
				countdown=0;
				while (sample[i+1]>=sample[i]) {
					countup++;
					if (i+1 > item->numframes)
						return(0);
					else
						i++;
				} 
				while (sample[i+1]<=sample[i]) {
  					countdown++;
					if (i+1 > item->numframes)
						return(0);
					else
						i++;
				}
				if ((countdown>peakwidth) && ((countup + total)>peakwidth) &&
				    (sample[reference + countup]>hupper)) {
					/* found peak */
					if (sample[reference + countup] > sample[mark]) {
						if (np) np[pnum]=(int)(reference + countup);
						pnum = pnum + 1;
						return(1);
					}
					else {
						if (np) np[pnum]=(int)mark;
						pnum = pnum + 1;
						return(1);
					}
				}
				reference = reference + countup;
				total = total + countup; 
			}
		}
	}
	/* end of peak detection */
}

/* Diff Lx peak/valley detection routine */
int 	maxima(sample,item,np,nv)
short 	*sample; 	/* diff. Lx */
struct item_header *item;
int  	*np; 		/* position of peak */
int  	*nv; 		/* position of valley */
{
	int 	ans1;
	int 	ans2;
	int 	npoint;

	i=0;
	pnum=0;
	vnum=0;
	countup=0;
	countdown=0;
	npoint = 0;

	/* find first peak */
	if (findpeak(np,sample,item)==1) {
		npoint = npoint+1;
	}
	else return(0);

	do {
		ans1 = findvalley(nv,sample,item);
		ans2 = peak1(np,sample,item);
		if (ans2==1) npoint = npoint+1;
	} while (ans1 && ans2);

	return(npoint);
}

/* threshold lx to enhance marker accuracy */
void threshold(lxdata,close,open,npoints)
short   *lxdata;  /* lx raw data */
int     *close;   /* close phase markers */
int     *open;    /* open phase markers */
int     npoints;  /* number of markers */
{
	int   i,n;
	int   delta,olddelta;
	int   result=0;
	int   end,inc;
	int   thresh;
	int   min=10;

	for (i=0;i<npoints-1;i++) {
		n = close[i];
		end = 1;
		olddelta = min;
		thresh = (lxdata[n]+lxdata[n-1])/2;
		n = close[i] +1;
		end = 1;
		/* skip forward to end of close phase */
		while ((end == 1) && (n < close[i+1])) {
			delta = abs(lxdata[n] - lxdata[n+1]);
			if ((lxdata[n] < thresh)) {
				end = 0;
				result = n;
			}
			n = n+1;
		}
		if (n >= close[i+1]) {
			inc = (close[i+1] - close[i]);
			inc = (int)(0.3 * inc);
			open[i] = close[i] + inc;
		}
		else {
			open[i] = result;
		}
	}
	open[npoints-1] = close[npoints-1] + 30;
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern  int	optind;		/* option index */
	extern  char	*optarg;	/* option argument ptr */
	char		c;		/* option switch */
	int		errflg = 0;	/* option error flag */
	int		it;
	char		*ty;
	char		*lxtype="0";

	int		i,last,hold,ntx;
	int		min,max;
	
	/* decode switches */
	while ((c = getopt(argc,argv,"Ii:d:h:l:w:f:F")) != EOF ) switch (c) {
	case 'I' :	/* Identify */
		fprintf(stderr,"%s: Generate open and close phase markers from Lx V%s\n",PROGNAME,PROGVERS);
		exit(0);
	case 'i' :	/* specific item */
		if (itspec(optarg,&it,&ty) == 0) {
			if (it == LX_TYPE)
				lxtype=ty;
			else 
				error("unsuitable item specifier %s",optarg);
		}
		else
			error("illegal item specifier %s",optarg);
		break;
	case 'd' :      /* set acoustic delay in microseconds */
		delay = atof(optarg)*1.0E-6;
		break;
	case 'h' :
		hupper = atoi(optarg);
		break;
	case 'l' :
		hlower = atoi(optarg);
		break;
	case 'w' :
		peakwidth = atoi(optarg);
		break;
	case 'f' :
		filtfreq = atof(optarg);
		break;
	case 'F' :
		dofilt=0;
		break;
	case '?' :	/* unknown */
		errflg++;
	}
	/* check for command line errors */
	if (errflg || (argc<2))
		error("usage: %s (-I) (-F|-f filtfreq) (-d delay) (-h threshold) (-l threshold) (-w peakwidth) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,argv[optind]);
	else
		error("no database file specified",NULL);

	/* read lx item should have already been filtered */
	getitem(filename,LX_TYPE,lxtype,&lxitem,(void *)&lx);

	/* perform low-pass filter */
	if (dofilt) {
		if ((lpf=filter_design(FILTER_LOW_PASS,4,filtfreq,1.0/lxitem.frameduration,1.0/lxitem.frameduration))==NULL)
			error("failed to design low-pass filter");
		filter_signal(lpf,lx,lx,lxitem.numframes);
		filter_free(lpf);
	}

	/* get buffer for differentiated waveform */
	if ((difflx=(short *)calloc(lxitem.numframes,sizeof(short)))==NULL)
		error("could not get memory buffer");

	/* differentiate the lx waveform */
	glottal_pulse(lx,difflx,lxitem.numframes);
	for (min=0,max=0,i=0;i<lxitem.numframes;i++)
		if (difflx[i] > max) max=difflx[i];
		else if (difflx[i]<min) min=difflx[i];
	if (hupper < 0) hupper = DEFPEAKTHRESH*max;
	if (hlower < 0) hlower = -DEFVALLTHRESH*min;
	if (peakwidth < 0)
		peakwidth = (int)(0.5+DEFPEAKWIDTH*(1.0/lxitem.frameduration));

	/* count closed and open markers */
	ntx = maxima(difflx,&lxitem,NULL,NULL);

	/* get buffers for Tx values */
	if ((cpbuff=(int *)calloc(ntx+1,sizeof(int)))==NULL)
		error("could not get memory buffer");
	if ((opbuff=(int *)calloc(ntx+1,sizeof(int)))==NULL)
		error("could not get memory buffer");
	
	/* find closed and open phase markers */
	ntx = maxima(difflx,&lxitem,cpbuff,opbuff);
	
	/* find better approximation of open phase by thresholding lx */
	threshold(lx,cpbuff,opbuff,ntx);

	/* correct for acoustic delay */
	correctdel(opbuff,cpbuff,ntx);
	cpbuff[ntx] = lxitem.numframes;
	opbuff[ntx] = lxitem.numframes;

	/* change values from absolute to durational */
	last=0;
	for(i=0;i<ntx;i++) {
		hold = cpbuff[i];
		cpbuff[i] -= last;
		last = hold;
	}
	last=0;
	for(i=0;i<ntx;i++) {
		hold = opbuff[i];
		opbuff[i] -= last;
		last = hold;
	}

	/* create output headers */
	sfsheader(&cpitem,TX_TYPE,0,4,1,lxitem.frameduration,lxitem.offset,1,0,0);
	sprintf(cpitem.history,"%s(%d.%02d;close,filtfreq=%g,hthresh=%d,lthresh=%d,peakwidth=%d,delay=%g)",
		PROGNAME,lxitem.datatype,lxitem.subtype,
		(dofilt)?filtfreq:0.0,
		hupper,hlower,peakwidth,delay*1.0E6);
	putNitems(filename,&cpitem,ntx,cpbuff,2);

	/* create output headers */
	sfsheader(&opitem,TX_TYPE,0,4,1,lxitem.frameduration,lxitem.offset,1,0,0);
	sprintf(opitem.history,"%s(%d.%02d;open,filtfreq=%g;hthresh=%d,lthresh=%d,peakwidth=%d,delay=%g)",
		PROGNAME,lxitem.datatype,lxitem.subtype,
		(dofilt)?filtfreq:0.0,
		hupper,hlower,peakwidth,delay*1.0E6);
	putNitems(filename,&opitem,ntx,opbuff,2);

	if (!sfsupdate(filename))
		error("update error on '%s'",filename);

	exit(0);
}


