/* functions -- code for implementing annotation measurement in SML */

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include "sml.h"

/* protected versions of maths functions */
double Sqrt()
{
	double	d;
	d=pop();
	if ((d==ERRVAL) || (d < 0.0))
		return(ERRVAL);
	else
		return(sqrt(d));
}
double Abs()
{
	double	d,fabs();
	d=pop();
	if (d==ERRVAL)
		return(ERRVAL);
	else
		return(fabs(d));
}
double Int()
{
	double	d;
	int	i;
	d=pop();
	if (d==ERRVAL)
		return(ERRVAL);
	else {
		i=d;
		return((double)i);
	}
}
double Log()
{
	double	d;
	d=pop();
	if ((d==ERRVAL) || (d <= 0.0))
		return(ERRVAL);
	else
		return(log(d));
}
double Exp()
{
	double	d;
	d=pop();
	if ((d==ERRVAL) || (d > 1000.0))
		return(ERRVAL);
	else
		return(exp(d));
}

#ifdef DOS
long times(buff)
long	*buff;
{
	buff[0]=time((time_t *)0);
	buff[1]=buff[2]=buff[2]=0;
	return(buff[0]);
}
#endif

/* stopwatch facility */
double Stopwatch()
{
	double	d;
	static long lasttime=0;
	struct tbuffer {
		long utime;
		long stime;
		long cutime;
		long cstime;
	} buff;
	d=pop();
	if (d == 0.0) {
		lasttime = times((long *)&buff);
		return(0.0);
	}
	else
		return((times((long *)&buff) - lasttime)/60.0);
}

/* execute system call function */
double System()
{
	char	*s;
	int	ret;
	s=spop();
	if (filecount > 0) closedbfile();
	ret = system(s);
	if (filecount > 0) opendbfile(filename,0);
	Free(s);
	if (ret == 0)
		return(0.0);
	else
		return(ERRVAL);
}

/* annotation functions */

/* primitive annotation match function */
int annmatch(s,start,first,last)
char	*s;		/* specification string */
int	start;		/* annotation number to start search */
int	*first,*last;	/* returned match indexes */
{
	int	i,j,count=0,flag;
	char	t[MAXSTRING],*u,v[MAXSTRING];
	char	*re[MAXRE];

	/* append final separator to input */
	strcpy(t,s);
	strcat(t,ANNSEPS);

	/* find components and compile */
	u = strtok(t,ANNSEPS);
	while ((count<MAXRE) && (u != NULL) && (u[0] != '\0')) {
		strcpy(v,"^");
		strcat(v,u);
		re[count] = regcmp(v);
		if (re[count] == (char *) 0)
			error("illegal regular expression: %s",s);
		count++;
		u = strtok(0,ANNSEPS);
	}

	/* run through list of annotation matches to find sequence */
	*first = -1;
	*last  = -1;
	for (i=start;i<anncnt;i++) {
		if ((anntab[i].time >= timerange[tr][0]) &&
		    (anntab[i].time <  timerange[tr][1]) &&
		    (regex(re[0],anntab[i].ann) != 0)) {
			flag=1;
			for (j=1;j<count;j++)
				if ((anntab[i+j].time >= timerange[tr][1]) ||
				    (regex(re[j],anntab[i+j].ann) == 0))
					flag=0;
			if (flag) {
				*first=i;
				*last=i+count-1;
				break;
			}
		}
	}
	for (i=0;i<count;i++) Free(re[i]);
	if (*first >= 0)
		return(0);
	else
		return(1);
}

/* functions to return time of an annotation specification */
double Time()
{
	char	*s;		/* spec string */
	int	first,last;
	s=spop();
	if (s==ERRSTRING) {
		Free(s);
		return(ERRVAL);
	}
	if (annmatch(s,0,&first,&last) == 0) {
		Free(s);
		return(anntab[first].time);
	}
	Free(s);
	return(ERRVAL);
}
double Timen()
{
	char	*s;		/* spec string */
	double	n;		/* n'th match required */
	int	i,first,last,annloc=0;
	n=pop();
	s=spop();
	if ((n==ERRVAL) || (s==ERRSTRING)) {
		Free(s);
		return(ERRVAL);
	}
	/* skip over first matches */
	for (i=1;i<n;i++) {
		if (annmatch(s,annloc,&first,&last) != 0) return(ERRVAL);
		annloc = last+1;
	}
	/* find required match */
	if (annmatch(s,annloc,&first,&last) == 0) {
		Free(s);
		return(anntab[first].time);
	}
	Free(s);
	return(ERRVAL);
}

/* functions to return length of an annotation specification */
double Length()
{
	char	*s;		/* spec string */
	int	first,last;
	s=spop();
	if (s==ERRSTRING) {
		Free(s);
		return(ERRVAL);
	}
	if (annmatch(s,0,&first,&last) == 0) {
		Free(s);
		return(anntab[last].time - anntab[first].time + anntab[last].length);
	}
	Free(s);
	return(ERRVAL);
}
double Lengthn()
{
	char	*s;		/* spec string */
	double	n;		/* n'th match required */
	int	i,first,last,annloc=0;
	n=pop();
	s=spop();
	if ((n==ERRVAL) || (s==ERRSTRING)) {
		Free(s);
		return(ERRVAL);
	}
	/* skip over first matches */
	for (i=1;i<n;i++) {
		if (annmatch(s,annloc,&first,&last) != 0) return(ERRVAL);
		annloc = last+1;
	}
	/* find required match */
	if (annmatch(s,annloc,&first,&last) == 0) {
		Free(s);
		return(anntab[last].time - anntab[first].time + anntab[last].length);
	}
	Free(s);
	return(ERRVAL);
}

/* function to return text of annotation match */
char *Match()
{
	char	*s;
	int	i,first,last;
	static char matchstr[MAXSTRING];
	s=spop();
	if (s==ERRSTRING) {
		Free(s);
		return(ERRSTRING);
	}
	/* find annotation match */
	if (annmatch(s,0,&first,&last) == 0) {
		Free(s);
		strcpy(matchstr,"");
		for (i=first;i<=last;i++) {
			if ((strlen(matchstr)+strlen(anntab[i].ann)) >= MAXSTRING) {
				runtimewarning("Matched string truncated","");
				return(matchstr);
			}
			else
				strcat(matchstr,anntab[i].ann);
		}
		return(matchstr);
	}
	Free(s);
	return(ERRSTRING);
}

/* function to return text of annotation match */
char *Matchn()
{
	char	*s;
	double	n;
	int	i,first,last,annloc=0;
	static char matchstr[MAXSTRING];
	n=pop();
	s=spop();
	if ((n==ERRVAL) || (s==ERRSTRING)) {
		Free(s);
		return(ERRSTRING);
	}
	/* skip over first matches */
	for (i=1;i<n;i++) {
		if (annmatch(s,annloc,&first,&last) != 0) return(ERRSTRING);
		annloc = last+1;
	}
	/* find relevant annotation match */
	if (annmatch(s,annloc,&first,&last) == 0) {
		Free(s);
		strcpy(matchstr,"");
		for (i=first;i<=last;i++) {
			if ((strlen(matchstr)+strlen(anntab[i].ann)) >= MAXSTRING) {
				runtimewarning("Matched string truncated","");
				return(matchstr);
			}
			else
				strcat(matchstr,anntab[i].ann);
		}
		return(matchstr);
	}
	Free(s);
	return(ERRSTRING);
}

/* function to return number of matches to annotation specification */
double Numberof()
{
	char	*s;
	double	count=0.0;
	int	annloc=0;
	int	first,last;
	s=spop();
	if (s==ERRSTRING) {
		Free(s);
		return(ERRVAL);
	}
	while (annmatch(s,annloc,&first,&last) == 0) {
		annloc = last+1;
		count += 1.0;
	}
	Free(s);
	return(count);
}

/* function to return index of match into string */
extern char *__loc1;
double Index()
{
	char	*r,*s;
	int	i,j,count=0;
	char	t[MAXSTRING],*u;
	char	*re[MAXRE];
	double	posn=0.0;

	s=spop();
	r=spop();
	if ((r==ERRSTRING) || (s==ERRSTRING)) {
		Free(s);
		Free(r);
		return(ERRVAL);
	}
	/* append final separator to input */
	strcpy(t,r);
	strcat(t,ANNSEPS);
	Free(r);

	/* find components and compile */
	u = strtok(t,ANNSEPS);
	while ((count<MAXRE) && (u != NULL) && (u[0] != '\0')) {
		re[count] = (char *)regcmp(u);
		if (re[count] == (char *) 0)
			error("illegal regular expression: %s",s);
		count++;
		u = strtok(0,ANNSEPS);
	}

	/* check each regular expression in turn */
	for (i=0;i<count;i++) {
		if (regex(re[i],s) != 0) {
			posn = __loc1 - s + 1;
			for (j=0;j<count;j++) Free(re[j]);
			Free(s);
			return(posn);
		}
	}
	for (j=0;j<count;j++) Free(re[j]);
	Free(s);
	return(ERRVAL);
}

/* function to compare two strings */
double Compare()
{
	char	*s,*t;
	int	i;
	t=spop();
	s=spop();
	if ((s==ERRSTRING) || (t==ERRSTRING)) {
		Free(s);
		Free(t);
		return(ERRVAL);
	}
	i=strcmp(s,t);
	Free(s);
	Free(t);
	return((double)i);
}

/* function to return index of string in an array of strings */
double Entry()
{
	char	*s,**a;
	int	l,h;
	int	i;
	h=(int)vpop();
	l=(int)vpop();
	a=(char **)vpop();
	s=spop();
	if (s==ERRSTRING) {
		Free(s);
		return(ERRVAL);
	}
	for (i=0;i<h-l+1;i++) {
		if (s && a[i] && (strcmp(s,a[i]) == 0)) {
			Free(s);
			return((double)(i+l));
		}
	}
	Free(s);
	return(ERRVAL);
}

/* record refreshing of item */
static int stale[MAXDATATYPE];

/* select an item to measure */
double Select()
{
	double	spec;		/* item specification */
	int	it,ty;
	char	itemno[8];

	/* get item type and subtype */
	spec=pop();
	if (!itdecode(spec,&it,&ty)) return(ERRVAL);
	/* request load of that type */
	sprintf(itemno,"%d",ty);
	if (ty != 0) ittype[it]=itemno;
	if (loadit(it)) return(ERRVAL);
	stale[it]=0;
	/* return # frames */
	return((double)dbitem[it].item.numframes);
}

/* select an item to measure - next item in file */
double Selectnext()
{
	double	spec;		/* item specification */
	int	it,ty;

	/* get item type and subtype */
	spec=pop();
	if (!itdecode(spec,&it,&ty)) return(ERRVAL);
	/* load next item */
	if (loadnextit(it,ty)) return(ERRVAL);
	stale[it]=0;
	/* encode item number */
	if (!itcode(it,dbitem[it].item.subtype,&spec)) return(ERRVAL);
	return(spec);
}

/* select an item to measure by history match */
double Selectmatch()
{
	char	*s;		/* item specification */
	int32	it;
	char	*ty;
	double	spec;

	s=spop();
	if (s==ERRSTRING) {
		Free(s);
		return(ERRVAL);
	}
	if (itspec(s,&it,&ty) != 0) {
		Free(s);
		return(ERRVAL);
	}
	ittype[it]=ty;
	if (loadit(it)) {
		Free(s);
		return(ERRVAL);
	}
	stale[it]=0;
	Free(s);
	/* encode item number */
	if (!itcode(it,dbitem[it].item.subtype,&spec)) return(ERRVAL);
	return(spec);
}

/* record last used position in Next function */
static int	nextlastposn[MAXDATATYPE];
static double	nextlasttime[MAXDATATYPE];
static double	nextlastsum[MAXDATATYPE];

/* select next time interval in an item */
#define NEXTOFFSET 1.0E-6
double Next()
{
	double	spec;		/* item specification */
	double	t;		/* time */
	double	sum=0.0;
	int	i,pos=0;
	int	it;
	struct lp_rec	*lprec;
	struct co_rec	*corec;
	struct fm_rec	*fmrec;
	struct vu_rec	*vurec;

	t = pop();
	spec=pop();
	if ((spec==ERRVAL) || (t==ERRVAL)) return(ERRVAL);
	it = spec;
	if ((it < 0) || (it > MAXDATATYPE)) {
		runtimewarning("illegal item in 'next' function","");
		return(ERRVAL);
	}
	else {
		if (!dbitem[it].load) {
			if (loadit(it)) return(ERRVAL);
			stale[it]=0;
		}
	}

	/* given a time, find next record */
	if ((it==SP_TYPE) || (it==LX_TYPE) || (it==FX_TYPE) || (it==SY_TYPE) || (it==TR_TYPE)) {
		pos=(t-dbitem[it].item.offset)/dbitem[it].item.frameduration;
		pos++;
		if (pos < 0) 
			return(dbitem[it].item.offset+NEXTOFFSET);
		else if (pos >= dbitem[it].item.numframes)
			return(ERRVAL);
		else
			return(dbitem[it].item.offset+NEXTOFFSET+pos*dbitem[it].item.frameduration);
	}
	else if (it==TX_TYPE) {
		pos=(t-dbitem[TX_TYPE].item.offset)/dbitem[TX_TYPE].item.frameduration;
		if (stale[TX_TYPE] && (t==nextlasttime[TX_TYPE])) {
			i = nextlastposn[TX_TYPE];
			sum = nextlastsum[TX_TYPE];
		}
		else {
			i = 0;
			sum = 0.0;
		}
		for (;i<dbitem[TX_TYPE].item.numframes;i++) {
			sum += dbitem[TX_TYPE].u.tx[i];
			if (pos < sum) {
				nextlastposn[TX_TYPE]=i+1;
				nextlastsum[TX_TYPE]=sum;
				stale[TX_TYPE]=1;
				return(nextlasttime[TX_TYPE]=dbitem[TX_TYPE].item.offset+0.000001+sum*dbitem[TX_TYPE].item.frameduration);
			}
		}
	}
	else if (it==AN_TYPE) {
		for (i=0;i<anncnt;i++) {
			if (t < anntab[i].time) return(NEXTOFFSET+anntab[i].time);
		}
	}
	else if (it==LP_TYPE) {
		pos=(t-dbitem[it].item.offset)/dbitem[it].item.frameduration;
		lprec = dbitem[it].u.lp;
		for (i=0;i<dbitem[it].item.numframes;i++) {
			sum += lprec->length2;
			if (pos < sum) return(dbitem[it].item.offset+NEXTOFFSET+sum*dbitem[it].item.frameduration);
			lprec++;
		}
	}
	else if ((it==CO_TYPE) || (it==PC_TYPE)) {
		pos=(t-dbitem[it].item.offset)/dbitem[it].item.frameduration;
		for (i=0;i<dbitem[it].item.numframes;i++) {
			corec = &dbitem[it].u.co[i];
			if (pos < corec->posn) return(dbitem[it].item.offset+NEXTOFFSET+corec->posn*dbitem[it].item.frameduration);
		}
	}
	else if (it==FM_TYPE) {
		pos=(t-dbitem[it].item.offset)/dbitem[it].item.frameduration;
		for (i=0;i<dbitem[it].item.numframes;i++) {
			fmrec = &dbitem[it].u.fm[i];
			if (pos < fmrec->posn) return(dbitem[it].item.offset+NEXTOFFSET+fmrec->posn*dbitem[it].item.frameduration);
		}
	}
	else if (it==VU_TYPE) {
		pos=(t-dbitem[it].item.offset)/dbitem[it].item.frameduration;
		for (i=0;i<dbitem[it].item.numframes;i++) {
			vurec = &dbitem[it].u.vu[i];
			if (pos < vurec->posn) return(dbitem[it].item.offset+NEXTOFFSET+vurec->posn*dbitem[it].item.frameduration);
		}
	}
	return(ERRVAL);
}

/* get item history */
char	*History()
{
	double	spec;
	int	it;

	spec=pop();
	if (spec==ERRVAL) return(ERRSTRING);
	it = spec;
	if ((it < 1) || (it > MAXDATATYPE)) {
		runtimewarning("illegal item in history function","");
		return(ERRSTRING);
	}
	else {
		if (!dbitem[it].load && loadit(it)) return(ERRSTRING);
		return(dbitem[it].item.history);
	}
}

/* get Tx value */
double Tx()
{
	double	t;	/* time */
	int	i,pos,sum=0;
	double	txval;
	t=pop();
	if (t==ERRVAL) return(ERRVAL);
	/* load tx item */
	if (!dbitem[TX_TYPE].load && loadit(TX_TYPE)) return(ERRVAL);
	pos=(t-dbitem[TX_TYPE].item.offset)/dbitem[TX_TYPE].item.frameduration;
	for (i=0;i<dbitem[TX_TYPE].item.numframes;i++) {
		sum += dbitem[TX_TYPE].u.tx[i];
		if (pos < sum) {
			txval = dbitem[TX_TYPE].u.tx[i] * dbitem[TX_TYPE].item.frameduration;
			return(txval);
		}
	}
	return(ERRVAL);
}

/* get Fx value */
double Fx()
{
	double	t;	/* time */
	int	pos;
	double	fxval;
	t=pop();
	if (t==ERRVAL) return(ERRVAL);
	/* load fx item */
	if (!dbitem[FX_TYPE].load && loadit(FX_TYPE)) return(ERRVAL);
	pos=(t-dbitem[FX_TYPE].item.offset)/dbitem[FX_TYPE].item.frameduration;
	if ((pos < 0) || (pos >= dbitem[FX_TYPE].item.numframes))
		return(ERRVAL);
	fxval = dbitem[FX_TYPE].u.fx[pos];
	return(fxval);
}

/* get Speech value */
double Sp()
{
	double	t;	/* time */
	int	pos;
	double	spval;
	t=pop();
	if (t==ERRVAL) return(ERRVAL);
	/* load sp item */
	if (!dbitem[SP_TYPE].load && loadit(SP_TYPE)) return(ERRVAL);
	pos=(t-dbitem[SP_TYPE].item.offset)/dbitem[SP_TYPE].item.frameduration;
	if ((pos < 0) || (pos >= dbitem[SP_TYPE].item.numframes))
		return(ERRVAL);
	spval = dbitem[SP_TYPE].u.sp[pos];
	return(spval);
}

/* get Track value */
double Tr()
{
	double	t;	/* time */
	int	pos;
	double	trval;
	t=pop();
	if (t==ERRVAL) return(ERRVAL);
	/* load tr item */
	if (!dbitem[TR_TYPE].load && loadit(TR_TYPE)) return(ERRVAL);
	pos=(t-dbitem[TR_TYPE].item.offset)/dbitem[TR_TYPE].item.frameduration;
	if ((pos < 0) || (pos >= dbitem[TR_TYPE].item.numframes))
		return(ERRVAL);
	trval = dbitem[TR_TYPE].u.tr[pos];
	return(trval);
}

/* get Synth Control value */
double  Syval(p,t)
double	p;	/* parameter, 0=FX,3=F1,4=A1,5=B1,6=F2,..,18=V */
double	t;	/* time */
{
	int	pos,offset;
	double	val;
	/* load sy item */
	if (!dbitem[SY_TYPE].load && loadit(SY_TYPE)) return(ERRVAL);
	/* get frame # */
	pos=(t-dbitem[SY_TYPE].item.offset)/dbitem[SY_TYPE].item.frameduration;
	if ((pos < 0) || (pos >= dbitem[SY_TYPE].item.numframes))
		return(ERRVAL);
	/* check offset into frame */
	offset = p;
	if ((offset < 0) || (offset >= dbitem[SY_TYPE].item.framesize))
		return(ERRVAL);
	val = dbitem[SY_TYPE].u.sy[dbitem[SY_TYPE].item.framesize*pos+offset];
	if (val < 0)
		return(ERRVAL);
	else
		return(val);
}

/* SY item access */
double Sy()
{
	double	p,t;
	t=pop();
	p=pop();
	if ((t==ERRVAL) || (p==ERRVAL)) return(ERRVAL);
	return(Syval(p,t));
}

/* F1 value */
double F1()
{
	double t;
	t=pop();
	if (t==ERRVAL) return(ERRVAL);
	return(Syval(3.0,t));
}

/* A1 value */
double A1()
{
	double	t;
	t=pop();
	if (t==ERRVAL) return(ERRVAL);
	return(Syval(4.0,t) / 10.0);
}

/* B1 value */
double B1()
{
	double	t;
	t=pop();
	if (t==ERRVAL) return(ERRVAL);
	return(Syval(5.0,t));
}

/* F2 value */
double F2()
{
	double	t;
	t=pop();
	if (t==ERRVAL) return(ERRVAL);
	return(Syval(6.0,t));
}

/* A2 value */
double A2()
{
	double	t;
	t=pop();
	if (t==ERRVAL) return(ERRVAL);
	return(Syval(7.0,t) / 10.0);
}

/* B2 value */
double B2()
{
	double	t;
	t=pop();
	if (t==ERRVAL) return(ERRVAL);
	return(Syval(8.0,t));
}

/* F3 value */
double F3()
{
	double	t;
	t=pop();
	if (t==ERRVAL) return(ERRVAL);
	return(Syval(9.0,t));
}

/* A3 value */
double A3()
{
	double	t;
	t=pop();
	if (t==ERRVAL) return(ERRVAL);
	return(Syval(10.0,t) / 10.0);
}

/* B3 value */
double B3()
{
	double	t;
	t=pop();
	if (t==ERRVAL) return(ERRVAL);
	return(Syval(11.0,t));
}

/* F4 value */
double F4()
{
	double	t;
	t=pop();
	if (t==ERRVAL) return(ERRVAL);
	return(Syval(12.0,t));
}

/* A4 value */
double A4()
{
	double	t;
	t=pop();
	if (t==ERRVAL) return(ERRVAL);
	return(Syval(13.0,t) / 10.0);
}

/* B4 value */
double B4()
{
	double	t;
	t=pop();
	if (t==ERRVAL) return(ERRVAL);
	return(Syval(14.0,t));
}

/* get attribute value from LP item */
double Lp()
{
	char	*aname;
	double	t;
	int		i,j,k,a,pos;
	struct lp_rec	*rec;
	double		totdur;
	char		cname[12];

	t=pop();
	aname=spop();
	if ((t==ERRVAL) || (aname==ERRSTRING)) return(ERRVAL);

	/* check dictionary loaded */
	if (loadlpdict()) return(ERRVAL);

	/* load lp item */
	if (!dbitem[LP_TYPE].load && loadit(LP_TYPE)) return(ERRVAL);

	/* find appropriate entry */
	i=0;
	rec = dbitem[LP_TYPE].u.lp;
	pos = (t - dbitem[LP_TYPE].item.offset)/dbitem[LP_TYPE].item.frameduration;
	if (pos < 0) return(ERRVAL);
	totdur = rec->length2;
	while ((i<dbitem[LP_TYPE].item.numframes) && (pos > totdur)) {
		rec++;
		totdur += rec->length2;
		i++;
	}
	if (i >= dbitem[LP_TYPE].item.numframes) return(ERRVAL);

	/* convert attribute name to compiled form */
	for (j=0;j<12;j++) cname[j]='\0';
	i=0;
	j=0;
	while ((i<strlen(aname)) && (j<12) && (aname[i]!='['))
		cname[j++] = aname[i++];
	if (aname[i] == '[') {
		i++;
		j=0;
		while ((i<strlen(aname)) && (aname[i]!=']')) {
			for (k=11;k>j;k--) cname[k] = cname[k-1];
			cname[j++]=aname[i++];
		}
		if (aname[i] != ']') return(ERRVAL);
		for (k=11;k>j;k--) cname[k] = cname[k-1];
		cname[j] = '|';
	}
	Free(aname);

	/* find attribute index */
	a=0;
	while ((a<lpdictlen) && (strncmp(lpdict[a].aname,cname,8) != 0)) a++;
	if (a >= lpdictlen) return(ERRVAL);

	return((double) rec->alist[a] * lpdict[a].A + lpdict[a].B);
}

/* return energy from coefficients item frame */
double Energy()
{
	double		f,t;
	int		i,ncoeff;
	double		totdur;
	struct co_rec	*rec;
	double		minf,maxf,param();

	t=pop();	/* 2nd argument = time */
	f=pop();	/* 1st argument = frequency */
	if ((t==ERRVAL) || (f==ERRVAL)) return(ERRVAL);

	/* load co item */
	if (!dbitem[CO_TYPE].load && loadit(CO_TYPE)) return(ERRVAL);

	/* find appropriate entry */
	i=0;
	rec = dbitem[CO_TYPE].u.co;
	t = (t - dbitem[CO_TYPE].item.offset);
	if (t < 0) return(ERRVAL);
	totdur = (rec->posn + rec->size)*dbitem[CO_TYPE].item.frameduration;
	while ((i<dbitem[CO_TYPE].item.numframes) && (t > totdur)) {
		rec = &dbitem[CO_TYPE].u.co[i];
		totdur = (rec->posn + rec->size)*dbitem[CO_TYPE].item.frameduration;
		i++;
	}
	if (t > totdur) return(ERRVAL);

	/* find appropriate entry in vector */
	minf = param(dbitem[CO_TYPE].item.params,"minf",0.0);
	maxf = param(dbitem[CO_TYPE].item.params,"maxf",(0.5/dbitem[CO_TYPE].item.frameduration));

	ncoeff = dbitem[CO_TYPE].item.framesize-5;
	i = ncoeff*(f - minf)/(maxf-minf);
	if ((i<0) || (i>=ncoeff))
		return(ERRVAL);
	else
		return((double) rec->data[i]);

}

/* return value from FM item frame */
double	Fm()
{
	double		p,t;
	int		i,j;
	double		totdur;
	struct fm_rec	*rec;
	float		*fp;

	t=pop();	/* 2nd argument = time */
	p=pop();	/* 1st argument = parameter */
	if ((t==ERRVAL) || (p==ERRVAL)) return(ERRVAL);

	/* load fm item */
	if (!dbitem[FM_TYPE].load && loadit(FM_TYPE)) return(ERRVAL);

	/* find appropriate entry */
	i=0;
	rec = dbitem[FM_TYPE].u.fm;
	t = (t - dbitem[FM_TYPE].item.offset);
	if (t < 0) return(ERRVAL);
	totdur = (rec->posn + rec->size)*dbitem[FM_TYPE].item.frameduration;
	while ((i<dbitem[FM_TYPE].item.numframes) && (t > totdur)) {
		rec = &dbitem[FM_TYPE].u.fm[i];
		totdur = (rec->posn + rec->size)*dbitem[FM_TYPE].item.frameduration;
		i++;
	}
	if (t > totdur) return(ERRVAL);

	if ((p < 0) || (p >= (5+3*rec->npeaks))) return(ERRVAL);
	j = p;
	switch (j) {
	case 0:	return((double)rec->posn);
	case 1: return((double)rec->size);
	case 2: return((double)rec->flag);
	case 3: return((double)rec->gain);
	case 4: return((double)rec->npeaks);
	default:fp = (float *) rec->formant;
		return((double) fp[j-5]);
		break;
	}
}

/* return value from VU item frame */
double	Vu()
{
	double		p,t;
	int		i,j;
	double		totdur;
	struct vu_rec	*rec;

	t=pop();	/* 2nd argument = time */
	p=pop();	/* 1st argument = parameter */
	if ((t==ERRVAL) || (p==ERRVAL)) return(ERRVAL);

	/* load vu item */
	if (!dbitem[VU_TYPE].load && loadit(VU_TYPE)) return(ERRVAL);

	/* find appropriate entry */
	i=0;
	rec = dbitem[VU_TYPE].u.vu;
	t = (t - dbitem[VU_TYPE].item.offset);
	if (t < 0) return(ERRVAL);
	totdur = (rec->posn + rec->size)*dbitem[VU_TYPE].item.frameduration;
	while ((i<dbitem[VU_TYPE].item.numframes) && (t > totdur)) {
		rec = &dbitem[VU_TYPE].u.vu[i];
		totdur = (rec->posn + rec->size)*dbitem[VU_TYPE].item.frameduration;
		i++;
	}
	if (t > totdur) return(ERRVAL);

	if ((p < 0) || (p > 4)) return(ERRVAL);
	j = p;
	switch (j) {
	case 0:	return((double)rec->posn);
	case 1: return((double)rec->size);
	case 2: return((double)rec->flag);
	case 3: return((double)rec->mix);
	case 4: return((double)rec->gain);
	}
	return(ERRVAL);
}

/* return value from PC item frame */
double	Pc()
{
	double		p,t;
	int		i,j,ncoeff;
	double		totdur;
	struct pc_rec	*rec;

	t=pop();	/* 2nd argument = time */
	p=pop();	/* 1st argument = parameter */
	if ((t==ERRVAL) || (p==ERRVAL)) return(ERRVAL);

	/* load pc item */
	if (!dbitem[PC_TYPE].load && loadit(PC_TYPE)) return(ERRVAL);

	/* find appropriate entry */
	i=0;
	rec = dbitem[PC_TYPE].u.pc;
	t = (t - dbitem[PC_TYPE].item.offset);
	if (t < 0) return(ERRVAL);
	totdur = (rec->posn + rec->size)*dbitem[PC_TYPE].item.frameduration;
	while ((i<dbitem[PC_TYPE].item.numframes) && (t > totdur)) {
		rec = &dbitem[PC_TYPE].u.pc[i];
		totdur = (rec->posn + rec->size)*dbitem[PC_TYPE].item.frameduration;
		i++;
	}
	if (t > totdur) return(ERRVAL);

	ncoeff = dbitem[PC_TYPE].item.framesize;
	if ((p < 0) || (p >= ncoeff)) return(ERRVAL);
	j = p;
	switch (j) {
	case 0:	return((double)rec->posn);
	case 1: return((double)rec->size);
	case 2: return((double)rec->flag);
	case 3: return((double)rec->mix);
	case 4: return((double)rec->gain);
	default: return((double) rec->data[j-5]);
		break;
	}
}

/* string-number conversions */
double Val()
{
	char	*s;
	double	d,atof();
	s=spop();
	if (s==ERRSTRING) {
		Free(s);
		return(ERRVAL);
	}
	if ((s[0] < '0') || (s[0] > '9')) {
		Free(s);
		return(ERRVAL);
	}
	else {
		d=atof(s);
		Free(s);
		return(d);
	}
}
double Ascii()
{
	char	*s;
	double	d;

	s=spop();
	if (s==ERRSTRING) {
		Free(s);
		return(ERRVAL);
	}
	d = (double)*s;
	Free(s);
	return(d);
}
char  *Istr()
{
	double	v;
	int	i;
	static char	s[MAXSTRING];
	v=pop();
	if (v==ERRVAL) return(ERRSTRING);
	i = v;
	sprintf(s,"%d",i);
	return(s);
}
char  *Str()
{
	double	v,w,d;
	int	iw,id;
	static char	s[MAXSTRING];
	d=pop();
	w=pop();
	v=pop();
	if ((d==ERRVAL) || (w==ERRVAL) || (v==ERRVAL)) return(ERRSTRING);
	iw = w;
	id = d;
	sprintf(s,"%*.*f",iw,id,v);
	return(s);
}
char  *Char()
{
	double	v;
	static char	s[2];
	v=pop();
	if ((v==ERRVAL) || (v < 0) || (v > 255)) return(ERRSTRING);
	s[0] = (char)(int)v;
	s[1] = '\0';
	return(s);
}
double Strlen()
{
	char	*s;
	double	d;
	s=spop();
	if (s==ERRSTRING) {
		Free(s);
		return(ERRVAL);
	}
	d=strlen(s);
	Free(s);
	return(d);
}
	
/* file functions */
/* open a file for input */
double Openin()
{
	Symbol		*sym;
	char		*name;
	name=spop();
	sym=fpop();
	if (name==ERRSTRING) {
		Free(name);
		return(ERRVAL);
	}
	if (name[strlen(name)-1] == '|') {
		name[strlen(name)-1]='\0';
		if ((sym->u.fil=popen(name,"r")) == NULL) {
			runtimewarning("cannot open pipe for input:",name);
			sym->vtype = BADFIL;
			Free(name);
			return(ERRVAL);
		}
		else {
			sym->vtype = INPIPE;
			Free(name);
			return(0);
		}
	}
	else if ((sym->u.fil=fopen(name,"r")) == NULL) {
		runtimewarning("cannot open for input:",name);
		sym->vtype = BADFIL;
		Free(name);
		return(ERRVAL);
	}
	else {
		sym->vtype = INPFIL;
		Free(name);
		return(0);
	}
}
/* open a file for output */
double Openout()
{
	Symbol		*sym;
	char		*name;
	name=spop();
	sym=fpop();
	if (name==ERRSTRING) {
		Free(name);
		return(ERRVAL);
	}
	if (name[0] == '|') {
		strcpy(name,name+1);
		if ((sym->u.fil=popen(name,"w")) == NULL) {
			runtimewarning("cannot open pipe for output:",name);
			sym->vtype = BADFIL;
			Free(name);
			return(ERRVAL);
		}
		else {
			sym->vtype = OUTPIPE;
			Free(name);
			return(0);
		}
	}
	else if ((sym->u.fil=fopen(name,"w")) == NULL) {
		runtimewarning("cannot open for output:",name);
		sym->vtype = BADFIL;
		Free(name);
		return(ERRVAL);
	}
	else {
		sym->vtype = OUTFIL;
		Free(name);
		return(0);
	}
}
/* open a file for appending */
double Openappend()
{
	Symbol		*sym;
	char		*name;
	name=spop();
	sym=fpop();
	if (name==ERRSTRING) {
		Free(name);
		return(ERRVAL);
	}
	if ((sym->u.fil=fopen(name,"a")) == NULL) {
		runtimewarning("cannot open for appending:",name);
		sym->vtype = BADFIL;
		Free(name);
		return(ERRVAL);
	}
	else {
		sym->vtype = OUTFIL;
		Free(name);
		return(0);
	}
}
/* close a file */
double Close()
{
	Symbol		*sym;
	sym=fpop();
	if ((sym->vtype==INPIPE) || (sym->vtype==OUTPIPE)) {
		if (pclose(sym->u.fil) != 0) {
			sym->vtype = BADFIL;
			return(ERRVAL);
		}
		else {
			sym->vtype = BADFIL;
			return(0.0);
		}
	}
	else if (fclose(sym->u.fil) != 0) {
		sym->vtype = BADFIL;
		return(ERRVAL);
	}
	else {
		sym->vtype = BADFIL;
		return(0.0);
	}
}

/* statistics functions */
struct threshold {
	float	freedom;
	float	thresh;
};
struct threshold t01[19]={
{	1,	63.66 },
{	2,	9.92 },
{	3,	5.84 },
{	4,	4.60 },
{	5,	4.03 },
{	6,	3.71 },
{	7,	3.50 },
{	8,	3.36 },
{	9,	3.25 },
{	10,	3.17 },
{	12,	3.05 },
{	15,	2.95 },
{	20,	2.85 },
{	24,	2.80 },
{	30,	2.75 },
{	40,	2.70 },
{	60,	2.66 },
{	120,	2.62 },
{	1E38,	2.58 }
};
double Ttest01(s1,s2)
double s1[],s2[];
{
	double	parent,diff,t,fabs();
	int	i,f;

	if ((s1[COUNT] < 2) || (s2[COUNT] < 2)) return(ERRVAL);
	f = s1[COUNT] + s2[COUNT] - 2;
	parent = (s1[COUNT]*s1[VARIANCE]+s2[COUNT]*s2[VARIANCE])/f;
	diff = sqrt(parent * (s1[COUNT]+s2[COUNT]) / (s1[COUNT]*s2[COUNT]));
	t = (s1[MEAN]-s2[MEAN])/diff;
	i=0;
	while (f > t01[i].freedom) i++;
	if (fabs(t) > t01[i].thresh)
		return(t);
	else
		return(ERRVAL);
}
double Correl(s1,s2,p)
double s1[],s2[],p[];
{
	double	sxy,sx,sy,r,sqrt();

	if ((s1[COUNT] < 2) || (s1[COUNT] != s2[COUNT]) || (s1[COUNT] != p[COUNT])) return(ERRVAL);

	sxy = p[MEAN] - s1[MEAN]*s2[MEAN];

	sx = sqrt(s1[SUMSQ]/s1[COUNT] - (s1[SUM]*s1[SUM])/(s1[COUNT]*s1[COUNT]));
	sy = sqrt(s2[SUMSQ]/s2[COUNT] - (s2[SUM]*s2[SUM])/(s2[COUNT]*s2[COUNT]));

	r = sxy / (sx * sy);

	return(r);
}
double Anovar(s,l)
double s[];
int    l;
{
	int	i;
	double	totnum=0,totval=0;
	double	dfb,dfw,dft;
	double	sumb=0,sumw=0,sumt=0;
	double	meanb,meanw;
	double	fratio;
	double	*p;

	if (l < 2 ) return(ERRVAL);
	for (i=0;i<l;i++) {
		p = &s[6*i];
		if (p[COUNT] < 2) return(ERRVAL);
		totnum += p[COUNT];
		totval += p[SUM];
		sumb += p[SUM]*p[SUM]/p[COUNT];
		sumt += p[SUMSQ];
	}
	dfb = l-1;
	dft = totnum - 1;
	dfw = dft - dfb;
	sumb -= totval*totval/totnum;
	sumt -= totval*totval/totnum;
	sumw = sumt - sumb;
	meanb = sumb / dfb;
	meanw = sumw / dfw;
	if (meanw == 0) return(ERRVAL);
	fratio = meanb / meanw;
	return(fratio);
}

/* item numbering encoding scheme */
int itcode(it,ty,val)
int	it,ty;
double	*val;
{
	*val=ERRVAL;
	if ((it < 0) || (it > MAXDATATYPE))
		return(0);
	if ((ty < 0) || (ty > 999))
		return(0);
	if (ty < 100)
		*val = (100.0*it + ty)/100.0;
	else
		*val = (100000.0*it + ty)/100000.0;
	return(1);
}
int itdecode(val,it,ty)
double	val;
int	*it,*ty;
{
	*it=0;
	*ty=0;
	if (val==ERRVAL)
		return(0);
	if ((val < 0) || (val > MAXDATATYPE+1))
		return(0);
	*it = val;
	val = val - *it;
	if (val <= 0.00999)
		*ty = 100000.0 * val + 0.5;
	else
		*ty = 100.0 * val + 0.5;
	return(1);
}

