/* enspan -- convert SY item to a minimal C-SPAN program */

/* m.a.huckvale - November 1989 */

/* version 1.0
	- simple line fitting
*/

/*--------------------------------------------------------------------------*/
/**MAN
.TH ENSPAN SFS1 UCL 
.SH NAME
enspan - convert synthesizer control data to minimal C-SPAN program
.SH SYNOPSIS
.B enspan
(-I) (-i item) (-a) (-w) (-m matchaccuracy) file
.SH DESCRIPTION
.I enspan
converts an SY item into a text file containing a synthesizer control
data description in the format of the C-SPAN language.  See C-SPAN
manual page.  The program is produced on the standard output.
An accuracy option controls the fidelity of fit between stored data
and C-SPAN form, a value of 2% produces acceptable synthesis, while
an accuracy of less than 0.5% converts most individual parameters
at every frame.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select input item type and subtype.
.TP 11
.B -a
Use an annotation item to comment the C-SPAN source produced.
.TP 11
.B -w
Use WAIT commands in C-SPAN program instead of AT commands 
to control timing.
.TP 11
.BI -m accuracy
Specify matching accuracy with which C-SPAN program fits original.
Default 2%.
.SH INPUT ITEMS
.IP SY 11
Any SY item.
.IP AN 11
(Optional) Annotation item.
.SH VERSION/AUTHOR
.IP 1.0s 11
Mark Huckvale
.SH SEE ALSO
cspan(5), span, syout, syload, sylist
*/
/*--------------------------------------------------------------------------*/

/* program name and version */
#define	PROGNAME "enspan"
#define PROGVERS "1.0s"
char *progname=PROGNAME;

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>		/* standard i-o routines */
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <time.h>
#include "sfs.h"		/* database filing system structures */

/* global data */
struct item_header anitem;	/* annotations header */
struct item_header syitem;	/* filing system item header for sy */
float	accuracy=0.02;

/* buffers */
short		*sy;
struct an_rec	*an;
float		*fline;
float		*pline;
int		*iline;

/* parameters to reproduce */
struct parm_rec {
	int	index;		/* index into SY frame */
	char	name[4];	/* name of parameter */
	int	factor;		/* multiplier for C-SPAN values */
	int	nl;		/* newline after this parameter */
} parmtab[]={
{	0,	"fx",	1,	0 },
{	18,	"vr",	1,	1 },
{	15,	"fn",	1,	0 },
{	3,	"f1",	1,	0 },
{	6,	"f2",	1,	0 },
{	9,	"f3",	1,	0 },
{	12,	"f4",	1,	1 },
{	16,	"an",	10,	0 },
{	4,	"a1",	10,	0 },
{	7,	"a2",	10,	0 },
{	10,	"a3",	10,	0 },
{	13,	"a4",	10,	1 },
};
#define NUMPARM	(sizeof(parmtab)/sizeof(struct parm_rec))

/* copy a line of parameters out of buffer */
void parmout(sy,offset,width,length,dst)
short	*sy;
int	offset;
int	width;
int	length;
float	*dst;
{
	int	i;

	sy += offset;
	for (i=0;i<length;i++) {
		*dst++ = (float)*sy;
		sy += width;
	}
}

/* copy a line of parameters into buffer */
void parmin(sy,offset,width,length,dst)
short	*sy;
int	offset;
int	width;
int	length;
float	*dst;
{
	int	i;

	sy += offset;
	for (i=0;i<length;i++) {
		*sy = (short) *dst++;
		sy += width;
	}
}

/* do linear interpolation */
void interp(line,pstart,pstop,vstart,vstop)
float	*line;
int	pstart,pstop;
float	vstart,vstop;
{
	int	i,len;
	float	slope;

	len = pstop-pstart;
	if (len)
		slope = (vstop-vstart)/len;
	else
		return;

	for (i=0;i<=len;i++)
		line[pstart+i] = vstart + i * slope;
}

/* recursive processing of line into linear sections */
void linerecur(input,output,pstart,pstop,accuracy)
float	*input;
float	*output;
int	pstart,pstop;
float	accuracy;
{
	int	i,imax;
	float	val,maxval=0;

	/* find suitable break point */
	imax= -1;
	for (i=pstart;i<=pstop;i++) {
		val = fabs(output[i]-input[i]);
		if ((val > maxval) && ((val/((input[i]>=1.0)?input[i]:1.0)) > accuracy)) {
			imax = i;
			maxval = val;
		}
	}
	if (imax < 0)
		return;
	else {
		/* save break point */
		iline[imax]=1;
		/* do left and right portions */
		interp(output,pstart,imax,input[pstart],input[imax]);
		linerecur(input,output,pstart,imax,accuracy);
		interp(output,imax,pstop,input[imax],input[pstop]);
		linerecur(input,output,imax,pstop,accuracy);
	}
}

/* process a line of parameters into linear segments */
void lineproc(line,length,accuracy)
float	*line;
int	length;
float	accuracy;
{
	int	i;
	float	val;

	/* clear interpolation points */
	for (i=0;i<length;i++) iline[i]=0;
	iline[0]=1;
	iline[length-1]=1;

	/* initialise first guess */
	interp(pline,0,length-1,line[0],line[length-1]);

	/* recursive process of line */
	linerecur(line,pline,0,length-1,accuracy);

	/* remove last fixed portion */
	val = pline[length-1];
	i=length-1;
	while ((i>0) && (pline[i-1]==val)) {
		iline[i] = 0;
		i--;
	}

	/* copy matched fit over at interpolation points */
	for (i=0;i<length;i++)
		if (iline[i])
			line[i] = pline[i];
		else
			line[i] = -1;
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		annreq=0;	/* annotation not required */
	int		waitreq=0;	/* WAIT commands required */
	char		*antype="0";	/* default annotation item = last */
	char		*sytype ="0";	/* default SY item = last */
	int		c;		/* option switch */
	int		it;		/* item/sub-type specifiers */
	char		*ty;
	double		atof();
	/* file variables */
	char		filename[80];	/* dbase file name */
	time_t		tim;
	char		timbuf[32];
	/* processing variables */
	int		i,j,p,anidx=0;
	double		sytime;		/* time of current frame */
	double		antime=0.0;		/* time of annotation */
	short		val;
	int		atdone=0;
	int		atlast=0;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:awm:")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: C-SPAN encoding of SY data V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'a' :	/* annotation required */
			annreq++;
			break;
		case 'w' :	/* WAITs required */
			waitreq++;
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SY_TYPE)
					sytype = ty;
				else if (it == AN_TYPE) {
					antype = ty;
					annreq++;
				}
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'm' :	/* accuracy */
			accuracy = atof(optarg)/100;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	/* check for option decoding error */
	if (errflg || (argc<2))
		error("usage: %s (-I) (-a) (-w) (-m accuracy) (-i item) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no speech file specified",NULL);

	/* load the SY item */
	getitem(filename,SY_TYPE,sytype,&syitem,(void *)&sy);

	/* load annotation item if required */
	if (annreq)
		getitem(filename,AN_TYPE,antype,&anitem,(void *)&an);

	/* get temporary buffers for line */
	if ((fline=(float *)calloc(syitem.numframes,sizeof(float)))==NULL)
		error("could not get temporary memory buffer");
	if ((pline=(float *)calloc(syitem.numframes,sizeof(float)))==NULL)
		error("could not get temporary memory buffer");
	if ((iline=(int *)calloc(syitem.numframes,sizeof(int)))==NULL)
		error("could not get temporary memory buffer");

	/* loop through parameters */
	for (i=0;i<NUMPARM;i++) {
		/* get parameter index */
		p = parmtab[i].index;
		/* get copy of parameter line */
		parmout(sy,p,syitem.framesize,syitem.numframes,fline);
		/* process the line */
		lineproc(fline,syitem.numframes,accuracy);
		/* copy the result back */
		parmin(sy,p,syitem.framesize,syitem.numframes,fline);
	}

	/* now produce the C-SPAN program header */
	tim = time((time_t *)0);
	strcpy(timbuf,ctime(&tim));
	timbuf[24]='\0';
	printf("/* enspan on file '%s', item %d.%02d, accuracy %.1f%%, performed on %s */\n\n",
		filename,syitem.datatype,syitem.subtype,100.0*accuracy,timbuf);
	printf("#include \"enspan.h\"\n\n");
	printf("SYNTH()\n{\n");

	/* produce the program listing */
	if (annreq) {
		antime = anitem.offset + an[anidx].posn * anitem.frameduration;
	}
	sytime=syitem.offset;
	for (i=0;i<syitem.numframes;i++) {
		sytime += syitem.frameduration;
		while ((annreq) && (antime < sytime) && (anidx<anitem.numframes-1)) {
			printf("/* %s */\n",an[anidx].label);
			antime = anitem.offset + an[++anidx].posn * anitem.frameduration;
		}
		atdone=0;
		for (j=0;j<NUMPARM;j++) {
			val = *(sy+(i*syitem.framesize)+parmtab[j].index);
			if (val >= 0) {
				/* print leading AT or tabs */
				if (atdone==0) {
					if (waitreq) {
						if (i) 
							printf("wait(%d);\n\t",(i*10)-atlast);
						else
							printf("\t");
						atlast = 10*i;
					}
					else
						printf("at(%d);\n\t",i*10);
				}
				else if (atdone==1)
					printf("\t");

				/* print value */
				printf("%s(%d); ",parmtab[j].name,val/parmtab[j].factor);
				atdone=2;
			}
			/* check for newline */
			if ((atdone==2) && (parmtab[j].nl)) {
				printf("\n");
				atdone=1;
			}
		}
	}
	/* check for final linear section */
	if (!atdone) printf("length(%d);\n",(syitem.numframes-1)*10);

	printf("}\n");

	/* and exit */
	exit(0);
}

