/* dsp -- Digital Signal Processing Routines for SPC */

/* M.A.Huckvale - University College London */

/* version 1.0 - December 1991 */

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <malloc.h>
#include "spcc.h"

/* complex data type */
typedef struct {
	double	real,imag;
} COMPLEX;

/* basic radix-2 FFT */
/* code borrowed from Digital Signal Processing in C book */

void fft(x,m)
COMPLEX 	*x;		/* in place transform */
int		m;		/* log2 length */
{
	/* static data to hold over coefficients */
	static COMPLEX *w;           /* used to store the w complex array */
	static int mstore = 0;       /* stores m for future reference */
	static int n = 1;            /* length of fft stored for future */

	COMPLEX u,temp,tm;
	COMPLEX *xi,*xip,*xj,*wptr;

	int i,j,k,l,le,windex;

	double arg,w_real,w_imag,wrecur_real,wrecur_imag,wtemp_real;

	/* calculate coefficients if new or changed */
	if (m != mstore) {

		/* free previously allocated storage and set new m */
		if (mstore != 0) free(w);
        	mstore = m;
        	if (m == 0) return;       /* if m=0 then done */

		/* n = 2**m = fft length */
		n = 1 << m;
		le = n/2;

		/* allocate the storage for w */
		if ((w = (COMPLEX *) calloc(le-1,sizeof(COMPLEX)))==NULL)
			runerror("out of memory");

		/* calculate the w values recursively */

		arg = 4.0*atan(1.0)/le;         /* PI/le calculation */
		wrecur_real = w_real = cos(arg);
		wrecur_imag = w_imag = -sin(arg);
		xj = w;
		for (j = 1 ; j < le ; j++) {
			xj->real = (float)wrecur_real;
			xj->imag = (float)wrecur_imag;
			xj++;
			wtemp_real = wrecur_real*w_real - wrecur_imag*w_imag;
			wrecur_imag = wrecur_real*w_imag + wrecur_imag*w_real;
			wrecur_real = wtemp_real;
		}
	}

	/* start fft */
	le = n;
	windex = 1;
	for (l = 0 ; l < m ; l++) {
		le = le/2;

		/* first iteration with no multiplies */
		for (i = 0 ; i < n ; i = i + 2*le) {
			xi = x + i;
			xip = xi + le;
			temp.real = xi->real + xip->real;
			temp.imag = xi->imag + xip->imag;
			xip->real = xi->real - xip->real;
			xip->imag = xi->imag - xip->imag;
			*xi = temp;
		}

		/* remaining iterations use stored w */
		wptr = w + windex - 1;
		for (j = 1 ; j < le ; j++) {
			u = *wptr;
			for (i = j ; i < n ; i = i + 2*le) {
				xi = x + i;
				xip = xi + le;
				temp.real = xi->real + xip->real;
				temp.imag = xi->imag + xip->imag;
				tm.real = xi->real - xip->real;
				tm.imag = xi->imag - xip->imag;             
				xip->real = tm.real*u.real - tm.imag*u.imag;
				xip->imag = tm.real*u.imag + tm.imag*u.real;
				*xi = temp;
			}
			wptr = wptr + windex;
		}
		windex = 2*windex;
	}            

	/* rearrange data by bit reversing */
	j = 0;
	for (i = 1 ; i < (n-1) ; i++) {
		k = n/2;
		while (k <= j) {
			j = j - k;
			k = k/2;
		}
		j = j + k;
		if (i < j) {
			xi = x + i;
			xj = x + j;
			temp = *xj;
			*xj = *xi;
			*xi = temp;
		}
	}
}

/* rfftmag - trig recombination real input FFT to magnitude */
void rfftmag_(x,y,m)
double	*x;		/* input real signal */
double	*y;		/* output squared magnitude */
int32	m;		/* log2(numsample) */
{
	static COMPLEX  *cf;
	static int      mstore = 0;
	int      	p,num,k;
	float     	Realsum, Realdif, Imagsum, Imagdif;
	double    	factor, arg;
	COMPLEX   	*ck, *xk, *xnk, *cx;
	COMPLEX		hold;

	/* First call the fft routine using the x array but with
   		half the size of the real fft */
	p = m - 1;
	cx = (COMPLEX *) x;
	fft(cx,p);

	/* Next create the coefficients for recombination, if required */

	num = 1 << p;    /* num is half the real sequence length.  */

	if (m != mstore) {
		if (mstore != 0) free(cf);
		cf = (COMPLEX *) calloc(num - 1,sizeof(COMPLEX));
		if (!cf)
			runerror("out of memory");

		factor = 4.0*atan(1.0)/num;
		for (k = 1; k < num; k++) {
			arg = factor*k;
			cf[k-1].real = (float)cos(arg);
			cf[k-1].imag = (float)sin(arg);
		}
	}  

	/* DC component, no multiplies */
    	y[0] = (cx[0].real + cx[0].imag) * (cx[0].real + cx[0].imag);

	/* other frequencies by trig recombination */
	ck = cf;
	xk = cx + 1;
	xnk = cx + num - 1;
	for (k = 1; k < num; k++) {
		Realsum = ( xk->real + xnk->real ) / 2;
		Imagsum = ( xk->imag + xnk->imag ) / 2;
		Realdif = ( xk->real - xnk->real ) / 2;
		Imagdif = ( xk->imag - xnk->imag ) / 2;

		hold.real = Realsum + ck->real * Imagsum
        	                  - ck->imag * Realdif;

		hold.imag = Imagdif - ck->imag * Imagsum
	                          - ck->real * Realdif;

		y[k] = sqrt((hold.real*hold.real)+(hold.imag*hold.imag));
		ck++;
		xk++;
		xnk--;
	}
}


