/* pasfunc -- emulation of Pascal functions for SPC */

/* Mark Huckvale - University College London - April 1991 */

/* external definitions */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#include "spc.h"
#include "spcc.h"

/*---------------------------- Constants -----------------------------*/

string stringNULL={0L,{NULL}};
waveform waveformNULL={0L,0.0,{NULL}};
static double realforml[20]={
	1.0E0,	1.0E1,	1.0E2,	1.0E3,	1.0E4,
	1.0E5,	1.0E6,	1.0E7,	1.0E8,	1.0E9,
	1.0E10,	1.0E11,	1.0E12,	1.0E13,	1.0E14,
	1.0E15,	1.0E16,	1.0E17,	1.0E18,	1.0E19,
};
static double realforms[20]={
	1.0E0,
	1.0E0,	1.0E-1,	1.0E-2,	1.0E-3,	1.0E-4,
	1.0E-5,	1.0E-6,	1.0E-7,	1.0E-8,	1.0E-9,
	1.0E-10,	1.0E-11,	1.0E-12,	1.0E-13,	1.0E-14,
	1.0E-15,	1.0E-16,	1.0E-17,	1.0E-18,
};

/*-------------------------- Memory Functions --------------------------*/
void _new(vptr,siz)
char	**vptr;
int	siz;
{
	if ((*vptr=calloc(siz,1))==NULL)
		runerror("out of memory");
}

void _dispose(vptr)
char	*vptr;
{
	free(vptr);
}

/*---------------------- Mathematical Functions ------------------------*/
int	_iabs(l)
int	l;
{
	return((l > 0) ? l : -l);
}
double	_rabs(r)
double	r;
{
	return((r > 0) ? r : -r);
}
int	_isqr(l)
int	l;
{
	return(l*l);
}
double	_rsqr(r)
double	r;
{
	return(r*r);
}

/*---------------------- Text I/O Functions ---------------------------*/

/* read a variable from standard input */
char	temp[1024];

void _READ(ip,type,addr)
filedesc ip;
int	type;
char	*addr;
{
	char	bc;
	string	*str;

	if (type==(int)EXPINT)
		fscanf(ip,"%d",(int *)addr);
	else if (type==(int)EXPREAL)
		fscanf(ip,"%lf",(double *)addr);
	else if (type==(int)EXPBOOL) {
		fscanf(ip,"%c",&bc);
		if (bc=='t')
			*addr = 1;
		else if (bc=='f')
			*addr = 0;
		else
			runerror("non-Boolean input: '%c'",bc);
	}
	else if (type==(int)EXPCHAR)
		fscanf(ip,"%c",addr);
	else if (type==(int)EXPSTR) {
		fscanf(ip,"%s",temp);
		str = (string *)addr;
		_sfree(*str);
		str->len = strlen(temp);
		if ((str->buf.ar=(char *)malloc((unsigned)str->len+1))==NULL)
			runerror("out of memory");
		strcpy(str->buf.ar,temp);
	}
	else
		runerror("Illegal type for read");
}

void _readln(ip)
filedesc ip;
{
	int	c;
	while (((c=getc(ip))!=EOF) && (c!='\n')) /* loop */;
}

/* test for end of file on input */
int _eof(ip)
filedesc ip;
{
	int	c;
	if ((c=getc(ip))==EOF) {
		return(1);
	}
	else {
		ungetc(c,ip);
		return(0);
	}
}
	
/* test for end of line on input */
int _eoln(ip)
filedesc ip;
{
	int	c;
	if ((c=getc(ip))=='\n') {
		ungetc(c,ip);
		return(1);
	}
	else if (c != EOF) {
		ungetc(c,ip);
		return(0);
	}
	else
		return(1);
}
	
#ifdef __STDC__
void _WRITE(op,type)
filedesc op;
int	type;
{
	va_list	arg_ptr;
	int	l;
	double	d;
	string	s;
	int	i,width,precision;

	va_start(arg_ptr,type);

	if (type==(int)EXPINT) {
		l = va_arg(arg_ptr,int);
		width = (int)va_arg(arg_ptr,int);
		fprintf(op,"%*d",width,l);
	}
	else if (type==(int)EXPREAL) {
		d = va_arg(arg_ptr,double);
		width = (int)va_arg(arg_ptr,int);
		precision = (int)va_arg(arg_ptr,int);
		i = width - precision - 1;
		if (d==0.0)
			fprintf(op,"%*.*f",width,precision,d);
		else if ((fabs(d) >= 1.0E20) || (fabs(d) < 1.0E-19))
			fprintf(op,"%*.*e",width,precision,d);
		else if ((i >= 0) && (i < 20) &&
			((fabs(d) >= realforml[i]) || (fabs(d) < realforms[i])))
			fprintf(op,"%*.*e",width,precision,d);
		else
			fprintf(op,"%*.*f",width,precision,d);
	}
	else if (type==(int)EXPBOOL) {
		i = va_arg(arg_ptr,int);
		width = (int)va_arg(arg_ptr,int);
		fprintf(op,"%*c",width,(i)?'t':'f');
	}
	else if (type==(int)EXPCHAR) {
		l = va_arg(arg_ptr,int);
		width = (int)va_arg(arg_ptr,int);
		if (width==0)
			fprintf(op,"%c",l);
		else
			fprintf(op,"%*c",width,l);
	}
	else if (type==(int)EXPSTR) {
		s = va_arg(arg_ptr,string);
		width = (int)va_arg(arg_ptr,int);
		if (width==0)
			fprintf(op,"%s",s.buf.ar);
		else
			fprintf(op,"%*s",width,s.buf.ar);
		_sfree(s);
	}
	else
		runerror("Illegal type for write");
}
#else
_WRITE(va_alist)
va_dcl
{
	filedesc op;
	int	type;
	va_list	arg_ptr;
	int	l;
	double	d;
	string	s;
	int	i,width,precision;

	va_start(arg_ptr);
	op = va_arg(arg_ptr,filedesc);
	type = va_arg(arg_ptr,int);

	if (type==(int)EXPINT) {
		l = va_arg(arg_ptr,int);
		width = va_arg(arg_ptr,int);
		fprintf(op,"%*d",width,l);
	}
	else if (type==(int)EXPREAL) {
		d = va_arg(arg_ptr,double);
		width = va_arg(arg_ptr,int);
		precision = va_arg(arg_ptr,int);
		fprintf(op,"%*.*f",width,precision,d);
	}
	else if (type==(int)EXPBOOL) {
		i = va_arg(arg_ptr,int);
		fprintf(op,"%c",(i)?'t':'f');
	}
	else if (type==(int)EXPCHAR) {
		l = va_arg(arg_ptr,int);
		width = va_arg(arg_ptr,int);
		fprintf(op,"%*c",width,l);
	}
	else if (type==(int)EXPSTR) {
		s = va_arg(arg_ptr,string);
		width = va_arg(arg_ptr,int);
		if (width==0)
			fprintf(op,"%s",s.buf.ar);
		else
			fprintf(op,"%*s",width,s.buf.ar);
		_sfree(s);
	}
	else
		runerror("Illegal type for write");
}
#endif

void _writeln(op)
filedesc op;
{
	putc('\n',op);
}

void _fileclose(op)
filedesc op;
{
	fclose(op);
}

static int ioerrortrapping;
static int ioerrorno;

void _ioerrortrap(flag)
int flag;
{
	if (flag)
		ioerrortrapping=1;
	else
		ioerrortrapping=0;
}

filedesc _reset(fname)
string	fname;
{
	filedesc ip;
	
	ioerrorno=0;
	if ((ip=fopen(fname.buf.ar,"r"))==NULL) {
		if (ioerrortrapping)
			ioerrorno = errno;
		else
			runerror("unable to open '%s'",fname);
	}
	_sfree(fname);
	return(ip);
}

filedesc _rewrite(fname)
string fname;
{
	filedesc ip;

	ioerrorno=0;
	if ((ip=fopen(fname.buf.ar,"w"))==NULL) {
		if (ioerrortrapping)
			ioerrorno = errno;
		else
			runerror("unable to open '%s'",fname);
	}
	_sfree(fname);
	return(ip);
}

int _ioerror(void)
{
	return(ioerrorno);
}
			
/*---------------------- Run-time Errors ----------------------------------*/

#ifdef __STDC__
void runerror(fmt)
char *fmt;
{
	va_list		arg_ptr;
	extern char *	progname;
	
	/* print message */
	fprintf(stderr,"%s: ",progname);
	va_start(arg_ptr,fmt);
	vfprintf(stderr,fmt,arg_ptr);
	va_end(arg_ptr);
	fprintf(stderr,"\n");
	exit(1);
}
#else
runerror(va_alist)
va_dcl
{
	char *fmt;
	va_list		arg_ptr;
	extern char *	progname;
	
	/* print message */
	fprintf(stderr,"%s: ",progname);
	va_start(arg_ptr);
	fmt = va_arg(arg_ptr,char *);
	vfprintf(stderr,fmt,arg_ptr);
	va_end(arg_ptr);
	fprintf(stderr,"\n");
	exit(1);
}
#endif

/*------------------- Memory Allocation --------------------------*/

char * strmalloc(size)
int size;
{
	char *str;
	if ((str=(char *)malloc((int)size))==NULL)
		runerror("insufficient memory for string");
	return(str);
}
#define strfree(p) free(p)
short * wavmalloc(size)
int size;
{
	short *wav;
	if ((wav=(short *)malloc((int)size*sizeof(short)))==NULL)
		runerror("insufficient memory for waveform");
	return(wav);
}
#define wavfree(p) free(p)
void wavcopy(dst,src,len)
short *dst;
short *src;
int	len;
{
	memcpy((char *)dst,(char *)src,(int)len*sizeof(short));
}

/*------------------- Dynamic Data Evaluation ---------------------*/

/* string dynamic data evaluation stack */
#define STACKSIZE 32
string	stack[STACKSIZE];
int	sp=0;

void _spush(str)
string	str;
{
	stack[sp].len=str.len;
	stack[sp].buf.ar=strmalloc(stack[sp].len+1);
	strcpy(stack[sp].buf.ar,str.buf.ar);
	sp++;
}

void _spushstr(str)
char	*str;
{
	stack[sp].len=strlen(str);
	stack[sp].buf.ar=strmalloc(stack[sp].len+1);
	strcpy(stack[sp].buf.ar,str);
	sp++;
}

void _spushchar(ch)
char	ch;
{
	stack[sp].len=1;
	stack[sp].buf.ar=strmalloc(stack[sp].len+1);
	stack[sp].buf.ar[0]=ch;
	stack[sp].buf.ar[1]='\0';
	sp++;
}

void _ssave(str)
string	str;
{
	stack[sp++]=str;
}

string	_spop(void)
{
	return(stack[--sp]);
}

void _sassign(v,str)
string	*v;
string	str;
{
	_sfree(*v);
	*v = str;
}

void _cassign(v,str)
char	*v;
string	str;
{
	*v = str.buf.ar[0];
	_sfree(str);
}

int _sord(str)
string	str;
{
	int	ch;
	ch = str.buf.ar[0];
	_sfree(str);
	return(ch);
}

void _sarith(code)
int	code;
{
	string	s1,s2;

	switch (code) {
	case '+': 	/* concatenate strings */
		s2 = _spop();
		s1 = _spop();
		stack[sp].len = s1.len + s2.len;
		stack[sp].buf.ar=strmalloc(stack[sp].len+1);
		strcpy(stack[sp].buf.ar,s1.buf.ar);
		strcpy(stack[sp].buf.ar+s1.len,s2.buf.ar);
		sp++;
		_sfree(s2);
		_sfree(s1);
		break;
	default:
		runerror("compiler error: unknown operator _sarith(%d)",code);
	}
}


int _scompare(rel)
int	rel;
{
	int	res=0;
	string	s1,s2;

	s2 = _spop();
	s1 = _spop();

	switch (rel) {
	case '=':
		res=(strcmp(s1.buf.ar,s2.buf.ar)==0)?TRUE:FALSE;
		break;
	case NE_TOK:
		res=(strcmp(s1.buf.ar,s2.buf.ar)==0)?FALSE:TRUE;
		break;
	case LT_TOK:
		res=(strcmp(s1.buf.ar,s2.buf.ar)<0)?TRUE:FALSE;
		break;
	case LE_TOK:
		res=(strcmp(s1.buf.ar,s2.buf.ar)<=0)?TRUE:FALSE;
		break;
	case GT_TOK:
		res=(strcmp(s1.buf.ar,s2.buf.ar)>0)?TRUE:FALSE;
		break;
	case GE_TOK:
		res=(strcmp(s1.buf.ar,s2.buf.ar)>=0)?TRUE:FALSE;
		break;
	default:
		runerror("compiler error: illegal string operator _scompare(%d)",rel);
	}
	_sfree(s1);
	_sfree(s2);
	return(res);
}

void _sfree(str)
string	str;
{
	if (str.buf.ar) strfree(str.buf.ar);
}

/* waveform dynamic data evaluation stack */
waveform wstack[STACKSIZE];
int	wsp=0;

void _wpush(wav)
waveform wav;
{
	wstack[wsp].len=wav.len;
	wstack[wsp].smr=wav.smr;
	wstack[wsp].buf.ar=wavmalloc(wstack[wsp].len);
	wavcopy(wstack[wsp].buf.ar,wav.buf.ar,wav.len);
	wsp++;
}

void _wsave(wav)
waveform wav;
{
	wstack[wsp++]=wav;
}

waveform _wpop(void)
{
	return(wstack[--wsp]);
}

void _wassign(v,wav)
waveform	*v;
waveform	wav;
{
	_wfree(*v);
	*v = wav;
}

void _warith(code)
int	code;
{
	waveform w1,w2;

	switch (code) {
	case '+': 	/* concatenate waveforms */
		w2 = _wpop();
		w1 = _wpop();
		wstack[wsp].len = w1.len + w2.len;
		if (w1.smr != w2.smr)
			runerror("incompatible waveforms");
		wstack[wsp].smr = w1.smr;
		wstack[wsp].buf.ar=wavmalloc(wstack[wsp].len);
		wavcopy(wstack[wsp].buf.ar,w1.buf.ar,w1.len);
		wavcopy(wstack[wsp].buf.ar+w1.len,w2.buf.ar,w2.len);
		wsp++;
		_wfree(w2);
		_wfree(w1);
		break;
	default:
		runerror("compiler error: unknown operator _warith(%d)",code);
	}
}

void _wfree(wav)
waveform	wav;
{
	if (wav.buf.ar) wavfree(wav.buf.ar);
}

/*--------------------- Program Arguments ------------------------*/

int numarg_()
{
	return((int)gargc-1);
}

string getarg_(n)
int	n;
{
	string	s;
	s.len = strlen(gargv[n]);
	s.buf.ar = strmalloc(s.len+1);
	strcpy(s.buf.ar,gargv[n]);
	return(s);
}
	
