/* divide -- divide up a raw binary file using prompt file */

/* M.A.Huckvale - University College London */

/* version 1.0 - May 1994 */

#define PROGNAME "divide"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH DIVIDE SFS1 UCL
.SH NAME
divide -- divide up binary file using recording prompts file
.SH SYNOPSIS
.B divide
(-l) (-c nchan) (-S spchan) (-L lxchan) (-P promptchan) (-T threshadjust)
(-p promptsfile|-n numwords) (-a) -f samprate (-b) (-h headerlen) binfile headfile
.SH DESCRIPTION
.I divide
is a program to break up a binary file into separate SFS files.  It is
able to cope with multiple channel data and can use prompt file listings
from the 'display(?)' program.  This program produces a series of clicks
which can be recorded on one of the input channels and can be used
to synchronise division of the file.  Otherwise an energy criterion is
used to find the appropriate sections.  Energy criteria are only applied
to channel 0, on the assumption that this is the speech signal.  Waveform
outputs are sent to chan0:speech, chan1:lx, chan2:speech, chan3:lx, etc. unless
over-rided by the -S and -L switches.
The SFS main header information is taken from the supplied file, which is
otherwise left unchanged.  The token field of each created file is generated
from the first word of the lines in the prompts file.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.B -l
Link output SFS files to binary file, rather than copy.
.TP 11
.BI -c nchan
Specify number of channels.  Default 1.
.TP 11
.BI -p promptsfile
Specify text file containing prompts, one per line.  A line ending in '#repeat'
is taken to be a corrected version of the previous prompt.  The output filename
is constructed from the first word on the line, with a numerical extension if
the name already exists.
.TP 11
.BI -n numwords
Look for this number of items in file.  Each is annotated as 'word1',
'word2', etc.
.TP 11
.B -a
Just link and create annotations directly in the SFS file supplied.
.TP 11
.BI -S spchan
Specify a channel containing a speech signal [1..N].  The first speech channel
specified is also used for energy determination.
.TP 11
.BI -L lxchan
Specify the channel containing a laryngographic signal [1..N].
.TP 11
.BI -P promptchan
Specify the channel containing the prompting clicks if present [1..N].  This channel
is not linked to the final file.
.TP 11
.BI -T threshadjust
Allows user to tinker with threshold for word detection.  In db, positive
value makes less sensitive, negative values more sensitive.  Default 0db.
.TP 11
.BI -f samprate
Specify the sampling rate of each channel in the binary file.  Default 20kHz.
.TP 11
.B -b
Swap bytes in binary file.
.TP 11
.BI -h headerlen
Specify length of any header in the binary file to skip in bytes.  Default 0.
.SH OUTPUT ITEMS
.IP SP
Even channels
.IP LX
Odd channels
.SH HISTORY
.IP file
input file name
.IP channel
input channel number
.IP start
start offset (samples)
.IP end
end offset (samples)
.SH VERSION/AUTHOR
.IP 1.0
Mark Huckvale
.SH SEE ALSO
npoint
.SH BUGS
*/
/*--------------------------------------------------------------------------*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <math.h>
#include <malloc.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "sfs.h"

/* manifest parameters */
#define MAXWORDS	1000
#define MAXCHANNELS	8
#define MPXBUFSIZE	32768
#define WINSIZE		0.1
#define ABS(x)	(((x)>0)?(x):(-x))

/* global data */
struct main_header	head;
struct item_header	item;

/* buffering */
short			*msp;
int			mpxbufsize;
short			*chan[MAXCHANNELS];
int			chanbufsize;
float			*ewin;
int			totwin;
int			totsamp;

/* prototype word positions */
struct wordpos_rec {
	int	posn;
	int	size;
	char	*label;
	float	wenergy;
	float	genergy;
} wordtab[2*MAXWORDS];
int	totword=0;

/* prompt file */
char	*ptable[MAXWORDS];
char	*ttable[MAXWORDS];
int	pcnt;

/* program parameters */
int	docut=1;		/* do the cutting up into separate files */
int	dolink=0;		/* link don't copy */
char	promptname[128];	/* prompt file name */
int	doprompt=0;		/* prompt file operational */
int	nword=1;		/* number of words to find */
double	srate=20000.0;		/* sampling rate of single channel */
int	winsamp=2000;		/* window size in samples */
int	swapbyte=0;		/* byte swapping */
int	dodebug=0;		/* debug mode */
double	threshadj=0;		/* threshold adjust */
long	headerlen=0;		/* header to skip */
struct item_header ditem;
struct item_header anitem;
struct an_rec *an,*antab;

/* channel allocation */
int	allochan[MAXCHANNELS];
int	nchan=1;		/* # channels */
int	dchan=1;		/* # data channels */
int	schan=-1;		/* speech channel */
int	pchan=-1;		/* prompt channel */

/* read N samples from file */
int	sampread(fid,buf,len)
int	fid;
unsigned char *buf;
int	len;
{
	int	count;
	int	tot=0;
	unsigned char t;
	int	i;

	len = len * sizeof(short);
	while (len > 0) {
		if ((count = read(fid,buf,len))<=0) break;
		len -= count;
		buf += count;
		tot += count;
	}
	if (swapbyte) {
		buf -= tot;
		for (i=0;i<tot;i=i+2) {
			t = buf[i];
			buf[i] = buf[i+1];
			buf[i+1] = t;
		}
	}
	tot /= sizeof(short);
	return(tot);
}

/* calculate energy in buffer */
double	energy(buf,len,oflag)
short	*buf;
int	len;
int	*oflag;
{
	double	tot;
	int	j;

	tot = 0.0;
	
	for (j=1;j<len;j++,buf++) {
		if ((buf[0] < -32760) || (buf[1] > 32760)) *oflag=1;
		tot += ((buf[0]-buf[-1])*(buf[0]-buf[-1]));
	}

	if (tot > 0)
		return(20.0*log10(tot));
	else
		return(0);
}			

void	magnitude(buf,len,mag)
short	*buf;
int	len;
int	*mag;
{
	int	i;
	int	siz;

	for (i=1;i<len;i++,buf++) {
		siz = (buf[0]-buf[-1]);
		if (ABS(siz) > ABS(*mag))
			*mag = siz;
	}
}

/* reduce # words found by 1 */
void reducewords(nw)
int	*nw;
{
	int	i;
	float	emin,emax;
	int	imin,imax;

	/* first look for a very weak peak */
	emin = emax = wordtab[0].wenergy;
	imin = imax = 0;
	for (i=1;i<(*nw);i++) {
		if (wordtab[i].wenergy > emax) {
			emax = wordtab[i].wenergy;
			imax = i;
		}
		if (wordtab[i].wenergy < emin) {
			emin = wordtab[i].wenergy;
			imin = i;
		}
	}
printf("imin=%d emin=%g imax=%d emax=%g\n",imin,emin,imax,emax);
	if (emin > emax/25) {
		/* failed to find a peak small enough */
	
		/* find weakest gap */
		emin = wordtab[0].genergy;
		imin = 0;
		for (i=1;i<(*nw-1);i++) {
			if (wordtab[i].genergy < emin) {
				emin = wordtab[i].genergy;
				imin = i;
			}
		}
	}

	/* collapse weak block with next */
	wordtab[imin].size = wordtab[imin+1].posn + wordtab[imin+1].size -
					wordtab[imin].posn;
	wordtab[imin].wenergy += wordtab[imin+1].wenergy + wordtab[i].genergy;
	wordtab[imin].genergy = wordtab[imin+1].genergy;

	/* pull up rest */
	*nw = *nw - 1;
	for (i=imin+1;i<*nw;i++)
		wordtab[i] = wordtab[i+1];

}

/* duplicate a string */
char	*strsave(str)
char	*str;
{
	char *p;

	p = malloc(strlen(str)+1);
	strcpy(p,str);
	return(p);
}

char	pline[1024];

/* read prompt file */
void readprompts(fname)
char	*fname;
{
	FILE	*ip;
	char	*pname;
	int	i,j;

	pcnt = 0;
	if ((ip=fopen(fname,"r"))==NULL)
		error("could not open '%s'",fname);

	while (fgets(pline,1024,ip)) {
		pname = pline;
		for (i=0;pline[i] && (pline[i]!=' ') && (pline[i]!=':') && (pline[i]!='#') && (pline[i]!='\n');i++)
			/* loop */;
		if (pline[i]=='\0')
			error("malformed prompt string in '%s'",fname);
		if ((pline[i]==' ') || (pline[i]==':') || (pline[i]=='\n')) pline[i++]='\0';
		if ((pcnt>0) && (pline[i]=='#')) {
			/* got a repeat */
			pline[i++]='\0';
			for (j=pcnt-1;j>=0;j--) {
				if (strcmp(pname,(ptable[j])?ptable[j]:"")==0) break;
				if (ttable[j]) free(ttable[j]);
				ttable[j]=NULL;
				if (ptable[j]) free(ptable[j]);
				ptable[j]=NULL;
			}
			if (j < 0)
				error("could not find repeated line '%s' in '%s'",pname,fname);
			if (ttable[j]) free(ttable[j]);
			ttable[j]=NULL;
			if (ptable[j]) free(ptable[j]);
			ptable[j]=NULL;
		}
		if ((pline[i]=='\0') || (pline[i]=='\n')) {
			pline[i] = '\0';
			ptable[pcnt] = strsave(pname);
			ttable[pcnt] = NULL;
		}
		else {
			ptable[pcnt] = strsave(pname);
			while (pline[i]==' ') i++;
			pname = pline + i;
			for (;pline[i] && (pline[i]!='\n');i++)
				/* loop */;			
			pline[i]='\0';
			ttable[pcnt] = strsave(pname);
		}
		pcnt++;
	}
	fclose(ip);
	printf("loaded %d words from '%s'\n",pcnt,fname);

}

/* add a link to an SFS file */
void addlink(fname,bname,itype,cno,cnum,srate,swap,binoff,nsamp)
char	*fname;
char	*bname;
int	itype;
int	cno;
int	cnum;
double	srate;
int	swap;
int	binoff;
int	nsamp;
{
	struct item_header	lnitem;
	struct link_header	lnlink;
	struct stat		filestat;

	/* create output item header */
	sfsheader(&lnitem,itype,0,2,1,1.0/srate,0.0,1,0,0);
	sprintf(lnitem.history,"%s(file=%s,freq=%g,channels=%d/%d%s)",
				PROGNAME,
				bname,
				srate,cno,cnum,
				(swap)?",swapped":"");

	/* create link header */
	memset(&lnlink,0,sizeof(struct link_header));
	strcpy(lnlink.filename,bname);
	strcpy(lnlink.filepath,"");
	lnlink.filetype |= BINARY_TYPE;
	lnlink.swab = swap;
 	lnlink.offset = binoff + (cno - 1)*2;
	lnlink.multiplex = cnum-1;
	stat(bname,&filestat);
	lnlink.linkdate = filestat.st_mtime;
	lnlink.machine = SFSMACHINE;

	/* add linked item to destination file */
	if (!sfswritelink(&lnitem,nsamp,&lnlink,fname))
		error("failed to write link to '%s'",fname);
	
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */

	/* file variables */
	char		binname[128];
	char		filename[SFSMAXFILENAME]; /* SFS data file name */
	int		fid;		/* input file descriptor */
	int		afid;		/* annotation output */

	/* processing variables */
	double		spmin,spavg;
	int		nwin;
	int		i,j,k,l;
	int		len;
	int		pulsemag;
	double		pulsetime;
	struct stat	st;
	int		found;
	double		sthresh;
	int		pthresh;
	int		cursamp;
	int		inword;
	int		skip;
	int		e1,e2,e3;
	int		oload;
	
	for (i=0;i<MAXCHANNELS;i+=2) {
		allochan[i] = SP_TYPE;
		allochan[i+1] = LX_TYPE;
	}

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:c:S:L:P:p:n:alf:DbT:h:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Divide up binary file V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'c' :	/* # channels */
			nchan = atoi(optarg);
			if (nchan > MAXCHANNELS)
				error("too many channels");
			dchan = nchan;
			break;
		case 'S' :	/* speech channel */
			i = atoi(optarg)-1;
			if (i < 0)
				error("specify channels in range 1..N");
			if (i < MAXCHANNELS)
				allochan[i] = SP_TYPE;
			if (schan < 0) schan = i;
			break;
		case 'L' :	/* lx channel */
			i = atoi(optarg)-1;
			if (i < 0)
				error("specify channels in range 1..N");
			if (i < MAXCHANNELS)
				allochan[i] = LX_TYPE;
			break;
		case 'P' :	/* prompt channel */
			pchan = atoi(optarg)-1;
			if (i < 0)
				error("specify channels in range 1..N");
			if (pchan < MAXCHANNELS)
				allochan[pchan] = 0;
			break;
		case 'p' :	/* prompt file */
			strcpy(promptname,optarg);
			doprompt = 1;
			break;
		case 'a' :	/* annotate only */
			docut=0;
			dolink=1;
			break;
		case 'n' :	/* # words */
			nword = atoi(optarg);
			break;
		case 'l' :	/* link */
			dolink = 1;
			break;
		case 'f' :	/* sample rate */
			srate = atof(optarg);
			winsamp = (int)(srate*WINSIZE);
			break;
		case 'b' :	/* byte swap */
			swapbyte = 1;
			break;
		case 'D' :	/* debug */
			dodebug++;
			break;
		case 'T' :	/* threshold adjust */
			threshadj = atof(optarg);
			break;
		case 'h' :	/* header to skip */
			headerlen = atol(optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<3))
		error("usage: %s (-I) (-c nchan) (-S spchan) (-L lxchan) (-P pulsechan) (-p promptfile|-n numword) (-a) (-l) (-T threshadj) (-f samprate) (-h headerlen) (-b) binfile sfsheaderfile",PROGNAME);

	if (pchan >= 0) dchan = nchan-1;
	if (schan < 0)
		for (schan=0;(schan < MAXCHANNELS) && (allochan[schan]!=SP_TYPE);schan++)
			/* loop */;

	/* check channels */
	if ((schan < 0) || (schan >= nchan))
		error("Speech channel out of range 1..%d",nchan);

	/* get binary file name */
	if (optind < argc)
		strcpy(binname,argv[optind]);
	else
		error("no database file specified",NULL);
	optind++;

	/* get SFS filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* get sfs header */
	if ((fid=sfsopen(filename,"r",&head))<0)
		error("access error on '%s'",filename);
	sfsclose(fid);

	/* load words from prompt file */
	if (doprompt) {
		readprompts(promptname);
		nword = pcnt;
	}

	/* open binary file */
#ifdef MSDOS
	if ((fid=open(binname,O_RDONLY|O_BINARY))<0)
#else
	if ((fid=open(binname,O_RDONLY))<0)
#endif
		error("could not open '%s'",binname);
	lseek(fid,headerlen,0);
	
	/* get # samples per channel */
	fstat(fid,&st);
	totsamp = (st.st_size-headerlen)/(nchan*sizeof(short));

	/* set up buffers */
	chanbufsize = winsamp;		/* 1/10 second */
	mpxbufsize = chanbufsize*nchan;
	if ((msp=(short *)calloc(mpxbufsize,sizeof(short)))==NULL)
		error("could not get input buffer");
	for (i=0;i<nchan;i++)
		if ((chan[i]=(short *)calloc(chanbufsize,sizeof(short)))==NULL)
			error("could not get input buffer");
	totwin = totsamp / winsamp;
	if ((ewin=(float *)calloc(totwin+1,sizeof(float)))==NULL)
		error("could not get input buffer");
	
	/* now skim through binary file, setting energies & pulse size */	
	pulsemag = 0;
	pulsetime=0;
	spmin = 1.0E10;
	spavg = 0.0;
	nwin = 0;
	while ((nwin < totwin) && ((len=sampread(fid,msp,mpxbufsize))>0)) {

		/* buffer MUST be an integer # channels */
		len = (len/nchan)*nchan;
		if (len==0) break;

		/* divide into channels */
		for (i=0,k=0;i<len;k++)
			for (j=0;(j<nchan) && (i<len);j++,i++)
				chan[j][k] = msp[i];

		/* get energy in speech channel */
		oload=0;
		ewin[nwin] = energy(chan[schan],k,&oload);
		if (oload)
			fprintf(stderr,"\n\007WARNING: overload at %.1f seconds\n",(float)nwin*WINSIZE);
		if (ewin[nwin] < spmin) spmin = ewin[nwin];
		spavg += ewin[nwin];
		nwin++;

		/* get pulse size in pulse channel */
		if (pchan >= 0) {
			i = pulsemag;
			magnitude(chan[pchan],k,&pulsemag);
			if (i!=pulsemag) pulsetime = nwin*WINSIZE;
		}

		if (((nwin%100)==0) && ttytest()) {
			if (pchan >= 0)
				printf("Pass1: %.1f seconds, pulsemax=%d @ %gs\r",(float)nwin*WINSIZE,pulsemag,pulsetime);
			else
				printf("Pass1: %.1f seconds\r",(float)nwin*WINSIZE);
			fflush(stdout);
		}
	}
	printf("Pass1: %.1f seconds\n",(float)nwin*WINSIZE);

	/* smooth energy */
	for (i=1;i<(totwin-1);i++)
		ewin[i] = 0.25*ewin[i-1] + 0.5*ewin[i] + 0.25*ewin[i+1];

	if (dodebug) {
		/* save energy as track */
		sfsheader(&ditem,TR_TYPE,1,4,1,WINSIZE,0.0,1,0,0);
		sprintf(ditem.history,"%s(file=%s,nchan=%d,schan=%d)",
			PROGNAME,binname,nchan,schan);
		putitem(filename,&ditem,totwin,ewin);
	}

	/* calculate thresholds */
	sthresh = spavg/nwin + threshadj;
	pthresh = ABS(pulsemag)/3;
	fprintf(stderr,"speech threshold = %.1fdB, pulse threshold = %d\n",
		sthresh,pthresh);

	/* if we have pulses, use these to locate windows */
	if (pchan >= 0) {
		/* scan again - this time recording chunks */
		lseek(fid,headerlen,0);
		cursamp=0;
		skip=0;
		totword=0;
		nwin = 0;
		while ((nwin < totwin) && ((len=sampread(fid,msp,mpxbufsize))>0)) {

			/* divide into channels */
			for (i=0,k=0;i<len;k++)
				for (j=0;(j<nchan) && (i<len);j++,i++)
					chan[j][k] = msp[i];

			/* if last window had pulse, skip this one */
			if (skip)
				skip=0;
			else {
				/* look for pulse */
				found=-1;
				for (i=1;i<k;i++) {
					j = chan[pchan][i]-chan[pchan][i-1];
					if ((j > pthresh) || (j < -pthresh))
						found=i;
				}

				if (found >= 0) {
					if (totword > 0)
						wordtab[totword-1].size =
							cursamp - wordtab[totword-1].posn;
					wordtab[totword].posn = cursamp;
					if (totword > 0)
						wordtab[totword].size = wordtab[totword-1].size;
					totword++;
					skip=1;
				}
			}
			cursamp += k;
			nwin++;
			if (((nwin%100)==0) && ttytest()) {
				printf("Pass2: %.1f seconds\r",(float)nwin*WINSIZE);
				fflush(stdout);
			}

		}
		printf("Pass2: %.1f seconds\n",(float)nwin*WINSIZE);
		/* refine positions */
		for (i=0;i<totword;i++) {
			/* move start position in */
			j = wordtab[i].posn/winsamp;
			k = (wordtab[i].posn+wordtab[i].size)/winsamp;
			if (k >= totwin) k=totwin-1;
			if (ewin[j] > sthresh) {
				/* possible run-over - look forward half-sec */
				l = j + 5;
				while ((j < l) && (ewin[j] > sthresh)) j++;
			}
			while ((j < k) && (ewin[j+1] < sthresh)) j++;
			if (ewin[k] > sthresh) {
				/* possible run-over - look forward half-sec */
				l = k + 5;
				while ((k < l) && (ewin[k] > sthresh)) k++;
			}
			while ((k > j) && (ewin[k-1] < sthresh)) k--;
			k++;
			wordtab[i].posn = j * winsamp;
			wordtab[i].size = (k-j) * winsamp;
		}
		if (totword != nword)
			fprintf(stderr,"WARNING: found %d pulses for %d words\n",totword,nword);
	}
	else {
		/* scan energy contour, finding prototype words */
		cursamp=0;
		inword=0;
		totword=0;
		wordtab[0].posn = 0;
		for (i=0;i<totwin;i++) {

			/* look for transitions of energy */
			if ((inword) && (ewin[i]<sthresh)) {
				wordtab[totword].size =
					cursamp - wordtab[totword].posn;
				inword=0;
				totword++;
			}
			else if ((!inword) && (ewin[i]>=sthresh)) {
				wordtab[totword].posn = cursamp;
				if (totword > 0)
					wordtab[totword].size = wordtab[totword-1].size;
				inword=1;
			}
			cursamp += winsamp;
		}
		if (inword) {
			wordtab[totword].size =
				cursamp-wordtab[totword].posn;
			inword=0;
			totword++;
		}
		printf("Initial pass finds %d words\n",totword);
	}
	if (dodebug && (totword > 0)) {
		/* save pulse positions as annotations */
		sfsheader(&anitem,AN_TYPE,-1,1,-1,1.0/srate,0.0,0,0,1);
		sprintf(anitem.history,"%s(file=%s,nchan=%d,pchan=%d)",
			PROGNAME,binname,nchan,pchan);
		antab = (struct an_rec *)sfsbuffer(&anitem,2*totword);
		for (i=0;i<totword;i++) {
			antab[2*i].posn = wordtab[i].posn;
			antab[2*i].size = wordtab[i].size;
			sprintf(antab[2*i].label,"word%d",i+1);
			antab[2*i+1].posn = wordtab[i].posn+wordtab[i].size;
			if (i<(totword-1))
				antab[2*i+1].size = wordtab[i+1].posn - antab[2*i+1].posn;
			else
				antab[2*i+1].size = 1;
			sprintf(antab[2*i+1].label,"/");
		}
		putitem(filename,&anitem,2*totword,antab);
	}

	if (totword < nword)
		fprintf(stderr,"WARNING: did not find enough words\n");
	else if (totword > nword) {
		/* measure energy in each word and gap */
		for (i=0;i<totword;i++) {
			e1 = wordtab[i].posn/winsamp;
			e2 = e1 + wordtab[i].size/winsamp;
			if (i < (totword-1))
				e3 = wordtab[i+1].posn/winsamp;
			else
				e3 = totwin;
			wordtab[i].wenergy = 0.0;
			for (;e1<e2;e1++)
				wordtab[i].wenergy += ewin[e1]-spmin;
			wordtab[i].genergy = 0.0;
			for (;e2<e3;e2++)
				wordtab[i].genergy += ewin[e2]-spmin;
		}
		/* reduce down to number chunks required */
		while (totword > nword)
			reducewords(&totword);
	}

	/* save links and annotations if required */
	if ((docut==0) && (totword > 0) && (totword==nword)) {
		/* link channels into one file file */
		for (j=0;j<nchan;j++)
			if (allochan[j])
				addlink(filename,binname,allochan[j],j+1,nchan,srate,swapbyte,headerlen,totsamp);

		/* save words as annotations */
		sfsheader(&anitem,AN_TYPE,-1,1,-1,1.0/srate,0.0,0,0,1);
		if (doprompt)
			sprintf(anitem.history,"%s(datfile=%s,prfile=%s)",
				PROGNAME,binname,promptname);
		else
			sprintf(anitem.history,"%s(datfile=%s,nword=%d)",
				PROGNAME,binname,nword);
		an = (struct an_rec *)sfsbuffer(&anitem,1);
		if ((afid = sfschannel(filename,&anitem)) < 0)
			error("could not open output channel to '%s'",filename);
		for (i=0;i<totword;i++) {
			an->posn = wordtab[i].posn;
			an->size = wordtab[i].size;
			if (doprompt) {
				if (!ptable[i]) continue;
				strncpy(an->label,ptable[i],MAXANLABEL);
			}
			else
				sprintf(an->label,"word%d",i+1);
			sfswrite(afid,1,an);
			an->posn = wordtab[i].posn+wordtab[i].size;
			if (i<(totword-1))
				an->size = wordtab[i+1].posn - an->posn;
			else
				an->size = 1;
			sprintf(an->label,"/");
			sfswrite(afid,1,an);
		}

		if (!sfsupdate(filename))
			error("update error on '%s'",filename);

	}

	/* that's all folks */
	exit(0);
}


