/* voc8 -- slow/simple 8-channel filterbank */

/* M.A.Huckvale - October 1990 */

/* version 1.0
	- from voc19 with new filters
*/
/* version 1.1 - August 1995
	- add 11025 Hz
*/

#define PROGNAME "voc8"
#define PROGVERS "1.1"
char	*progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH VOC8 SFS1 UCL SFS
.SH NAME
voc8 - slow and simple 8 channel filterbank/vocoder
.SH SYNOPSIS
.B voc8
(-i item) file
.SH DESCRIPTION
.I voc8
is a straightforward implementation of an 8-channel filterbank analyser
using two-pole Butterworth bandpass filters spaced roughly in
critical band proportions.
Energy smoothing done at 25Hz to give default 50Hz framerate.
.PP
.I voc8
has built in filters for signals at 10, 11, 12.8, 16 and 20kHz.  There is no 
limit to the length of the input signal.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version number.
.TP 11
.BI -i item
Select input item.
.SH INPUT ITEMS
.IP SP
Any speech signal sampled at 10, 11, 12.8, 16 or 20kHz.
.SH OUTPUT ITEMS
.IP CO 11
8 channel vocoder energies.
.SH VERSION/AUTHOR
1.1 - Mark Huckvale
.SH SEE ALSO
voc19(1)
*/
/*--------------------------------------------------------------------------*/

/* standard definitions */
#include "SFSCONFG.h"
#include <stdio.h>		/* standard io library */
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"		/* database structures */

/* manifest constants */
#define NCHAN 8		/* # channels */
int	window;			/* window size in samples */


/* filter coefficients */
#include "voc8-100.h"
#include "voc8-110.h"
#include "voc8-128.h"
#include "voc8-160.h"
#include "voc8-200.h"

/* filter data */
short	sp[450];		/* input waveform */
float	fsp[450];		/* filtered waveform */
float	oldsmooth[NCHAN][3];	/* smoothing filter memory */
float	oldfilt[NCHAN][5];		/* channel filter memory */

/* filter processing */
void process(fid,window,numframes,ofid,co)
int		fid;		/* input file */
int		window;		/* input window size */
int		numframes;	/* # input frames */
int		ofid;		/* output file */
struct co_rec	*co;		/* output buffer */
{
	float	*smden=NULL,*den=NULL,*smnum=NULL,*num=NULL;	/* coefficients */
	float	*old;				/* filter memory */
	int	i;				/* sample */
	int	j;				/* channel */
	int	k;				/* window value */
	float	smrect=0.0;			/* smoothing filter output */
	int	cnt=1;				/* framecounter */
	int	winsize;			/* window size */

	/* get analysis window size */
	winsize = window;

	/* estimate # output frames */
	numframes /= winsize;

	/* process each window in turn */
	for (i=0;sfsread(fid,i,winsize,sp)==winsize;i+=winsize) {

		/* for each channel in turn */
		for (j=0;j<NCHAN;j++) {

			/* choose coefficients */
			switch (window) {
			case 200:	/* 10000 Hz */
				smden=smden10;
				smnum=smnum10;
				den=den10[j];
				num=num10[j];
				break;
			case 220:	/* 11000 Hz */
				smden=smden11;
				smnum=smnum11;
				den=den11[j];
				num=num11[j];
				break;
			case 256:	/* 12800 Hz */
				smden=smden12;
				smnum=smnum12;
				den=den12[j];
				num=num12[j];
				break;
			case 320:	/* 16000 Hz */
				smden=smden16;
				smnum=smnum16;
				den=den16[j];
				num=num16[j];
				break;
			case 400:	/* 20000 Hz */
				smden=smden20;
				smnum=smnum20;
				den=den20[j];
				num=num20[j];
				break;
			default:
				return;
			}

			/* select filter memory */
			old=oldfilt[j];

			/* for each sample in window */
			for (k=0;k<winsize;k++) {

				/* get sample */
				old[0]=sp[k];

				/* dot product with denominator */
				old[0] = den[0]*old[0]+
					 den[1]*old[1]+
					 den[2]*old[2]+
					 den[3]*old[3]+
					 den[4]*old[4];

				/* dot product with numerator */
				fsp[k] = num[0]*old[0]+
					 num[1]*old[1]+
					 num[2]*old[2]+
					 num[3]*old[3]+
					 num[4]*old[4];

				/* shift memory */
				old[4]=old[3];
				old[3]=old[2];
				old[2]=old[1];
				old[1]=old[0];

				/* rectify */
				if (fsp[k] < 0) fsp[k] = -fsp[k];
			}

			/* smooth rectified energy */
			old = oldsmooth[j];
			for (k=0;k<winsize;k++) {

				/* dot product with denominator */
				old[0] = smden[0]*fsp[k]+
					 smden[1]*old[1]+
					 smden[2]*old[2];

				/* dot product with numerator */
				if (k==winsize-1) {
					smrect = smnum[0]*old[0]+
						 smnum[1]*old[1]+
						 smnum[2]*old[2];
				}

				/* shift memory */
				old[2]=old[1];
				old[1]=old[0];

			}

			/* store energy for filter */
			if (smrect < 1E-5)
				co->data[j] = -100.0;
			else
				co->data[j] = 20.0 * log10(smrect);
			
		}

		/* write output record */
		co->posn=i;
		co->size=winsize;
		co->flag=0;
		co->mix=0.0;
		co->gain=1.0;
		if (sfswrite(ofid,1,co)!=1)
			error("write error on output file",NULL);

		/* report progress */
		if (ttytest()) {
			printf("\rFrame %d/%d",cnt++,numframes);
			fflush(stdout);
		}
	}

}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it=SP_TYPE;	/* item type selection */
	char		*ty="0";	/* item match selection */
	/* input file variables */
	char		filename[SFSMAXFILENAME]; /* sfs file name */
	int		fid;		/* input file descriptor */
	struct item_header spitem;	/* item header for speech data */
	/* output file variables */
	int		ofid;		/* output file descriptor */
	struct item_header coitem;	/* item header for vocoder data */
	struct co_rec	*co;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: 8-channel vocoder V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it != SP_TYPE)
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open data file */
	if ((fid=sfsopen(filename,"w",NULL)) < 0) {
		if (fid==-1)
			error("cannot find file: %s",filename);
		else
			error("access error on %s",filename);
	}

	/* locate input speech */
	if (!sfsitem(fid,it,ty,&spitem))
		error("cannot find SP data in %s",filename);
	window = (int)(0.45 + 0.02/spitem.frameduration);
	if ((window!=200) && (window!=220) && (window!=256) &&
		(window!=320) && (window!=400))
		error("input signal not sampled at 10, 11, 12.8, 16 or 20kHz",NULL);

	/* create output item header */
	sfsheader(&coitem,CO_TYPE,1,sizeof(float),
		NCHAN+sfsstruct[CO_TYPE]/sizeof(float),
		spitem.frameduration,spitem.offset,window,0,0);
	sprintf(coitem.history,"%s(%d.%02d)",
			PROGNAME,spitem.datatype,spitem.subtype);
	strcpy(coitem.params,"labels=280|455|675|970|1370|1900|2600|3500");

	/* open output channel */
	if ((ofid=sfschannel(filename,&coitem)) < 0)
		error("cannot open output channel",NULL);

	/* get buffer */
	if ((co=(struct co_rec *)sfsbuffer(&coitem,1))==NULL)
		error("cannot get output buffer",NULL);

	/* do processing */
	process(fid,window,spitem.numframes,ofid,co);

	/* update data file */
	if (!sfsupdate(filename))
		error("update error on %s",filename);

	/* clear progress */
	if (ttytest()) {
		printf("\r                       \r");
		fflush(stdout);
	}

	/* that's all folks */
	exit(0);
}

