/* andict -- build dictionary of annotation realisations from coefficient item */

/* M.A.Huckvale - January 1990 */

/* version 1.0 */

#define PROGNAME "andict"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*-------------------------------------------------------------------*/
/**MAN
.TH ANDICT 1 UCL SFS
.SH NAME
andict - build/update dictionary of annotation templates
.SH SYNOPSIS
.B andict
(-i item) dictfile file
.SH DESCRIPTION
.I andict
takes as input a coefficient item and an annotation item and 
maintains a dictionary of coefficient templates in a separate dictionary
file.  All coefficient types in the dictionary file must be the same.
The default operation is to add templates to the dictionary file for annotations
that are currently not present.  To replace an annotation template, remove it
with 
.I remove(SFS1)
before calling 
.I andict.
.PP
.I Options:
.TP 11
.B -I
Identify the program name and version.
.TP 11
.BI -i item
Select input item number.
.SH INPUT ITEMS
.IP AN
Annotations
.IP CO
Coefficient data
.SH OUTPUT ITEMS
.IP CO
One coefficient segment for each different annotation.
.SH VERSION/AUTHOR
1.0 - Mark Huckvale
.SH SEE ALSO
codict
*/
/*---------------------------------------------------------------*/

/* include files */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <malloc.h>
#include <math.h>
#include "sfs.h"

/* global data */
struct item_header	anitem;		/* input annotation item */
struct an_rec		*an;
struct item_header	coitem;		/* input coefficient item */
struct co_rec		*co;
double			*cotime1,*cotime2;
struct item_header	rfitem;		/* reference coefficient item */
int			newdict=0;
int			ref_framesize;	/* reference vector size */
int			ref_samplerate;	/* reference sampling rate */
struct item_header	opitem;		/* output coefficient item */
double			*cotime;

char			filename[SFSMAXFILENAME];
char			rfilename[SFSMAXFILENAME];

/* annotation list */
#define MAXANN		500		/* max # different annotations */
char	*label[MAXANN];
int	numlab=0;

#define MAX(x,y)	(((x)>(y))?(x):(y))


/* save string in dynamic memory */
char	*strsave(char *s)
{
	char	*p;
	int	len=strlen(s)+1;
	if ((p=malloc(len))==NULL)
		error("cannot save string",NULL);
	strcpy(p,s);
	return(p);
}

/* binary string search */
int strfind(char *s,char **t,int num)
{
	int	i,j,k;
	int	c;
	
	if (num==0)
		return(-1);
	else {
		i=0;
		j=num-1;
		do {
			k=(i+j)/2;
			if ((c=strcmp(s,t[k])) > 0)
				i=k+1;
			else
				j=k-1;
		} while (c && (i <= j));
		if (c)
			return(-1);
		else
			return(k);
	}
}

/* maintain string table */
void strtable(char *s,char **t,int *num)
{
	int	i;
	
	/* see if string in table */
	if (strfind(s,t,*num) < 0) {
		/* add to table */
		i = *num;
		while ((i>0) && (strcmp(s,t[i-1])<0)) {
			t[i] = t[i-1];
			i--;
		}
		t[i]=strsave(s);
		(*num)++;
	}
}

/* main program */
int main(int argc,char **argv)
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */

	/* processing variables */
	int		fid,rfid,ofid;
	int		it;
	char		*ty;
	char		*antype="0";
	char		*cotype="0";
	int		i,j;
	double		anstart,anstop;
	int		coidx,coidx2;
	int		coposn;
	char		*anlab;
	int		ocount=0;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Annoatation Template Dictionary V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* item number */
			if (itspec(optarg,&it,&ty)==0) {
				if (it==AN_TYPE)
					antype=ty;
				else if (it==CO_TYPE)
					cotype=ty;
				else
					error("bad item specification",NULL);
			}
			else
				error("illegal item specification",NULL);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	/* check command line */
	if (errflg || (argc<3))
		error("usage: %s (-I) (-i item) dictfile file",PROGNAME);

	/* get dictionary filename */
	if (optind < argc)
		strcpy(rfilename,sfsfile(argv[optind]));
	else
		error("no dictionary file specified",NULL);
	optind++;

	/* get data filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open dictionary file */
	if ((rfid=sfsopen(rfilename,"w",NULL)) < 0)
		error("access error on '%s'",rfilename);

	/* see if templates in dictionary file */
	if (sfsitem(rfid,CO_TYPE,"*",&rfitem)) {
		/* some templates in file */
		ref_framesize = rfitem.framesize;
		ref_samplerate = (int)(0.5 + 0.01/rfitem.frameduration);

		/* get annotation from history */
		anlab = params(rfitem.history,"label","");
		if (!*anlab)
			error("unlabelled coeffient item in '%s'",rfilename);
		strtable(anlab,label,&numlab);

		/* scan coefficient items */
		while (sfsnextitem(rfid,&rfitem)) if (rfitem.datatype==CO_TYPE) {
			if ((rfitem.framesize != ref_framesize) ||
			    (ref_samplerate != (int)(0.5+0.01/rfitem.frameduration)))
				error("mismatched coefficient item in '%s'",rfilename);
			/* get annotation from history */
			anlab = params(rfitem.history,"label","");
			if (!*anlab)
				error("unlabelled coeffient item in '%s'",rfilename);
			strtable(anlab,label,&numlab);
			if (numlab==MAXANN)
				error("too many annotation templates in '%s'",rfilename);
		}
		fprintf(stderr,"%s: %d annotation templates loaded from '%s'\n",PROGNAME,numlab,rfilename);
	}
	else {
		fprintf(stderr,"%s: starting new dictionary '%s'\n",PROGNAME,rfilename);
		newdict++;
	}
	sfsclose(rfid);

	/* open data file */
	if ((fid=sfsopen(filename,"r",NULL)) < 0)
		error("access error on '%s'",filename);
	
	/* load annotation item */
	getitem(filename,AN_TYPE,antype,&anitem,(void *)&an);

	/* locate input coefficients */
	if (!sfsitem(fid,CO_TYPE,cotype,&coitem))
		error("could not find input CO item in '%s'",filename);
	if (newdict) {
		ref_framesize = coitem.framesize;
		ref_samplerate = (int)(0.5 + 0.01/coitem.frameduration);
	}
	else if ((ref_framesize != coitem.framesize) ||
	    (ref_samplerate != (int)(0.5+0.01/coitem.frameduration)))
		error("mismatch in coeffient items between dictionary and file");
	if ((co=(struct co_rec *)sfsbuffer(&coitem,1))==NULL)
		error("could not get buffer for CO data");

	/* load coefficient positions */
	if ((cotime=(double *)calloc(coitem.numframes,sizeof(double)))==NULL)
		error("could not get buffer for CO timing",NULL);
	for (i=0;sfsread(fid,i,1,co);i++) {
		cotime[i] = coitem.offset + co->posn * coitem.frameduration;
	}

	/* process each annotation in turn */
	for (i=0;i<anitem.numframes;i++) if (strfind(an[i].label,label,numlab) < 0) {
		/* annotation not in dictionary */
		anstart = anitem.offset + an[i].posn*anitem.frameduration;
		anstop  = anstart + an[i].size*anitem.frameduration;
		coidx = 0;
		while ((coidx < coitem.numframes) &&
		       (anstart > cotime[coidx]))
			coidx++;
		coidx2=coidx;
		while ((coidx2 < coitem.numframes) &&
		       (anstop > cotime[coidx2]))
			coidx2++;
		if (coidx2==coitem.numframes) {
			fprintf(stderr,"insufficient CO data for annotation '%s'",an[i].label);
			break;
		}

		/* create template item in dictionary */
		sfsheader(&opitem,CO_TYPE,coitem.floating,coitem.datasize,
				coitem.framesize,coitem.frameduration,
				coitem.offset,coitem.windowsize,
				coitem.overlap,coitem.lxsync);
		sprintf(opitem.history,
			"%s(source=%s,items=%d.%02d,%d.%02d,label=%s)",
			PROGNAME,filename,
			anitem.datatype,anitem.subtype,
			coitem.datatype,coitem.subtype,
			an[i].label);
		if ((ofid=sfschannel(rfilename,&opitem)) < 0)
			error("could not open output channel to '%s'",rfilename);
		coposn = -1;
		for (j=coidx;j<=coidx2;j++) {
			sfsread(fid,j,1,co);
			if (coposn < 0) coposn = co->posn;
			co->posn -= coposn;
			sfswrite(ofid,1,co);
		}
		if ((++ocount%10)==0)
			if (!sfsupdate(rfilename))
				error("update error on '%s'",rfilename);
		strtable(an[i].label,label,&numlab);
		printf("add %s\n",an[i].label);
		if (numlab==MAXANN)
			error("too many annotation templates in '%s'",rfilename);
	}
	
	/* that's all folks */
	sfsclose(fid);
	if (!sfsupdate(rfilename))
		error("update error on '%s'",rfilename);
	exit(0);
}

