{ complex.inc - SPC support for complex arithmetic }
{
	Defines basic arithmetic operations on complex numbers
	as well as complex exponential, magnitude and argument

	Pascal (c) 1993 Mark Huckvale, University College London
}

{ make a complex number }
function CMake(rval,ival:real):complex;
var	c:complex;
begin
	c.re := rval;
	c.im := ival;
	CMake := c;
end;

{ add complex numbers }
function CAdd(c1,c2:complex):complex;
var	c:complex;
begin
	c.re := c1.re + c2.re;
	c.im := c1.im + c2.im;
	CAdd := c;
end;

{ subtract complex numbers }
function CSub(c1,c2:complex):complex;
var	c:complex;
begin
	c.re := c1.re - c2.re;
	c.im := c1.im - c2.im;
	CSub := c;
end;

{ multiply complex numbers }
function CMult(c1,c2:complex):complex;
var	c:complex;
begin
	c.re := c1.re * c2.re - c1.im * c2.im;
	c.im := c1.re * c2.im + c1.im * c2.re;
	CMult := c;
end;

{ divide complex numbers }
function CDiv(c1,c2:complex):complex;
var
	c:complex;
	denom:real;
begin
	if (c2.re=0) and (c2.im=0) then begin
		writeln('CDiv: division by zero');
		c := c2;
	end
	else begin
		denom := c2.re * c2.re + c2.im * c2.im;
		c.re := (c1.re * c2.re + c1.im * c2.im) / denom;
		c.im := (c1.im * c2.re - c1.re * c2.im) / denom;
	end;
	CDiv := c;
end;

{ magnitude of complex number }
function CMag(c:complex):real;
begin
	CMag := sqrt(c.re*c.re + c.im*c.im);
end;

{ 'C' function for arc tangent }
external function _atan2(y,x:real):real;

{ argument of complex number }
function CArg(c:complex):real;
begin
	if (c.im=0) and (c.re=0) then
		CArg := 0
	else
		CArg := _atan2(c.im,c.re);
end;

{ complex exponential }
function CExp(c1:complex):complex;
var
	c:complex;
	r:real;
begin
	r := exp(c1.re);
	c.re := r * cos(c1.im);
	c.im := r * sin(c1.im);
	CExp := c;
end;

function CSqrt(cin:complex):complex;
var
	c:complex;
	x,y,w,r:real;
begin
	if (cin.re=0) and (cin.im=0) then begin
		c.re := 0.0;
		c.im := 0.0;
	end
	else begin
		x := abs(cin.re);
		y := abs(cin.im);
		if (x >= y) then begin
			r := y/x;
			w := sqrt(x)*sqrt(0.5*(1.0+sqrt(1.0+r*r)));
		end
		else begin
			r := x/y;
			w := sqrt(y)*sqrt(0.5*(r+sqrt(1.0+r*r)));
		end;
		if cin.re >= 0.0 then begin
			c.re := w;
			c.im := cin.im/(2.0*w);
		end
		else if (cin.im >= 0.0) then begin
			c.im := w;
			c.re := cin.im/(2.0*c.im);
		end
		else begin
			c.im := -w;
			c.re := cin.im/(2.0*c.im);
		end;
	end;
	CSqrt := c;
end;

function CScale(cin:complex;fac:real):complex;
var
	c:complex;
begin
	c.re := cin.re * fac;
	c.im := cin.im * fac;
	CScale := c;
end;
