/* winp -- DIG routines for WIN32 Printer Driver compatible graphics */

/* Mark Huckvale - University College London */

/* version 1.0 - November 1997 */

/* written for Microsoft Visual C/C++ V4 */

#include "SFSCONFG.h"
#ifdef	EVD

#include <windows.h>
#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <time.h>
#include <string.h>
#include "dig.h"
#include "digdata.h"

/* local storage of parameters */
struct {
	HDC	hDC;		/* printer device context */
	HDC	hmemDC;		/* memory hDC for bitmaps */
	HFONT	hFont;		/* suitable printing font */
	int	mode;		/* system modes */
	int	subtype;	/* subtype as supplied from digmap */
	int	xoff,yoff;	/* page offset */
	int	dpi;		/* dots per inch */
	int	chwidth,chheight;
} digDEVD;

#define W32P_HDC_OK	1
#define W32P_START	2
#define W32P_FONT	4

/* forward declarations */
static void w32p_pen_colour();
static void w32p_brush_colour();

static void w32p_get_hdc()
{
	char	lbuf[128];
	char	*n,*d,*p;

	if (!(digDEVD.mode & W32P_HDC_OK)) {
		GetProfileString("windows", "device", ",,,", lbuf, 128);
		n = strtok(lbuf,",");
		d = strtok(NULL,", ");
		p = strtok(NULL,", ");
		if (n && d && p && *n && *d && *p) {
			digDEVD.hDC = CreateDC(d,n,p,NULL);
			if (digDEVD.hDC==NULL)
				error("failed to create printer device context");
			digDEVD.hmemDC = CreateCompatibleDC(digDEVD.hDC);
			digDEVD.mode |= W32P_HDC_OK;
		}
		else
			error("failed to find printer driver");
	}
}

static void w32p_get_font()
{
	LOGFONT		lf;
	HFONT		oldfont;
	TEXTMETRIC	tm;
	
	if (!(digDEVD.mode & W32P_FONT)) {
		memset((char *)&lf,0,sizeof(LOGFONT));

		lf.lfHeight = digDEVD.dpi/10;
		lf.lfWeight = FW_NORMAL;
		lf.lfEscapement = 0;
		lf.lfOrientation = 0;
		lf.lfPitchAndFamily=FIXED_PITCH | FF_MODERN;

		digDEVD.hFont = CreateFontIndirect(&lf);

		/* set up font */
		oldfont = SelectObject(digDEVD.hDC,digDEVD.hFont);
		DeleteObject(oldfont);
		SetTextAlign(digDEVD.hDC,TA_BOTTOM|TA_LEFT);
		GetTextMetrics(digDEVD.hDC,&tm);

		digDEVD.chheight = tm.tmHeight;
		digDEVD.chwidth = tm.tmAveCharWidth;

		digDEVD.mode |= W32P_FONT;
	}
	
}

/* get graphics parameters */
int w32p_params(subtype,digdev)
int	subtype;
struct digdevice *digdev;
{
	w32p_get_hdc();
	
	digDEVD.dpi = GetDeviceCaps(digDEVD.hDC,LOGPIXELSX);
	digdev->nhoriz = GetDeviceCaps(digDEVD.hDC,HORZRES)-2*digDEVD.dpi/3;
	digdev->nvert  = GetDeviceCaps(digDEVD.hDC,VERTRES)-2*digDEVD.dpi/3;
	digDEVD.xoff = digDEVD.dpi/3;
	digDEVD.yoff = digdev->nvert + digDEVD.dpi/3;

	w32p_get_font();
	
	/* return other image characteristics */
	digdev->aspect = (float)1.0;
 	digdev->chwidth = digDEVD.chwidth;
	digdev->chheight = digDEVD.chheight;
	digdev->greylevels = 16;
	digdev->greypixwidth = 4;
	digdev->greypixheight = 4;

	return(0);
}

/* open output */
int w32p_open()
{
	DOCINFO	di;

	if (!(digDEVD.mode & W32P_START)) {
		w32p_get_hdc();
		w32p_get_font();

		di.cbSize = sizeof(DOCINFO);
		di.lpszDocName = "SFS/DIG Print Job";
		di.lpszOutput = NULL;
		di.lpszDatatype = NULL;
		di.fwType = 0;

		if (StartDoc(digDEVD.hDC,&di)==SP_ERROR)
			error("StartDoc fails");

		if (StartPage(digDEVD.hDC)<=0)
			error("StartPage fails");

		digDEVD.mode |= W32P_START;
	}
	return(0);
}

/* close file */
void w32p_close()
{
	HFONT	font;

	if (digDEVD.mode & W32P_START) {

		/* End of page */
		if (EndPage(digDEVD.hDC)<=0)
			error("EndPage fails");

		/* End of document */
		if (EndDoc(digDEVD.hDC)<=0)
			error("EndDoc fails");

	} 

	if (digDEVD.mode & W32P_FONT) {
		font = SelectObject(digDEVD.hDC,GetStockObject(SYSTEM_FONT));
		DeleteObject(font);
	}

	if (digDEVD.mode & W32P_HDC_OK) {
		/* Delete the printer DC. */
		DeleteDC(digDEVD.hmemDC);
		DeleteDC(digDEVD.hDC);
	}

	digDEVD.mode = 0;
}

void w32p_inter()
{
	w32p_close();
}

/* clear screen */
int w32p_clear()
{
	return(0);
}

/* set up pen colour from bundle */
static void w32p_pen_colour(bundle)
int	bundle;
{
	static int oldbundle = -1;
	HPEN	pen,oldpen;
	int	dash=0,width=0;

	if (bundle==oldbundle) return;
	oldbundle = bundle;

	if (bundle < 6)
		width = 4;
	else if (bundle < 20)
        	width = 2;
	else
		width = 1;

        if (bundle < 35)
		dash = PS_SOLID;
        else if (bundle < 50)
		dash = PS_DASH;
        else if (bundle < 65)
		dash = PS_DOT;
        else if (bundle < 80)
		dash = PS_DASHDOT;
        else 
		dash = PS_DASHDOTDOT;

	pen = CreatePen(dash,width,RGB(0,0,0));

	oldpen = SelectObject(digDEVD.hDC,pen);

	DeleteObject(oldpen);
}

/* set up brush colour from bundle */
static void w32p_brush_colour(bundle)
int	bundle;
{
	static int oldbundle = -1;
	HBRUSH	brush,oldbrush;

	if (bundle==oldbundle) return;
	oldbundle = bundle;

	brush = (bundle==0) ? GetStockObject(WHITE_BRUSH) : GetStockObject(BLACK_BRUSH);

	oldbrush = SelectObject(digDEVD.hDC,brush);

	DeleteObject(oldbrush);
}

/* plot poly-line */
void w32p_polyline(bundle,buff,len)
int bundle;
short *buff;
int	len;
{
	int     x,y;
	
	/* set up plot colour */
	w32p_pen_colour(bundle);

	/* move to first point */
	x = digDEVD.xoff + *buff++;
	y = digDEVD.yoff - *buff++;
	len-=(int32)2;
	MoveToEx(digDEVD.hDC,x,y,NULL);

	/* plot lines */
	while (len > (int32)0) {
		x = digDEVD.xoff + *buff++;
		y = digDEVD.yoff - *buff++;
		len -= (int32)2;
		LineTo(digDEVD.hDC, x, y);
	}
}

/* draw text */
void w32p_text(bundle,x,y,str)
int bundle;
int x;
int y;
char *str;
{
	/* set up plot colour */
	w32p_pen_colour(bundle);

	/* draw string */
	SetBkMode(digDEVD.hDC,TRANSPARENT);
	TextOut(digDEVD.hDC,digDEVD.xoff+x,digDEVD.yoff-y,str,strlen(str));
	SetBkMode(digDEVD.hDC,OPAQUE);

}

/* draw a filled rectangle */
void w32p_fillrect(bundle,x1,y1,x2,y2)
int bundle;
int x1;
int y1;
int x2;
int y2;
{
	w32p_pen_colour(bundle);
	w32p_brush_colour(bundle);

	Rectangle(digDEVD.hDC,digDEVD.xoff+x1,digDEVD.yoff-y1,digDEVD.xoff+x2,digDEVD.yoff-y2);

}

/* draw a circle */
void w32p_circle(bundle,xc,yc,r)
int bundle;
int xc;
int yc;
int r;
{
	/* set up plot colour */
	w32p_pen_colour(bundle);

	/* draw circle */
	Arc(digDEVD.hDC,digDEVD.xoff+xc-r,digDEVD.yoff-yc-r,digDEVD.xoff+xc+r,digDEVD.yoff-yc+r,
		digDEVD.xoff+xc+r,digDEVD.yoff-yc,digDEVD.xoff+xc+r,digDEVD.yoff-yc);
}

/* draw a filled circle */
void w32p_fcircle(bundle,xc,yc,r)
int bundle;
int xc;
int yc;
int r;
{
	w32p_pen_colour(bundle);
	w32p_brush_colour(bundle);

	Ellipse(digDEVD.hDC,digDEVD.xoff+xc-r,digDEVD.yoff-yc-r,digDEVD.xoff+xc+r,digDEVD.yoff-yc+r);

}

/* dot dither bitmaps */
int	initdither=0;
HBITMAP	dither[16];
unsigned short bitdata[16][4]={
	{ 0x00F0, 0x00F0, 0x00F0, 0x00F0 },
	{ 0x00F0, 0x00B0, 0x00F0, 0x00F0 },
	{ 0x00F0, 0x0090, 0x00F0, 0x00F0 },
	{ 0x00F0, 0x0090, 0x00D0, 0x00F0 },
	{ 0x00F0, 0x0090, 0x0090, 0x00F0 },
	{ 0x00F0, 0x0090, 0x0010, 0x00F0 },
	{ 0x00F0, 0x0010, 0x0010, 0x00F0 },
	{ 0x0070, 0x0010, 0x0010, 0x00F0 },
	{ 0x0030, 0x0010, 0x0010, 0x00F0 },
	{ 0x0010, 0x0010, 0x0010, 0x00F0 },
	{ 0x0000, 0x0010, 0x0010, 0x00F0 },
	{ 0x0000, 0x0000, 0x0010, 0x00F0 },
	{ 0x0000, 0x0000, 0x0000, 0x00E0 },
	{ 0x0000, 0x0000, 0x0000, 0x00C0 },
	{ 0x0000, 0x0000, 0x0000, 0x0080 },
	{ 0x0000, 0x0000, 0x0000, 0x0000 },
};

/* grey-level display */
void w32p_gscale(x,y,height,buff)
int x;
int y;
int height;
unsigned char *buff;
{
	int		i;
	unsigned char 	*p,lastc;

	if (initdither==0) {
		for (i=0;i<16;i++)
			dither[i] = CreateBitmap(4,4,1,1,bitdata[i]);
		initdither=1;
	}

	/* 4x4 dot dither */
	p = buff;
	lastc = *p;
	SelectObject(digDEVD.hmemDC,dither[*p & 0xF]);
	for (i=height-1;i>=0;i--,p++) {
		if (*p != lastc) {
			SelectObject(digDEVD.hmemDC,dither[*p & 0xF]);
			lastc = *p;
		}
		BitBlt(digDEVD.hDC,digDEVD.xoff+x,digDEVD.yoff-y-4*i,4,4,digDEVD.hmemDC,0,0,SRCCOPY);
	}
}

/* wait for key routine - NULL don't wait */
void w32p_pause()
{
}	
#endif
