/* DSP routines from Mike Brookes, Imperial College London */

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "libic.h"

/* Would probably be speeded up if COMPLEX datatype was not used */

#define NAME "cfdift"
#define ROOTHALF (0.5*ROOT2)

void cfdift(data,len,m)
COMPLEX *data;
int m,len;
/*
	Complex dif fourier transform with len = 2**m

	data[len] -> data[len]
	Output is bit reversed

	Algorithm from ASSP34 p155 Feb 1986
	Uses 2 butterfly stages plus special radix 4 and 8 stages

	Minimum length is 8
*/
{

    REAL *sines,*psine,*psine3;
    COMPLEX *datalen,*datan4,*dataj,*data0,*data1,*data2,*data3;
    COMPLEX *data4,*data5,*data6,*data7;
    COMPLEX t0,t1,t2,t3,t4,t5,t6,t7;
    REAL ss1,ss3,cc1,cc3;
    int n2,n4;
    int inc,inc3,sinc,cosoff;
    int id;

#ifdef IAG
fprintf(stderr,"cfdift(x,%d,%d)\n",len,m);
#endif
    sintable(m,&sines,&sinc);
    cosoff = sinc*len/4;
    datalen = data + len;

/* L shaped butterflies */

    inc = sinc;
    inc3 = 3*inc;
    for (n4 = len >> 2; n4 >= 4; n4 = n4 >> 1) {

#ifdef IAG
fprintf(stderr,"n4 = %d, n2 = %d\n",n4,n2);
#endif

	n2 = n4 << 2;
	data0 = dataj = data;
	id = n2 << 1;
	do {
	    do {
		data1 = data0 + n4;
		data2 = data1 + n4;
		data3 = data2 + n4;
#ifdef IAG
fprintf(stderr,"No Mult Radix 2+4: %d,%d,%d,%d\n",data0-data,data1-data,data2-data,data3-data);
#endif

		t1.r = data0->r - data2->r;
		data0->r = data0->r + data2->r;
		t2.r = data1->r - data3->r;
		data1->r = data1->r + data3->r;
		t1.i = data0->i - data2->i;
		data0->i = data0->i + data2->i;
		t2.i = data1->i - data3->i;
		data1->i = data1->i + data3->i;

		data2->r = t1.r + t2.i;
		data2->i = t1.i - t2.r;
		data3->r = t1.r - t2.i;
		data3->i = t2.r + t1.i;
#ifdef IAG
fprintf(stderr,"(%6.2g,%6.2g),(%6.2g,%6.2g),(%6.2g,%6.2g),(%6.2g,%6.2g)\n",
data0->r,data0->i, data1->r,data1->i,
data2->r,data2->i, data3->r,data3->i);
#endif

		} while ((data0 += id) < datalen);
	    data0 = id + id - n2 + dataj;
	    id = id << 2;
	    } while (data0 < datalen);

	psine = psine3 = sines;
	datan4 = data + n4;
	while (++dataj < datan4) {
	    psine += inc;
	    psine3 += inc3;
	    cc1 = psine[cosoff];
	    cc3 = psine3[cosoff];
	    ss1 = *psine;
	    ss3 = *psine3;
	    data0 = dataj;
	    id = n2 << 1;
	    do {
		do {
		    data1 = data0 + n4;
		    data2 = data1 + n4;
		    data3 = data2 + n4;
#ifdef IAG
fprintf(stderr,"Radix 2+4: %d,%d,%d,%d\n",data0-data,data1-data,data2-data,data3-data);
#endif

		    t1.r = data0->r - data2->r;
		    data0->r = data0->r + data2->r;
		    t2.r = data1->r - data3->r;
		    data1->r = data1->r + data3->r;
		    t1.i = data0->i - data2->i;
		    data0->i = data0->i + data2->i;
		    t2.i = data1->i - data3->i;
		    data1->i = data1->i + data3->i;

		    t3.i = t1.r - t2.i;
		    t1.r = t1.r + t2.i;
		    t2.i = t1.i - t2.r;
		    t2.r = t2.r + t1.i;

		    data2->r = t1.r*cc1 + t2.i*ss1;
		    data2->i = t2.i*cc1 - t1.r*ss1;
		    data3->r = t3.i*cc3 + t2.r*ss3;
		    data3->i = t2.r*cc3 - t3.i*ss3;
#ifdef IAG
fprintf(stderr,"(%6.2g,%6.2g),(%6.2g,%6.2g),(%6.2g,%6.2g),(%6.2g,%6.2g)\n",
    data0->r,data0->i, data1->r,data1->i,
    data2->r,data2->i, data3->r,data3->i);
#endif

		    } while ((data0 += id) < datalen);
		data0 = id + id - n2 + dataj;
		id = id << 2;
		} while (data0 < datalen);
	    }
	inc = inc << 1;
	inc3 = inc3 << 1;
	}

/* Radix 8 stage */
/* n4 = 2, n2 = 8 */

    data0 = data;
    id = 16;
    do {
	do {
	    data1 = data0 + 1;
	    data2 = data1 + 1;
	    data3 = data2 + 1;
	    data4 = data3 + 1;
	    data5 = data4 + 1;
	    data6 = data5 + 1;
	    data7 = data6 + 1;

#ifdef IAG
fprintf(stderr,"Radix 8: %d - %d\n",data0-data,data7-data);
#endif

	    t1.r = data0->r - data4->r;
	    data0->r = data0->r + data4->r;
	    t2.r = data2->r - data6->r;
	    data2->r = data2->r + data6->r;
	    t1.i = data0->i - data4->i;
	    data0->i = data0->i + data4->i;
	    t2.i = data2->i - data6->i;
	    data2->i = data2->i + data6->i;

	    t4.r = t1.r + t2.i;
	    t4.i = t1.i - t2.r;
	    t6.r = t1.r - t2.i;
	    t6.i = t1.i + t2.r;

	    t1.r = data1->r - data5->r;
	    data1->r = data1->r + data5->r;
	    t2.r = data3->r - data7->r;
	    data3->r = data3->r + data7->r;
	    t1.i = data1->i - data5->i;
	    data1->i = data1->i + data5->i;
	    t2.i = data3->i - data7->i;
	    data3->i = data3->i + data7->i;

	    t3.i = t2.i - t1.r;
	    t1.r = t1.r + t2.i;
	    t2.i = t1.i - t2.r;
	    t2.r = t2.r + t1.i;

	    t5.r = (t1.r + t2.i)*ROOTHALF;
	    t5.i = (t2.i - t1.r)*ROOTHALF;
	    t7.r = (t2.r + t3.i)*ROOTHALF;
	    t7.i = (t3.i - t2.r)*ROOTHALF;

	    data4->r = t4.r + t5.r;
	    data4->i = t4.i + t5.i;
	    data5->r = t4.r - t5.r;
	    data5->i = t4.i - t5.i;
	    data6->r = t6.r + t7.r;
	    data6->i = t6.i + t7.i;
	    data7->r = t6.r - t7.r;
	    data7->i = t6.i - t7.i;
#ifdef IAG
fprintf(stderr,"(%6.2g,%6.2g),(%6.2g,%6.2g),(%6.2g,%6.2g),(%6.2g,%6.2g)\n",
    data0->r,data0->i, data1->r,data1->i,
    data2->r,data2->i, data3->r,data3->i);
fprintf(stderr,"(%6.2g,%6.2g),(%6.2g,%6.2g),(%6.2g,%6.2g),(%6.2g,%6.2g)\n",
    data4->r,data4->i, data5->r,data5->i,
    data7->r,data7->i, data7->r,data7->i);
#endif

	    } while ((data0 += id) < datalen);
	data0 = id + id - 8 + data;
	id = id << 2;
	} while (data0 < datalen);


/* Radix 4 stage */

/*    n4 = 1; n2 = 4 */

#ifdef IAG
fprintf(stderr,"n4 = %d, n2 = %d\n",n4,n2);
#endif

    data0 = data;
    id = 8;
    do {
	do {
	    data1 = data0 + 1;
	    data2 = data1 + 1;
	    data3 = data2 + 1;
#ifdef IAG
fprintf(stderr,"Radix 4: %d - %d\n",data0-data,data3-data);
#endif

	    t2.r = data0->r - data2->r;
	    t0.r = data0->r + data2->r;
	    t3.r = data1->r - data3->r;
	    t1.r = data1->r + data3->r;
	    t2.i = data0->i - data2->i;
	    t0.i = data0->i + data2->i;
	    t3.i = data1->i - data3->i;
	    t1.i = data1->i + data3->i;

	    data0->r = t0.r + t1.r;
	    data0->i = t0.i + t1.i;
	    data1->r = t0.r - t1.r;
	    data1->i = t0.i - t1.i;
	    data2->r = t2.r + t3.i;
	    data2->i = t2.i - t3.r;
	    data3->r = t2.r - t3.i;
	    data3->i = t2.i + t3.r;
#ifdef IAG
fprintf(stderr,"(%6.2g,%6.2g),(%6.2g,%6.2g),(%6.2g,%6.2g),(%6.2g,%6.2g)\n",
data0->r,data0->i, data1->r,data1->i,
data2->r,data2->i, data3->r,data3->i);
#endif

	    } while ((data0 += id) < datalen);
	data0 = id + id - 4 + data;
	id = id << 2;
	} while (data0 < datalen);

    return;
    }

#ifdef EMO

#define LEN 32
#define LOGLEN 5

main ()
{
    COMPLEX error,data[LEN],odata[LEN];
    REAL sum,theta;
    int i;

    printf("Test for cfdift: (1.0,1.1) + 2 [f] + 3j [3f]\n");
    for (i=0; i<LEN; i++) {
	theta = (TWOPI * i) / LEN;
	odata[i].r = data[i].r = 1.0 + 2.0*cos(theta) - 3.0*sin(3.0*theta);
	odata[i].i = data[i].i = 1.1 + 2.0*sin(theta) + 3.0*cos(3.0*theta);
	}

    cfdift(data,LEN,LOGLEN);
    cbitrev(data,LEN);

    for (i=0; i<LEN; i++) {
	data[i].r = data[i].r / LEN;
	data[i].i = data[i].i / LEN;
	if ((data[i].r*data[i].r + data[i].i*data[i].i) > 0.01) {
	    printf("%3d %8.2f %8.2f\n",i,data[i].r,data[i].i);
	    }
	}

    for (i=0; i<LEN; i++) data[i].i = -data[i].i;

    cfdift(data,LEN,LOGLEN);
    cbitrev(data,LEN);

    for (i=0; i<LEN; i++) data[i].i = -data[i].i;

    sum = 0.0;
    for (i=0; i<LEN; i++) {

#ifdef IAG
fprintf(stderr,"%3d: (%7g,%7g) (%7g,%7g)\n",i,odata[i].r,odata[i].i,data[i].r,data[i].i);
#endif
	error.r = data[i].r - odata[i].r;
	error.i = data[i].i - odata[i].i;
	sum = sum + error.r * error.r + error.i * error.i;
	}
    printf("\nTotal square error after forward + inverse = %8g\n",sum);
    }
#endif
