/* DSP routines from Mike Brookes, Imperial College London */

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"
#include "libic.h"

#define NAME "makechirp"

void makechirp(len,halftlen,table)
int len,*halftlen;
COMPLEX **table;
/*
Creates the chirp functions

This subroutine return *halftlen = power of 2 >= len
and the following COMPLEX arrays:

       chirp[len]		Chirp funtion: exp(-k*k*jPI/len)
       Tchirp[halftlen+1]	Half of symmetric transform of chirp function
       tdata[2*halftlen]	Array for storing transform in

The address of the chirp array is returned, the others may be obtained
by addition.
The full chirp function has dimension [2*halftlen] but is symmetrical
and all but the first [len] of the first half are zero. Within
the first [len], the values are symmetric if len is even (i.e.
chirp[len-i] = chirp[i]) or antisymmetric if len is odd. This fact
could be used to reduce computation (but isn't).
The transform of the chirp function is symmetrical.
The transform of the conjugate chirp function is the conjugate of the
chirp transform.

This program would probably work just as fast using a table of sines
rather than the output from exptable. This would save on storage
*/
{
    static COMPLEX *chirp;
    static COMPLEX czero = {0.0,0.0};
    static int olen=0,htlen,maxlen=0;

    COMPLEX *exptab,*Tchirp;
    register COMPLEX *ptab,*pchirp,*pTchirp,*qTchirp;
    int inc,tlen,twicelen;

#ifdef IAG
    int i;
#endif
/* Allocate temporary memory if necessary */

    if (len != olen) {
	tlen = 4;
	while (tlen<len) tlen *= 2;
	htlen = tlen;
	tlen *=2;		/* tlen = power of 2 > 2*len */
	if (len > maxlen) {
	    if (olen)  free(chirp);
	    chirp = (COMPLEX *) malloc((htlen+1+tlen+len)*sizeof(COMPLEX));
	    maxlen = len;
	    }
	if (!chirp) error("%s: Cannot allocate dynamic memory",NAME);
	olen = len;
	exptable(len,&exptab);
#ifdef IAG
fprintf(stderr,"Exp table\n");
for (i=0; i<=(len+len); i++) fprintf(stderr,"%4d (%g,%g)\n",i,exptab[i].r,exptab[i].i);
#endif
	Tchirp = chirp+len;


/* Create the Chirp Function */

	pTchirp = Tchirp;
	qTchirp = Tchirp + tlen -1;
	pchirp = chirp;
	ptab = exptab;
	*pchirp++ = *ptab;
	*pTchirp++ = *ptab--;	/* W**0 */
	inc = 1;
	twicelen = len << 1;
	while (pchirp < Tchirp) {
	    if (ptab < exptab) ptab += twicelen;
#ifdef IAG
fprintf(stderr,"chirp[%d] = Tchirp[%d] = Tchirp[%d] = exptab[%d] = (%g,%g)\n",
pchirp-chirp,pTchirp-Tchirp,qTchirp-Tchirp,ptab-exptab,ptab->r,ptab->i);
#endif
	    *pchirp++ = *ptab;
	    *pTchirp++ = *ptab;
	    *qTchirp-- = *ptab;
	    ptab -= ( inc += 2);
	    }
	while(pTchirp <= qTchirp) *pTchirp++ = czero;
	fftc(Tchirp,Tchirp,tlen);
	}
    *halftlen = htlen;
    *table = chirp;
    return;
    }
#ifdef EMO

#include <stdio.h>
#include <math.h>

main ()
{
    COMPLEX *chirp,*Tchirp;
    int len,i,halftlen;

    for(;;) {
	do {
	    printf("\nEnter length : ");
	    scanf("%d",&len);
	    } while (len <2);
	makechirp(len,&halftlen,&chirp);
	Tchirp = chirp + len;

	printf("Chirp Function:\n\n");
	for (i=0; i<len; i++) {
	    printf ("%8d %10g %10g\n",i,chirp[i].r,chirp[i].i);
	    }

	printf("Transform:\n\n");
	for (i=0; i<=halftlen; i++) {
	    printf ("%8d %10g %10g\n",i,Tchirp[i].r,Tchirp[i].i);
	    }
	}
    }
#endif
