/* anmap -- map between sets of annotations via mapping file */

/* M.A.Huckvale - February 1989 */

/* version 1.0 */

/* version 1.1 - April 1989
	- copy across old 'type' if present
*/

/* version 1.2 - September 1992
	- add 'boundary' marking option
*/
/* version 1.3 - July 1994
	- add 'compact multiple' option
*/

#define PROGNAME "anmap"
#define PROGVERS "1.3"
char *progname=PROGNAME;

/*-------------------------------------------------------------------*/
/**MAN
.TH ANMAP 1 UCL SFS
.SH NAME
anmap -- map between sets of annotations using mapping file
.SH SYNOPSIS
.B anmap
(-i item) (-r samplingrate) (-b boundarylength) (-t newtype) (-m) mapfile file
.br
.B anmap
(-i item) (-b boundarylength) -o file
.SH DESCRIPTION
.I anmap
maps one set of annotations onto another, using a mapping file to
substitute annotation labels.  Labels not in the mapping file are
copied unchanged, but reported as errors.  The sampling rate of the 
annotation set can also be changed by an option.  If executed with the
"-o" flag, a mapping file is not required, and instead an ordered list of 
annotations is produced from the input AN item.  The boundary marking option
allows additional annotations to be created of a fixed size at each boundary
point between existing annotations.
.PP
.I Options:
.TP 11
.B -I
Identify the program name and version.
.TP 11
.BI -i item
Select input item number.
.TP 11
.BI -r "sampling rate"
Select the sampling rate for the output annotation item.  
Default is sampling rate of input annotation item.
.TP 11
.BI -b boundarylength
Creates additional annotations at the junctions between existing annotations.
The width of the juncture region is specified by the 'boundarylength' parameter
in seconds.  For each junction, two additional annotations are added: the
left boundary: <label1>(<label2>) and the right boundary (<label1>)<label2>.
Each of these will be half the juncture length specified.
.TP 11
.BI -t newtype
Supply a new type= string for the output annotation item history.  Default is the
input item type.
.TP 11
.B -m
Compact multiple annotations of a given output label to a single annotation.
.TP 11
.B -o
Select annotation output mode.  A mapping file must not be given in this
mode.  Prints to standard output an ordered list of annotations from
the input AN item.
.SH MAPPING FILE FORMAT
Each line in the mapping file must have the following format:
.nf
      <old annotation> <new annotation>
.fi
where <old annotation> is the name of an annotation to be converted, and 
<new annotation> if the new annotation.  If <new annotation> is blank
then the input annotation will be deleted.  For example
.nf
     ee     i
     ih     I
     $		
     r      r
.fi
.SH VERSION/AUTHOR
1.2 - Mark Huckvale
*/
/*---------------------------------------------------------------*/

/* include files */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <malloc.h>
#include <math.h>
#include "sfs.h"

/* global data */
struct item_header	anitem;		/* input annotation item */
struct an_rec		*an;
struct an_rec		compan[3];
char			complabel[3][256];
struct item_header	opitem;		/* output annotation item */
struct an_rec		*oan;
char			mapname[80];
char			filename[80];
double			srate = 0.0;
int			anoutput=0;
int			doboundary=0;
int			docompact=0;
double			junclen=0.0;
char			newtype[256];

/* list of mappings */
char	ilab[256],olab[256];		/* temporary storage */
char	**anin;				/* mapping input */
char	**anout;			/* mapping output */
int	nummap=0;

/* list of errors */
#define NUMERROR	500
char	*anerr[NUMERROR];
int	numerr=0;
		 
/* map file line */
char	mline[256];

/* save string in dynamic memory */
char	*strsave(s)
char	*s;
{
	char	*p;
	int	len=strlen(s)+1;
	if ((p=malloc(len))==NULL)
		error("cannot save string",NULL);
	strcpy(p,s);
	return(p);
}

/* binary string search */
int strfind(s,t,num)
char	*s;
char	*t[];
int	num;
{
	int	i,j,k;
	int	c;
	
	if (num==0)
		return(-1);
	else {
		i=0;
		j=num-1;
		do {
			k=(i+j)/2;
			if ((c=strcmp(s,t[k])) > 0)
				i=k+1;
			else
				j=k-1;
		} while (c && (i <= j));
		if (c)
			return(-1);
		else
			return(k);
	}
}

/* maintain sring table */
void strtable(s,t,num)
char	*s;
char	*t[];
int	*num;
{
	int	i;
	
	/* see if string in table */
	if (strfind(s,t,*num) < 0) {
		/* add to table */
		i = *num;
		while ((i>0) && (strcmp(s,t[i-1])<0)) {
			t[i] = t[i-1];
			i--;
		}
		t[i]=strsave(s);
		(*num)++;
	}
}

/* get entry from map file */
int	getmapentry(ip,ilab,olab)
FILE	*ip;
char	*ilab;
char	*olab;
{
	char	*p;

	if (fgets(mline,256,ip)) {
		/* get input */
		p=strtok(mline," \t\n");
		if (p && *p)
			strcpy(ilab,p);
		else
			*ilab='\0';
		/* get output */
		p=strtok(NULL," \t\n");
		if (p && *p)
			strcpy(olab,p);
		else
			*olab='\0';
		return(1);
	}
	return(0);
}


/* return boundary components of an annotation */
int getleft(comp,an)
struct an_rec comp[];
struct an_rec *an;
{
	double	andur;
	double 	cuttime=junclen;

	/* check is a left annotation */
	if (comp[0].label[0]=='\0') return(0);

	/* check centre is long enough */
	andur = comp[1].size * anitem.frameduration;
	if (comp[2].label[0] && (andur < 2*junclen))
		cuttime = andur/2;
	else if (!comp[2].label[0] && (andur < junclen))
		cuttime = andur;
	else
		cuttime = junclen;

	/* build left annotation */
	an->posn = comp[1].posn;
	an->size = (int)(0.5+cuttime/anitem.frameduration);
	sprintf(an->label,"(%s)%s",comp[0].label,comp[1].label);
	return(1);
}
int getmiddle(comp,an)
struct an_rec comp[];
struct an_rec *an;
{
	double	andur;
	double	cut1,cut2;
	
	/* check centre is long enough */
	andur = comp[1].size * anitem.frameduration;
	cut1 = (comp[0].label[0]) ? junclen : 0.0;
	cut2 = (comp[2].label[0]) ? andur - junclen - cut1 : andur - cut1;
	if (cut2 <= cut1) return(0);
	
	/* build middle annotation */
	an->posn = comp[1].posn + (int)(0.5+cut1/anitem.frameduration);
	an->size = (int)(0.5+(cut2-cut1)/anitem.frameduration);
	strcpy(an->label,comp[1].label);
	return(1);
}
int getright(comp,an)
struct an_rec comp[];
struct an_rec *an;
{
	double	andur;
	double 	cuttime=junclen;

	/* check is a right annotation */
	if (comp[2].label[0]=='\0') return(0);

	/* check centre is long enough */
	andur = comp[1].size * anitem.frameduration;
	if (comp[0].label[0] && (andur < 2*junclen))
		cuttime = andur/2;
	else if (!comp[0].label[0] && (andur < junclen))
		cuttime = 0.0;
	else
		cuttime = andur - junclen;

	/* build left annotation */
	an->posn = comp[1].posn + (int)(0.5+cuttime/anitem.frameduration);
	an->size = (int)(0.5+(andur-cuttime)/anitem.frameduration);
	sprintf(an->label,"%s(%s)",comp[1].label,comp[2].label);
	return(1);
}

/* move an annotation */
void movean(dan,san)
struct an_rec *dan;
struct an_rec *san;
{
	dan->posn = san->posn;
	dan->size = san->size;
	strcpy(dan->label,san->label);
}

/* map an annotation */
void mapan(ilab,olab)
char	*ilab;
char	*olab;
{
	int	idx;

	idx=strfind(ilab,anin,nummap);
	if (idx < 0) {
		/* annotation missing */
		strtable(ilab,anerr,&numerr);
		if (numerr >= NUMERROR)
			error("too many annotations not in '%s'",mapname);
		/* rewrite existing label */
		strcpy(olab,ilab);
	}
	else
		/* rewrite new label */
		strcpy(olab,anout[idx]);
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */

	/* processing variables */
	int		fid,ofid;
	FILE		*ip;
	int32		it;
	char		*ty;
	char		*antype="0";
	int		i,idx;
	char		*params();
	double		atof();
	char		messg[32];
	char		newlabel[256];

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:r:ob:t:m")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Annotation mapping V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* item number */
			if (itspec(optarg,&it,&ty)==0) {
				if (it==AN_TYPE)
					antype=ty;
				else
					error("bad item specification",NULL);
			}
			else
				error("illegal item specification",NULL);
			break;
		case 'r' :	/* sampling rate */
			srate = atof(optarg);
			break;
		case 'b' :	/* boundary mode */
			doboundary++;
			junclen = 0.5*atof(optarg);
			break;
		case 't' :	/* output type */
			strcpy(newtype,optarg);
			break;
		case 'o' :	/* output mode */
			anoutput++;
			break;
		case 'm' :	/* compact multiple */
			docompact++;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	/* check command line */
	if (errflg || (argc<3))
		error("usage: %s (-I) (-i item) (-r samprate) (-b boundarylen) (-t newtype) (-m) -o|mapname file",PROGNAME);

	/* get map filename */
	if (!anoutput) {
		if (optind < argc)
			strcpy(mapname,argv[optind]);
		else
			error("no annotation mapfile specified",NULL);
		optind++;
	}
	
	/* get data filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open data file */
	if ((fid=sfsopen(filename,(anoutput)?"r":"w",NULL)) < 0)
		error("access error on '%s'",filename);

	/* find input annotation item */
	if (!sfsitem(fid,AN_TYPE,antype,&anitem))
		error("cannot find input AN item in '%s'",filename);

	/* get output sampling duration */
	if (srate <= 0) 
		srate = anitem.frameduration;
	else
		srate = 1.0 / srate;
		
	/* get annotation buffer */
	if ((an=(struct an_rec *)malloc(sizeof(struct an_rec) + 256))==NULL)
		error("could not get annotation buffer");
	compan[0].label = &complabel[0][0];
	compan[1].label = &complabel[1][0];
	compan[2].label = &complabel[2][0];
	
	/* output mode */
	if (anoutput) {
		numerr=0;
		if (doboundary) {
			compan[0].label[0] = '\0';
			sfsread(fid,0,1,an);
			movean(&compan[1],an);
			for (i=1;sfsread(fid,i,1,an);i++) {
				movean(&compan[2],an);
				if (getleft(compan,an)) {
					strtable(an->label,anerr,&numerr);
					if (numerr >= NUMERROR)
						error("too many different annotations",NULL);
				}
				if (getmiddle(compan,an)) {
					strtable(an->label,anerr,&numerr);
					if (numerr >= NUMERROR)
						error("too many different annotations",NULL);
				}
				if (getright(compan,an)) {
					strtable(an->label,anerr,&numerr);
					if (numerr >= NUMERROR)
						error("too many different annotations",NULL);
				}
				movean(&compan[0],&compan[1]);
				movean(&compan[1],&compan[2]);
			}
			compan[2].label[0]='\0';
			if (getleft(compan,an)) {
				strtable(an->label,anerr,&numerr);
				if (numerr >= NUMERROR)
					error("too many different annotations",NULL);
			}
			if (getmiddle(compan,an)) {
				strtable(an->label,anerr,&numerr);
				if (numerr >= NUMERROR)
					error("too many different annotations",NULL);
			}
		}
		else {
			for (i=0;sfsread(fid,i,1,an);i++) {
				strtable(an->label,anerr,&numerr);
				if (numerr >= NUMERROR)
					error("too many different annotations",NULL);
			}
		}
		for (i=0;i<numerr;i++) printf("%s\n",anerr[i]);
		sfsclose(fid);
		exit(0);		
	}

	/* open map file */
	if ((ip=fopen(mapname,"r"))==NULL)
		error("cannot open map file '%s'",mapname);

	/* get number of annotations from map */
	nummap=0;
	while (getmapentry(ip,ilab,olab)) nummap++;

	/* get space for annotation table */
	if ((anin=(char **)calloc(nummap,sizeof(char *)))==NULL)
		error("cannot get memory for annotation table",NULL);
	if ((anout=(char **)calloc(nummap,sizeof(char *)))==NULL)
		error("cannot get memory for annotation table",NULL);
	
	/* get list of input annotations from map */
	nummap=0;
	rewind(ip);
	while (getmapentry(ip,ilab,olab))
		strtable(ilab,anin,&nummap);

	/* save output annotations */
	rewind(ip);
	while (getmapentry(ip,ilab,olab)) {
		idx=strfind(ilab,anin,nummap);
		anout[idx]=strsave(olab);
	}
	if (ttytest())
		fprintf(stderr,"%d annotation mappings loaded from '%s'\n",nummap,mapname);
	
	/* create output item */
	sfsheader(&opitem,AN_TYPE,-1,1,-1,srate,anitem.offset,0,0,1);
	sprintf(messg,",boundary=%g",junclen*2);
	if (!*newtype)
		strcpy(newtype,params(anitem.history,"type",""));
	if (!*newtype)
		sprintf(opitem.history,"%s(%d.%02d;map=%s,rate=%g%s)",
			PROGNAME,
			anitem.datatype,anitem.subtype,
			mapname,1.0/srate,
			(doboundary)?messg:""
			);
	else
		sprintf(opitem.history,"%s(%d.%02d;map=%s,rate=%g%s,type=%s)",
			PROGNAME,
			anitem.datatype,anitem.subtype,
			mapname,1.0/srate,
			(doboundary)?messg:"",
			newtype);

	/* open output channel */
	if ((ofid=sfschannel(filename,&opitem)) < 0)
		error("cannot open output channel to '%s'",filename);
		
	/* get output annotation buffer */
	if ((oan=(struct an_rec *)sfsbuffer(&opitem,1))==NULL)
		error("could not get annotation buffer",NULL);

	/* process annotations in sequence */
	if (doboundary) {
		compan[0].label[0] = '\0';
		sfsread(fid,0,1,an);
		movean(&compan[1],an);
		for (i=1;sfsread(fid,i,1,an);i++) {
			movean(&compan[2],an);
			if (getleft(compan,an)) {
				/* find annotation in map */
				mapan(an->label,oan->label);
				/* recalculate time */
				oan->posn = (int)(0.5+(an->posn*anitem.frameduration)/srate);
				oan->size = (int)(0.5+(an->size*anitem.frameduration)/srate);
				/* write non-NULL labels */
				if (oan->label[0]) sfswrite(ofid,1,oan);
			}
			if (getmiddle(compan,an)) {
				/* find annotation in map */
				mapan(an->label,oan->label);
				/* recalculate time */
				oan->posn = (int)(0.5+(an->posn*anitem.frameduration)/srate);
				oan->size = (int)(0.5+(an->size*anitem.frameduration)/srate);
				/* write non-NULL labels */
				if (oan->label[0]) sfswrite(ofid,1,oan);
			}
			if (getright(compan,an)) {
				/* find annotation in map */
				mapan(an->label,oan->label);
				/* recalculate time */
				oan->posn = (int)(0.5+(an->posn*anitem.frameduration)/srate);
				oan->size = (int)(0.5+(an->size*anitem.frameduration)/srate);
				/* write non-NULL labels */
				if (oan->label[0]) sfswrite(ofid,1,oan);
			}
			movean(&compan[0],&compan[1]);
			movean(&compan[1],&compan[2]);
		}
		compan[2].label[0]='\0';
		if (getleft(compan,an)) {
				/* find annotation in map */
				mapan(an->label,oan->label);
				/* recalculate time */
				oan->posn = (int)(0.5+(an->posn*anitem.frameduration)/srate);
				oan->size = (int)(0.5+(an->size*anitem.frameduration)/srate);
				/* write non-NULL labels */
				if (oan->label[0]) sfswrite(ofid,1,oan);
		}
		if (getmiddle(compan,an)) {
				/* find annotation in map */
				mapan(an->label,oan->label);
				/* recalculate time */
				oan->posn = (int)(0.5+(an->posn*anitem.frameduration)/srate);
				oan->size = (int)(0.5+(an->size*anitem.frameduration)/srate);
				/* write non-NULL labels */
				if (oan->label[0]) sfswrite(ofid,1,oan);
		}
	}
	else if (docompact) {
		sfsread(fid,0,1,an);
		/* recalculate time */
		oan->posn = (int)(0.5+(an->posn*anitem.frameduration)/srate);
		oan->size = (int)(0.5+(an->size*anitem.frameduration)/srate);
		mapan(an->label,oan->label);
		for (i=1;sfsread(fid,i,1,an);i++) {
			/* find annotation in map */
			mapan(an->label,newlabel);
			if (strcmp(oan->label,newlabel)==0) {
				/* extend current */
				oan->size = (int)(0.5+((an->posn+an->size)*anitem.frameduration)/srate) - oan->posn;
			}
			else {
				/* write non-NULL labels */
				if (oan->label[0]) sfswrite(ofid,1,oan);
				oan->posn = (int)(0.5+(an->posn*anitem.frameduration)/srate);
				oan->size = (int)(0.5+(an->size*anitem.frameduration)/srate);
				strcpy(oan->label,newlabel);
			}
		}
		/* write last */
		if (oan->label[0]) sfswrite(ofid,1,oan);
	}
	else for (i=0;sfsread(fid,i,1,an);i++) {
		/* find annotation in map */
		mapan(an->label,oan->label);
		/* recalculate time */
		oan->posn = (int)(0.5+(an->posn*anitem.frameduration)/srate);
		oan->size = (int)(0.5+(an->size*anitem.frameduration)/srate);
		/* write non-NULL labels */
		if (oan->label[0]) sfswrite(ofid,1,oan);

	}
	
	/* report any errors */
	if (numerr) {
		fprintf(stderr,"%s: the following annotations could not be processed:\n",PROGNAME);
		for (i=0;i<numerr;i++) fprintf(stderr,"%s\n",anerr[i]);
	}

	/* that's all folks */
	sfsclose(fid);
	if (!sfsupdate(filename))
		error("update error on '%s'",filename);
	exit(0);
}


