/* fmlist -- list FM item in seconds, Hertz and dB */

/* m.a.huckvale - january 1987 */

/* version 1.1s - November 1987
	- SFS version
*/

#define PROGNAME "fmlist"
#define PROGVERS "1.1s"
char *progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH FMLIST SFS1 UCL
.SH NAME
fmlist - list raw formant estimates item
.SH SYNOPSIS
.B fmlist
(-i item) (-a) (-f numformants) file
.SH DESCRIPTION
.I fmlist
lists a FM item in seconds, Hertz and and dB to the standard output.  
An option allows the inclusion of markers from an annotation item.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version.
.TP 11
.BI -i item
Select input item number.
.TP 11
.B -a
Delimit segments in output using annotations.
.TP 11
.BI -f numformants
Select number of formants to list, default is 5.
.SH INPUT ITEMS
.IP 12.xx 11
Any FM item.
.IP 5.xx 11
Optional annotation item
.SH VERSION/AUTHOR
1.1 - Mark Huckvale
.SH SEE ALSO
sylist, fmload
*/
/*--------------------------------------------------------------------------*/

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "sfs.h"		/* database filing system structures */

/* global data */
struct item_header fmitem;	/* filing system item header for fm */
struct item_header anitem;	/* item header for An item */
int	nformant = 5;

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		annreq=0;	/* annotation not required */
	char		*antype="0";	/* use first annotation item */
	char		*fmtype="0";	/* default FM sub-type = last */
	int		c;		/* option switch */
	int32		it;		/* item/sub-type specifiers */
	char		*ty;
	/* file variables */
	char		filename[SFSMAXFILENAME];	/* dbase file name */
	int		fid,afid=0;
	/* processing variables */
	struct fm_rec	*fmrec;		/* input formant record */
	int		i,j;
	double		time = 0.0;	/* time of current frame */
	struct an_rec   *anrec=NULL;	/* annotation record */
	double		antime=0;	/* time of annotation */
	int		anidx=0;	/* annotation index */

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:af:")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: List formant estimates V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'a' :	/* annotation required */
			annreq++;
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == FM_TYPE)
					fmtype = ty;
				else if (it == AN_TYPE) {
					antype = ty;
					annreq++;
				}
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'f' :	/* number of listed formants */
			nformant = atoi(optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	/* check for option decoding error */
	if (errflg || (argc<2))
		error("usage: %s (-I) (-a) (-i item) (-f numformants) dbase_file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open file */
	if ((fid=sfsopen(filename,"r",NULL)) < 0)
		error("access error on %s",filename);

	/* locate FM item */
	if (!sfsitem(fid,FM_TYPE,fmtype,&fmitem))
		error("unable to find input FM item in %s",filename);
	if ((fmrec=(struct fm_rec *)sfsbuffer(&fmitem,1))==NULL)
		error("unable to create memory buffer",NULL);

	/* load annotation item if required */
	if (annreq) {
		afid=sfsopen(filename,"r",NULL);
		if (!sfsitem(afid,AN_TYPE,antype,&anitem))
			error("unable to find input AN item in %s",filename);
		if ((anrec=(struct an_rec *)sfsbuffer(&anitem,1))==NULL)
			error("unable to create memory buffer",NULL);
	}

	/* list data */
	printf("! item=%d.%02d file=%s\n",fmitem.datatype,fmitem.subtype,filename);
	printf("! posn size v   gain;");
	for (j=1;j<=nformant;j++) printf("   F%d   B%d A%d;",j,j,j);
	printf("\n!\n");
	if (annreq) {
		if (sfsread(afid,0,1,anrec) != 1)
			antime=10E6;
		else
			antime = anitem.offset + anrec->posn * anitem.frameduration;
		anidx=1;
	}
	for (i=0;i<fmitem.numframes;i++) {
		sfsread(fid,i,1,fmrec);
		time = fmitem.offset + fmrec->posn * fmitem.frameduration;
		while ((annreq) && (antime < time)) {
			printf("%s\n",anrec->label);
			if (sfsread(afid,anidx++,1,anrec) != 1)
				antime=10E6;
			else
				antime = anitem.offset + anrec->posn * anitem.frameduration;
		}
		printf("%6.1f ",time*1000);
		printf("%4.1f ",fmrec->size*fmitem.frameduration*1000);
		printf("%1d ",  fmrec->flag);
		printf("%6.1f; ",fmrec->gain);
		for (j=0;(j<fmrec->npeaks) && (j<nformant);j++) 
			printf("%4d %4d %2d; ",
				(int)fmrec->formant[j].freq,
				(int)fmrec->formant[j].band,
				(fmrec->formant[j].amp > 0.0) ? (int)fmrec->formant[j].amp : 0);
		for (j=fmrec->npeaks;j<nformant;j++)
			printf("%4d %4d %2d; ",0,0,0);
		printf("\n");
	}

	/* and exit */
	exit(0);
}

