/* record -- simple record into SFS file */

/* M.A.Huckvale - University Colleg London */

/* version 1.0 - October 1995 */

#define PROGNAME "record"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH RECORD 1 UCL 
.SH NAME
record - simple recording into Speech and/or Lx item
.SH SYNOPSIS
.B record
(-I) (-f sampfreq) (-2|-l|-L) (-t time) (-e) file
.SH DESCRIPTION
.I record
is a program to acquire speech and/or lx directly into an SFS file.
It is based on the generic SFSADC routines.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -f sampfreq
Select input sampling frequency.  Default 20000Hz.
.TP 11
.BI -t time
Specify maximum duration of input.  Default 5 seconds.
.TP 11
.B -e
Automatic end-pointing (single speech channel only).
.TP 11
.B -2
Stereo recording into 2 Speech items.
.TP 11
.B -l
Simultaneous Speech and Lx recording into Speech(right) and Lx(left) items.
.TP 11
.B -L
Single channel recording into Lx item.
.SH OUTPUT ITEMS
.IP SP
Recorded speech
.IP LX
Recorded Lx
.SH HISTORY
.IP freq
Sampling rate.
.SH ENVIRONMENT
.IP ADC
This environment variable selects input device.
.SH VERSION/AUTHOR
.IP 1.0
Mark Huckvale
.SH SEE ALSO
sunin
.SH BUGS
*/
/*--------------------------------------------------------------------------*/

/* include files */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <malloc.h>
#include "sfs.h"

/* ADC defines */
extern double adc_selected_rate;

/* global constants */
#define DEF_SRATE	20000.0
#define DEF_TIME	5.0
#define MAXTIME		600.0
#define XFERSIZE	4096

/* global data */
struct item_header spitem;
struct item_header spitem2;
struct item_header lxitem;
short	*sbuf;			/* signal buffer */
short	tbuf1[XFERSIZE/2];	/* de-multiplexing buffers */
short	tbuf2[XFERSIZE/2];

/* options */
int	nchan=1;			/* number of channels */
double	srate=DEF_SRATE;		/* sampling rate */
double	atime=DEF_TIME;			/* acquire time */
int	dolx=0;				/* Lx output */
int	endpoint=0;			/* do endpointing */

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */

	/* file variables */
	char		filename[SFSMAXFILENAME]; /* SFS data file name */
	int		fid;		/* input file descriptor */
	int		sfid,lfid=0;	/* output file descriptors */
	register int	i;
	register short	*ptr;
	long		msamp,nsamp;
	char		ans[80];
	int		cnt;
	
	/* decode switches */
	while ( (c = getopt(argc,argv,"If:t:2lLe")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: SFS signal record V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case '2' :	/* acquire stereo Sp */
			nchan = 2;
			dolx = 0;
			break;
		case 'l' :	/* acquire sp & lx */
			nchan = 2;
			dolx = 1;
			break;
		case 'L' :	/* acquire lx only */
			nchan = 1;
			dolx = 1;
			break;
		case 'f' :	/* set sample rate */
			srate = atoi(optarg);
			if ((srate < 8000) || (srate > 48000))
				error("sampling rate out of range");
			break;
		case 't' :	/* set acquisition time */
			atime = atof(optarg);
			if ((atime <= 0.001) || (atime >= MAXTIME))
				error("invalid acquisition time");
			break;
		case 'e' :	/* endpoint */
			endpoint=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-2|-l|-L) (-f sfreq) (-t time) (-e) file",PROGNAME);

	if (endpoint && (dolx || (nchan > 1)))
		error("endpointing only on single channel speech input");

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* check file suitable */
	if ((fid=sfsopen(filename,"w",NULL)) < 0)
		error("access error on '%s'",filename);
	else
		sfsclose(fid);

	/* get a memory buffer */
	msamp = (int)(atime * srate * nchan);
	if ((sbuf=(short *)calloc(msamp,sizeof(short)))==NULL)
		error("could not get memory buffer");

	/* OK, get a key press */
	printf("Acquisition of %s%.1f seconds of %d channels at %g samples/sec\n",
		(endpoint)?"up to ":"",atime,nchan,srate);
	if (endpoint)
		printf("Speak to start.  ");
	else {
		printf("Press RETURN to start ... (EOF to exit)");
		fflush(stdout);
		if (!fgets(ans,80,stdin)) exit(1);
	}
#ifdef DOS
	printf("Press ESCAPE to stop ...");
#else
	printf("Press RETURN or CTRL/C to stop ...");
#endif
	fflush(stdout);
	
	/* do it */
	if (adc_open(NULL)<0) {
		free(sbuf);
		error("could not open ADC device");
	}
	if ((nsamp = adc_record(sbuf,msamp,srate,nchan,endpoint))<=0) {
		free(sbuf);
		error("acquisition failed");
	}
	srate = adc_selected_rate;
	adc_close(0);
	printf("\nAcquired %.1f seconds\n",nsamp/(srate*nchan));

	/* open output channels to SFS file */
	if (nchan==1) {
		sfsheader(&spitem,(dolx)?LX_TYPE:SP_TYPE,0,2,1,1.0/(double)srate,0.0,1,0,0);
		sprintf(spitem.history,"%s(sfreq=%g)",PROGNAME,srate);
		sprintf(spitem.params,"bits=16");
		if ((sfid = sfschannel(filename,&spitem)) < 0)
			error("could not open output channel to '%s'",filename);
	}
	else {
		sfsheader(&spitem,SP_TYPE,0,2,1,1.0/(double)srate,0.0,1,0,0);
		sprintf(spitem.history,"%s(sfreq=%g,chan=1)",PROGNAME,srate);
		sprintf(spitem.params,"bits=16");
		if ((sfid = sfschannel(filename,&spitem)) < 0)
			error("could not open output channel to '%s'",filename);
		sfsheader(&lxitem,(dolx)?LX_TYPE:SP_TYPE,0,2,1,1.0/(double)srate,0.0,1,0,0);
		sprintf(lxitem.history,"%s(sfreq=%g,chan=2)",PROGNAME,srate);
		sprintf(lxitem.params,"bits=16");
		if ((lfid = sfschannel(filename,&lxitem)) < 0)
			error("could not open output channel to '%s'",filename);
	}
	
	/* write out data */
	ptr = sbuf;
	while (nsamp > 0) {
		cnt = (nsamp < XFERSIZE) ? nsamp : XFERSIZE;
		if (nchan==1) {
			if (sfswrite(sfid,cnt,ptr)!=cnt) {
				free(sbuf);
				error("write error on output");
			}
		}
		else {
			for (i=0;i<cnt/2;i++) {
				tbuf1[i] = ptr[2*i];
				tbuf2[i] = ptr[2*i+1];
			}
			if (sfswrite(sfid,cnt/2,tbuf1)!=cnt/2) {
				free(sbuf);
				error("write error on output");
			}
			if (sfswrite(lfid,cnt/2,tbuf2)!=cnt/2) {
				free(sbuf);
				error("write error on output");
			}
		}
		ptr += cnt;
		nsamp -= cnt;
	}
	free(sbuf);

	/* update SFS file */
	if (!sfsupdate(filename))
		error("update error on '%s'",filename);

	exit(0);
}

