/* bbc -- DIG routines for BBC microcomputer with GTERM emulator */

/* Mark Huckvale - April 1990 */

#include "SFSCONFG.h"
#ifdef EVb

#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <fcntl.h>
#include "dig.h"
#include "digdata.h"

/* bbc control codes */
#define	BBCCLEAREOL	1
#define	BBCMOUSEON	2
#define BBCMOUSEOFF	3
#define BBCCURSON	4
#define BBCCURSOFF	5
#define	BBCCLEAR	16
#define BBCDRAWMODE	18
#define BBCMODE		22
#define	BBCPLOT		25
#define BBCRESTORE	26
#define	BBCMOVETO	4
#define BBCDRAWTO	5
#define BBCDASHTO	21
#define	BBCHOME		30
#define BBCPOINT	69
#define BBCFILLTRI	85
#define BBCFILLBOX	101

/* bbc code transmission for Cambridge Computer Labs GTERM chip */
/* sendbbc(x) unsigned char x; */
#define sendbbc(x) if ((x)<16) putc(0x60+(x),digdata.outfile); else if ((x) >= 0x60) { putc(0x70 + (((x) >> 4) & 0xF),digdata.outfile); putc(0x60 + ((x) & 0xF),digdata.outfile);} else putc(x,digdata.outfile);

#ifdef EVb
struct {
	short subtype;		/* 0=BBC, 1=BBCMaster */
	short mode;		/* 0=?, 1=initialised */
	short rop;		/* raster operation */
} digDEVb;
#endif

/* put x,y coordinates to bbc */
static digbbccoord(x,y)
int	x,y;		/* pixel co-ordinates */
{
	int	xhi,xlo,yhi,ylo;

	xhi = x >> 7;
	xlo = (x & 0x7F) << 1;
	yhi = y >> 6;
	ylo = (y & 0x3F) << 2;
	sendbbc(xlo);
	sendbbc(xhi);
	sendbbc(ylo);
	sendbbc(yhi);
}

/* device characteristics */
bbc_params(subtype,digdev)
int	subtype;
struct digdevice *digdev;
{
	digdev->nhoriz = 640;
	digdev->nvert = 256;
	digdev->aspect = 2.0;
	digdev->chwidth = 8;
	digdev->chheight = 8;
	digdev->greylevels = 5;
	digdev->greypixwidth = 2;
	digdev->greypixheight = 2;
	digDEVb.subtype = subtype;
}

/* open graphics */
bbc_open()
{
	if (!digDEVb.mode) {
		putc(BBCMODE,digdata.outfile);
		putc(0x60,digdata.outfile);
		putc(BBCRESTORE,digdata.outfile);
		fflush(digdata.outfile);
		digDEVb.mode++;
	}
}

/* interrupt graphics */
bbc_inter()
{
	fprintf(stderr,"Interrupt\n");
	fflush(stderr);
}

/* close graphics - return to text mode */
bbc_close()
{
	putc(4,digdata.outfile);
	putc(30,digdata.outfile);
	putc(BBCMODE,digdata.outfile);
	putc(0x63,digdata.outfile);
	fflush(digdata.outfile);
	digDEVb.mode = 0;
}

/* clear screen */
bbc_clear()
{
	putc(BBCCLEAR,digdata.outfile);
	fflush(digdata.outfile);
}

/* draw a poly-line */
bbc_polyline(bundle,buff,len)
int	bundle;
short	*buff;
int	len;
{
	int	x,y;

	/* sort out colours */
	if (!bundle) {
		putc(BBCDRAWMODE,digdata.outfile);
		putc(0x60,digdata.outfile);
		putc(0x60,digdata.outfile);
	}

	/* position to first point */
	x = *buff++;
	y = *buff++;
	putc(BBCPLOT,digdata.outfile);
	putc(BBCMOVETO,digdata.outfile);
	digbbccoord(x,y);
	len -= 2;

	/* plot rest of points */
	while (len > 0) {
		x = *buff++;
		y = *buff++;
		putc(BBCPLOT,digdata.outfile);
		if (bundle < 35)
			putc(BBCDRAWTO,digdata.outfile);
		else
			putc(BBCDASHTO,digdata.outfile);
		digbbccoord(x,y);
		len -= 2;
	}
	if (!bundle) {
		putc(BBCDRAWMODE,digdata.outfile);
		putc(0x60,digdata.outfile);
		putc(1,digdata.outfile);
	}

}

/* draw 'hardware' text */
bbc_text(bundle,x,y,str)
int	bundle;
int	x,y;
char	*str;
{

	/* sort out colours */
	putc(BBCCURSOFF,digdata.outfile);
	if (!bundle) {
		putc(BBCDRAWMODE,digdata.outfile);
		putc(0x60,digdata.outfile);
		putc(0x60,digdata.outfile);
	}
	/* write string */
	putc(BBCPLOT,digdata.outfile);
	putc(BBCMOVETO,digdata.outfile);
	digbbccoord(x,y+7);
	fputs(str,digdata.outfile);
	if (!bundle) {
		putc(BBCDRAWMODE,digdata.outfile);
		putc(0x60,digdata.outfile);
		putc(0x1,digdata.outfile);
	}
	putc(BBCCURSON,digdata.outfile);
	fflush(digdata.outfile);
}


/* BBC Model B only has filled triangles */
static bbcfilltriangle(bundle,x1,y1,x2,y2,x3,y3)
int	x1,y1,x2,y2,x3,y3;
int	bundle;
{

	putc(BBCPLOT,digdata.outfile);
	putc(BBCMOVETO,digdata.outfile);
	digbbccoord(x3,y3);

	putc(BBCPLOT,digdata.outfile);
	putc(BBCMOVETO,digdata.outfile);
	digbbccoord(x2,y2);

	if (!bundle) {
		putc(BBCDRAWMODE,digdata.outfile) ;
		putc(0x60,digdata.outfile) ;
		putc(0x60,digdata.outfile) ;
	}

	putc(BBCPLOT,digdata.outfile);
	putc(BBCFILLTRI,digdata.outfile);
	digbbccoord(x1,y1);

	if (!bundle) {
		putc(BBCDRAWMODE,digdata.outfile) ;
		putc(0x60,digdata.outfile) ;
		putc(1,digdata.outfile) ;
	}
}
/* BBC Master has fill box command */
static bbcfillbox(bundle,x1,y1,x2,y2)
int	x1,y1,x2,y2;
int	bundle ;
{

	putc(BBCPLOT,digdata.outfile);
	putc(BBCMOVETO,digdata.outfile);
	digbbccoord(x2,y2);

	if (!bundle) {
		putc(BBCDRAWMODE,digdata.outfile) ;
		putc(0x60,digdata.outfile) ;
		putc(0x60,digdata.outfile) ;
	}

	putc(BBCPLOT,digdata.outfile);
	putc(BBCFILLBOX,digdata.outfile);
	digbbccoord(x1,y1);

	if (!bundle) {
		putc(BBCDRAWMODE,digdata.outfile) ;
		putc(0x60,digdata.outfile) ;
		putc(1,digdata.outfile) ;
	}
	return ;
}

/* draw a filled rectangle */
bbc_fillrect(bundle,x1,y1,x2,y2)
int	bundle;
int	x1,y1,x2,y2;
{
	/* draw filled box */
	if (digDEVb.subtype) 
		bbcfillbox(bundle,x1,y1,x2,y2);
	else {
		bbcfilltriangle(bundle,x1,y1,x1,y2,x2,y2);
		bbcfilltriangle(bundle,x1,y1,x2,y1,x2,y2);
	}
}

/* change line/text raster operation */
bbc_rasterop(mode)
int	mode;
{
	int	ret;
	putc(BBCDRAWMODE,digdata.outfile);
	if (mode)
		putc(0x3,digdata.outfile);
	else
		putc(0x60,digdata.outfile);
	putc(0x1,digdata.outfile);
	ret=digDEVb.rop;
	digDEVb.rop=mode;

	return(ret);
}

/* initialise pointing device */
bbc_pointer(flag,xm,ym)
int	flag;
int	xm,ym;
{
	extern int	errno;
	struct termio 	newmode;

	if (flag) {
		/* open channel to terminal in raw mode */
		if (digdata.tty == -1) {
			if ((digdata.tty = open ("/dev/tty",O_RDONLY)) >= 0) {
				ioctl(digdata.tty,TCGETA,&digdata.oldmode);
				newmode = digdata.oldmode;
				newmode.c_lflag = newmode.c_lflag & ~ICANON;
				newmode.c_lflag = newmode.c_lflag & ~ECHO;
				newmode.c_cc[4]=1;		/* 1 character burst */
				newmode.c_cc[5]=0;		/* wait */
				ioctl(digdata.tty,TCSETA,&newmode);
			}
			else {
				fprintf(stderr,"diginitmouse: unable to open keyboard, errno=%d\n",errno);
				return(-1);
			}
		}
		/* mouse on */
		putc(BBCMOUSEON,digdata.outfile);
		digbbccoord(xm,ym);
		fflush(digdata.outfile);
	}
	else {
		/* restore keyboard */
		if (digdata.tty >= 0) {
			ioctl(digdata.tty,TCFLSH,0);
			ioctl(digdata.tty,TCSETA,&digdata.oldmode) ;
			close(digdata.tty);
		}
		digdata.tty = -1;
		/* mouse off */
		putc(BBCMOUSEOFF,digdata.outfile);
		fflush(digdata.outfile);
	}
}

/* bbc co-ordinate decoding */
static diggetcoords(array,xm,ym)
char 	*array;
int	*xm,*ym;
{
	int	i;
	int	xhi,xlo,yhi,ylo;

	for (i=0;i<8;i++) array[i] = array[i] & 0xF;

	xlo = (array[0] << 4) + array[1] ;
	xhi = (array[2] << 4) + array[3] ;
	ylo = (array[4] << 4) + array[5] ;
	yhi = (array[6] << 4) + array[7] ;

	*xm = ((xhi << 8) + xlo) >> 1;
	*ym = ((yhi << 8) + ylo) >> 2;
}

/* get mouse status */
bbc_mouse(but,xm,ym,ich)
int	*but;
int	*xm;
int	*ym;
int	*ich;
{
	static int	butold=0,xmold=0,ymold=0;
	char		ch;
	char		coords[8];
	int		i;

	*but = butold;
	*xm = xmold;
	*ym = ymold;

	read(digdata.tty,&ch,1);
	*ich = ch;
	
	/* check for a mouse button event */
	if (ch == ESC) {
		/* have potential bbc mouse string */
		read(digdata.tty,&ch,1);
		*ich=0;
		switch (ch) {
		case 'L':	
			*but |= 1;  
			break;
		case 'M':	
			*but &= ~1; 
			break;
		case 'R':	
			*but |= 2;  
			break;
		case 'S':	
			*but &= ~2; 
			break;
		case 'O':	
			*but |= 4;  
			break;
		case 'Q':	
			*but &= ~4; 
			break;
		case 'E':	/* co-ords following */
			for (i=0;i<8;i++) {
				read(digdata.tty,&coords[i],1);
			}
			diggetcoords(coords,xm,ym);
			break;
		default:
			*ich = ch;
			return(DIG_KEY);
		}
		butold = *but;
		xmold  = *xm;
		ymold  = *ym;
		return(DIG_MOUSE);
	}
	return(DIG_KEY);
}

/* display text on prompt line */
bbc_prompt(str)
char	*str;
{
	putc(BBCHOME,digdata.outfile);
	fputs(str,digdata.outfile);
	putc(BBCCLEAREOL,digdata.outfile);
	fflush(digdata.outfile);
}

/* plot a single point */
static bbcpoint(x,y)
int	x,y;
{
	putc(BBCPLOT,digdata.outfile);
	putc(BBCPOINT,digdata.outfile);
	digbbccoord(x,y);
}

/* grey-level display */
bbc_gscale(x,y,height,buff)
int	x,y;
int	height;
unsigned char	*buff;
{
	int	xout,yout,k;
	xout = x;
	for (k=height-1;k>=0;k--,buff++) {
		yout = y + 2*k;
		switch (*buff) {
		case 4:
			bbcpoint(xout+1,yout);
		case 3:
			bbcpoint(xout,yout+1);
		case 2:
			bbcpoint(xout+1,yout+1);
		case 1:
			bbcpoint(xout,yout);
			break;
		}
	}
}

#endif


