/* kyocera -- DIG routines for Kyocera K1000 laser printer */

/* Mark Huckvale - April 1990 */

#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include "SFSCONFG.h"
#include "dig.h"
#include "digdata.h"

#ifdef EVk

/* Kyocera laser constants */
#define KYSIZEX		3000		/* page length */
#define KYSIZEY		2100		/* page width */
#define	KYOFFX		200		/* x-axis offset = 200 dots */
#define KYOFFY		120		/* y-axis offset = 120 dots */

struct {
	short 	mode;		/* 0=?, 1=initialised */
	int	subtype;	/* 0=landscape, 1=portrait */
	FILE	*gpipe;		/* record of pipe output */
} digDEVk;


/* device characteristics */
kyoc_params(subtype,digdev)
int	subtype;
struct digdevice *digdev;
{
	if (subtype==0) {
		/* landscape mode */
		digdev->nhoriz = KYSIZEX;
		digdev->nvert  = KYSIZEY;
	}
	else {
		/* portrait mode */
		digdev->nhoriz = KYSIZEY;
		digdev->nvert  = KYSIZEX;
	}
	digdev->aspect = 1.0;
	digdev->chwidth = 25;
	digdev->chheight = 50;
	digdev->greylevels = 10;
	digdev->greypixwidth = 3;
	digdev->greypixheight = 3;
	digDEVk.subtype = subtype;
}

/* open output file */
kyoc_outfile(fp)
FILE	**fp;
{
	if ((*fp = popen("/usr/local/bin/kyocera","w"))==NULL)
		error("unable to open pipe to printer on '%s'",DIGPIPE);
	digDEVk.gpipe = *fp;
}

/* open graphics */
kyoc_open()
{
	if (!(digDEVk.mode)) {
		fprintf(digdata.outfile,"%cR! RES;UNIT D;",'!');
		fprintf(digdata.outfile,"SLM 0; STM 0;");
		if (digDEVk.subtype) {
			/* portrait mode */
			fprintf(digdata.outfile,"FONT 8;SCPI 12;");
		}
		else {
			/* landscape mode */
			fprintf(digdata.outfile,"FONT 26;SCPI 12;");
		}
	}
	digDEVk.mode++;
}

/* interrupt graphics */
kyoc_inter()
{
	fflush(digdata.outfile);
}

/* close graphics - return to text mode */
kyoc_close()
{
	fprintf(digdata.outfile,"PAGE; RES; EXIT;");
	if (digdata.outfile == digDEVk.gpipe)
		pclose(digdata.outfile);
	else
		fclose(digdata.outfile);
	digDEVk.mode = 0;
}

/* clear screen */
kyoc_clear()
{
	fprintf(digdata.outfile,"PAGE;");
}

/* set up colour from bundle */
static kyoc_colour(bundle)
int	bundle;
{
	if (bundle == 0)
		fprintf(digdata.outfile,"SPD 0;");
	else if (bundle < 5)
		fprintf(digdata.outfile,"SPD 6;");
	else if (bundle < 20)
		fprintf(digdata.outfile,"SPD 4;");
	else if (bundle < 35)
		fprintf(digdata.outfile,"SPD 2;");
	else
		fprintf(digdata.outfile,"SPD 1;");
}

/* draw a poly-line */
kyoc_polyline(bundle,buff,len)
int	bundle;
short	*buff;
int	len;
{
	int	x,y;

	/* sort out colours */
	if (!bundle) return;
	kyoc_colour(bundle);

	/* position to first point */
	x = KYOFFX + *buff++;
	y = KYOFFY + (digdata.nvert - *buff++);
	fprintf(digdata.outfile,"MAP %d,%d;",x,y);
	len -= 2;

	/* plot rest of points */
	while (len > 0) {
		x = KYOFFX + *buff++;
		y = KYOFFY + (digdata.nvert - *buff++);
		fprintf(digdata.outfile,"DAP %d,%d;",x,y);
		len -= 2;
	}
}

/* draw 'hardware' text */
kyoc_text(bundle,x,y,str)
int	bundle;
int	x,y;
char	*str;
{
	int	xout,yout;

	/* sort out colours */
	if (!bundle) return;
	kyoc_colour(bundle);

	/* write string */
	xout = KYOFFX + x;
	yout = KYOFFY + (digdata.nvert - y) - 10;	/* 10 pixels to alpha baseline */
	fprintf(digdata.outfile,"MAP %d,%d;",xout,yout);
	while (*str) {
		if (*str != '"') {
			fprintf(digdata.outfile,"TEXT \"");
			while (*str && (*str != '"'))
				fputc(*str++,digdata.outfile);
			fprintf(digdata.outfile,"\",E;");
		}
		else {
			fprintf(digdata.outfile,"TEXT '");
			while (*str && (*str != '\''))
				fputc(*str++,digdata.outfile);
			fprintf(digdata.outfile,"',E;");
		}
	}
}

/* draw a filled rectangle */
kyoc_fillrect(bundle,x1,y1,x2,y2)
int	bundle;
int	x1,y1,x2,y2;
{
	int	xout,yout;

	/* sort out colours */
	if (!bundle) return;

	/* draw filled box */
	xout = KYOFFX + x1;
	yout = KYOFFY + (digdata.nvert - y1);
	fprintf(digdata.outfile,"MAP %d,%d;",xout,yout);
	fprintf(digdata.outfile,"BLK %d,%d;",x2-x1,y2-y1);
}

/* draw a circle */
kyoc_arc(bundle,xc,yc,r,ang1,ang2)
int	bundle;
int	xc,yc,r;
{
	int	xout,yout,thick;

	/* sort out colours */
	if (!bundle) return;
	kyoc_colour(bundle);

	/* draw an arc */
	xout = KYOFFX + xc;
	yout = KYOFFY + (digdata.nvert - yc);
	if (bundle < 20)
		thick=4;
	else if (bundle < 35)
		thick=2;
	else
		thick=1;
	fprintf(digdata.outfile,"MAP %d,%d;",xout,yout);
	fprintf(digdata.outfile,"ARC %d,%d,%d,%d;",r-thick,r,(450-ang2)%360,(450-ang1)%360);
}

/* draw a circle */
kyoc_circle(bundle,xc,yc,r)
int	bundle;
int	xc,yc,r;
{
	int	xout,yout;

	/* sort out colours */
	if (!bundle) return;
	kyoc_colour(bundle);

	/* draw a circle outline */
	xout = KYOFFX + xc;
	yout = KYOFFY + (digdata.nvert - yc);
	fprintf(digdata.outfile,"MAP %d,%d;",xout,yout);
	fprintf(digdata.outfile,"CIR %d;",r);

}

/* draw a filled circle */
kyoc_fcircle(bundle,xc,yc,r)
int	bundle;
int	xc,yc,r;
{
	int	xout,yout;
	/* sort out colours */
	kyoc_colour(bundle);

	/* draw a filled circle */
	xout = KYOFFX + xc;
	yout = KYOFFY + (digdata.nvert - yc);
	fprintf(digdata.outfile,"MAP %d,%d;",xout,yout);
	fprintf(digdata.outfile,"ARC 0,%d,0,360;",r);

}

/* dot dithers for 10-level grey scale */
static unsigned int dither[10][3] =
{
	0x0,0x0,0x0,  /* 0 (white) */
	0x4,0x0,0x0,  /* 1 */
	0x5,0x0,0x0,  /* 2 */
	0x5,0x0,0x4,  /* 3 */
	0x5,0x0,0x5,  /* 4 */
	0x5,0x2,0x5,  /* 5 */
	0x7,0x2,0x5,  /* 6 */
	0x7,0x6,0x5,  /* 7 */
	0x7,0x7,0x5,  /* 8 */
	0x7,0x7,0x7  /* 9 (black) */
};

/* grey-level display */
kyoc_gscale(x,y,height,buff)
int		x,y;
int		height;
unsigned char	*buff;
{
	register int 	j,k;
	register unsigned char	*p;
	int		xout,yout,xfin;
	unsigned int	pixel;

	/* get co-ordinates @ 720 dpi */
	xout = ((KYOFFX + (KYSIZEX-9-x))*12+4)/5;		/* 9 is fiddle factor */
	yout = ((KYOFFY + (KYSIZEY-(y+height*3)))*12+4)/5;

	/* switch to Hewlett-Packard emulation */
	fprintf(digdata.outfile,"FONT 1;EXIT;");
	fprintf(digdata.outfile,"\033*t300R");
	fprintf(digdata.outfile,"\033&l00");
	fprintf(digdata.outfile,"\033&a%dV",xout);
	fprintf(digdata.outfile,"\033&a%dH",yout);

	for (j=0;j<3;j++) {	/* for each 3 lines of dots */
		fprintf(digdata.outfile,"\033*r1A\033*b%dW",3*(1+height/8));
		p = buff;
		for (k=0;k<=height/8;k++) {	/* for each pixel in slice */
			pixel  = dither[*p++][j] << 5;
			pixel |= dither[*p++][j] << 2;
			pixel |= dither[*p][j] >> 1;
			putc(pixel,digdata.outfile);
			pixel  = dither[*p++][j] << 7;
			pixel |= dither[*p++][j] << 4;
			pixel |= dither[*p++][j] << 1;
			pixel |= dither[*p][j] >> 2;
			putc(pixel,digdata.outfile);
			pixel  = dither[*p++][j] << 6;
			pixel |= dither[*p++][j] << 3;
			pixel |= dither[*p++][j];
			putc(pixel,digdata.outfile);
		}
		fprintf(digdata.outfile,"\033*rB");
	}

	/* switch back to Kyocera mode */
	fprintf(digdata.outfile,"%cR! UNIT D;",'!');
	fprintf(digdata.outfile,"SLM 0; STM 0;");
	if (digDEVk.subtype) {
		/* portrait mode */
		fprintf(digdata.outfile,"FONT 8;SCPI 12;");
	}
	else {
		/* landscape mode */
		fprintf(digdata.outfile,"FONT 26;SCPI 12;");
	}
}

#endif


