/* sbp8dma -- SoundBlaster Compatible 8-bit replay by DMA */

/* Mark Huckvale - University College London */

/* version 1.0 - December 1994 */

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <dos.h>
#include <conio.h>
#include <go32.h>
#include <dpmi.h>

/* routines for Protected Mode replay of (top 8 bits of) 12-bit/16-bit data */
int	SB8Init(void);
int	SB8Play(short *buff,int numf,int srate,int nchan,int nbits);
void	SB8Close(void);

/* defines for SoundBlaster */
#define SB_DSP_RESET		0x06
#define SB_DSP_READ_DATA	0x0A
#define SB_DSP_WRITE_DATA	0x0C
#define SB_DSP_WRITE_STATUS	0x0C
#define SB_DSP_DATA_AVAIL	0x0E

/* DSP Commands */
#define SB_DMA_8_BIT_DAC	0x14
#define SB_TIME_CONSTANT	0x40
#define SB_HALT_DMA		0xD0
#define SB_CONTINUE_DMA		0xD4
#define SB_SPEAKER_ON		0xD1
#define SB_SPEAKER_OFF		0xD3
#define SB_DSP_ID		0xE0
#define SB_DSP_VER		0xE1

#define SB_SET_BLOCKSIZE	0x48
#define SB_HIGH_DMA_8_BIT_DAC	0x91
#define SB_HIGH_DMA_8_BIT_ADC	0x99

/* Card parameters */
unsigned int	sb_ioaddr;
unsigned int	sb_irq;
unsigned int	sb_dmachan;
unsigned int	sb_type;
unsigned int	sb_stereo;
unsigned int	sb_nbits;

/* write a DSP command to SoundBlaster board */
#define sb_writedsp(x) {						\
	while(inportb(sb_ioaddr + SB_DSP_WRITE_STATUS) & 0x80);		\
	outportb(sb_ioaddr + SB_DSP_WRITE_DATA, (x));			\
}

/* Defines for 8237 DMA Controller IO addresses */
#define SB_DMA		0
#define SB_DMA_STATUS	(SB_DMA+8)
#define SB_DMA_CMD	(SB_DMA+8)
#define SB_DMA_REQUEST	(SB_DMA+9)
#define SB_DMA_MASK	(SB_DMA+10)
#define SB_DMA_MODE	(SB_DMA+11)
#define SB_DMA_FF	(SB_DMA+12)
#define SB_DMA_TMP	(SB_DMA+13)
#define SB_DMA_CLEAR	(SB_DMA+13)
#define SB_DMA_CLRMSK	(SB_DMA+14)
#define SB_DMA_WRMSK	(SB_DMA+15)
#define SB_DMAPAGETOP	0x84

/*  GO32 DPMI structures for accessing DOS memory. */
static _go32_dpmi_seginfo dosmem;   /* DOS (conventional) memory buffer */

/* Is a sound currently playing or recorded ? */
volatile int		sb_dma_active;

/* Conventional memory buffers for DMA. */
static char		*sb_buf[2];
static unsigned int	sb_buflen[2];

/* DMA chunk size, in bytes, must be power of 2 */
#define DMA_CHUNK (0x4000)	/* NOTE: I have found that small values
					may be allocated above 640k, and
					that this leads to problems */

/* read a value returned from DSP */
static int sb_read_dsp(int iobase)
{
	int	i;

	for (i=0;i<10000;i++)
	        if (inportb(iobase + SB_DSP_DATA_AVAIL) & 0x080)
			break;
	return(inportb(iobase+SB_DSP_READ_DATA));
}

/* reset SoundBlaster card */
static void sb_reset(void)
{
	int j;

	outportb(sb_ioaddr + SB_DSP_RESET,1);

	inportb(sb_ioaddr + SB_DSP_RESET);
	inportb(sb_ioaddr + SB_DSP_RESET);
	inportb(sb_ioaddr + SB_DSP_RESET);
	inportb(sb_ioaddr + SB_DSP_RESET);
    
	outportb(sb_ioaddr + SB_DSP_RESET, 0);
	for (j=0;j<100;j++)
		if (sb_read_dsp(sb_ioaddr) == 0xAA)
			 break;
}

#define SB_NUM_BASES	6
static int sbbase[SB_NUM_BASES]={0x210,0x220,0x230,0x240,0x250,0x260};

/* try and find card if port address not given */
static int sb_findcard(void)
{
	int i,j;

	/* search possible base addresses */
	for (i=0;i<SB_NUM_BASES;i++) {

		/* send a reset */
	        outportb(sbbase[i] + SB_DSP_RESET,1);

	        /* wait for a little while */
		inportb(sbbase[i] + SB_DSP_RESET);
		inportb(sbbase[i] + SB_DSP_RESET);
		inportb(sbbase[i] + SB_DSP_RESET);
		inportb(sbbase[i] + SB_DSP_RESET);

		/* request acknowledgement */
		outportb(sbbase[i] + SB_DSP_RESET, 0);

		/* look for reply */
		for(j=0;j<100;j++)
			if (sb_read_dsp(sbbase[i]) == 0xAA)
				/* got it ! */
				return(sbbase[i]);
	}
	/* nope */
	return(0);
}

/* Read soundblaster card parameters from BLASTER enivronment variable .*/
static int sb_getparams()
{
	char *t, *blaster;

	/* set Interrupt and DMA channel to usual defaults */
	sb_irq = 7;
	sb_dmachan = 1;

	/* assume port address unknown */
	sb_ioaddr = 0;

	/* see if environment variable set */
	if ((t=getenv("BLASTER"))) {

		/* take a copy */
		blaster = strdup(t);

		/* Parse the BLASTER variable */
		t = strtok(blaster, " \t");
		while (t && *t) {
			switch (t[0]) {
			case 'A':
			case 'a':
				/* I/O address */
				sscanf(t + 1, "%x", &sb_ioaddr);
				break;
			case 'I':
			case 'i':
				/* IRQ */
				sb_irq = atoi(t + 1);
				break;
			case 'D': 
			case 'd':
				/* DMA channel */
				sb_dmachan = atoi(t + 1);
				break;
			default:
				/* ignore everything else */
				break;
			}
			t = strtok(NULL," \t");
		}
		free(blaster);
        }

#ifdef EMO
printf("BLASTER: port=0x%X dma=%d int=%d\n",sb_ioaddr,sb_dmachan,sb_irq);
#endif

        if (sb_ioaddr==0)
        	return(sb_findcard());
        else
        	return(sb_ioaddr);
}

/* get DOS memory buffers */
static int sb_DOS_alloc()
{
	/* get a block in low memory big enough to hold
	   two buffers aligned on multiple of DMA_CHUNK */
	dosmem.size = (DMA_CHUNK*3)/16;
	if (_go32_dpmi_allocate_dos_memory(&dosmem)) {
		dosmem.size = -1;
		return(0);
	}

	(unsigned long) sb_buf[0] = dosmem.rm_segment * 16;
	(unsigned long) sb_buf[0] += (DMA_CHUNK-1);
	(unsigned long) sb_buf[0] &= ~(long)(DMA_CHUNK-1);
	(unsigned long) sb_buf[1] = (unsigned long) sb_buf[0] + DMA_CHUNK;

#ifdef EMO
	printf("DOS buffers at %08lX, %08lX\n",(unsigned long)(sb_buf[0]),
						(unsigned long)(sb_buf[1]));
#endif
	return(1);
}

/* free DOS memory */
static void sb_DOS_free()
{
	if (dosmem.size != -1)
		_go32_dpmi_free_dos_memory(&dosmem);
	dosmem.size = -1;
}

/* set SoundBlaster sample rate */
static void sb_set_sample_rate(unsigned int rate)
{
	unsigned char tc;

	tc = (unsigned char) (256 - 1000000/rate);

	inportb(sb_ioaddr + SB_DSP_DATA_AVAIL);
	sb_writedsp(SB_TIME_CONSTANT);  /* Command byte for sample rate */
	sb_writedsp(tc);                /* Sample rate time constant */
}

/* SoundBlaster output switch */
static void sb_voice(int state)
{
	sb_writedsp(state ? SB_SPEAKER_ON : SB_SPEAKER_OFF);
}

/* fill a DOS buffer */
static int sb_fill_buffer(int bno,short *buff,int numf)
{
	int	i;
	int	count;
	unsigned char	xbuf[DMA_CHUNK];

	/* copy a chunk into an unsigned byte buffer */
	if (numf<=0) {
		sb_buflen[bno]=0;
		return(0);
	}
	count = (numf <= DMA_CHUNK) ? numf : DMA_CHUNK;
	if (sb_nbits==16)
		for (i=0;i<count;i++,buff++)
			xbuf[i] = (*buff >> 8) + 0x80;
	else
		for (i=0;i<count;i++,buff++)
			xbuf[i] = (*buff >> 4) + 0x80;

	/* copy down into DOS memory for DMA */
	sb_buflen[bno] = count;
        dosmemput(xbuf, count, (unsigned long) sb_buf[bno]);
	return(count);
}

/* set a DOS buffer playing */
static void sb_play_buffer(int bno)
{
	int     t;

	/* check if already done */
	if (sb_buflen[bno] <= 0) { 
	        sb_dma_active = 0;
	        return;
	}

	/* set DMA mode to read memory */
	outportb(SB_DMA_MASK, 4+sb_dmachan);
	outportb(SB_DMA_FF, 0);
	outportb(SB_DMA_MODE, 0x48+sb_dmachan);

	/* set DMA transfer address */
	t = (int) ((unsigned long) sb_buf[bno] & 0xFFFF);
	outportb(SB_DMA + 2 * sb_dmachan, t & 0xFF);
	outportb(SB_DMA + 2 * sb_dmachan, t >> 8);
	t = (int) ((unsigned long) sb_buf[bno] >> 16);
	outportb(SB_DMAPAGETOP - sb_dmachan, t);

	/* set transfer length */
	outportb(SB_DMA + 2 * sb_dmachan + 1, (sb_buflen[bno]-1) & 0xFF);
	outportb(SB_DMA + 2 * sb_dmachan + 1, (sb_buflen[bno]-1) >> 8);

	/* OK set it going */
	outportb(SB_DMA_MASK, sb_dmachan);

	/* set SoundBlaster to receive data */
	sb_writedsp(SB_DMA_8_BIT_DAC);
	sb_writedsp((sb_buflen[bno]-1) & 0xFF);   /* write length */
	sb_writedsp((sb_buflen[bno]-1) >> 8);

	/* a sound is playing now. */
	sb_dma_active = 1;
}

/* return how many bytes still to be DMA-ed */
static int sb_read_counter(void)
{
	int	len;
	len = inportb(SB_DMA + 2 * sb_dmachan + 1);
	len |= inportb(SB_DMA + 2 * sb_dmachan + 1) << 8;
	if (len==0xFFFF) inportb(sb_ioaddr + SB_DSP_DATA_AVAIL);
	return(len);
}

/* SoundBlaster initialise - interface routine */
int	SB8Init(void)
{
	if (sb_getparams()==0)
		return(0);
	if (sb_DOS_alloc()==0)
		return(0);
	return(1);
}

/* play top 8-bits of a signed 16-bit data buffer */
int	SB8Play(short *buff,int numf,int srate,int nchan,int nbits)
{
	int	count,len;
	int	bno;

	sb_reset();
	sb_set_sample_rate(srate);
	sb_voice(1);
	sb_stereo = (nchan > 1);
	sb_nbits = (nbits==12) ? 12 : 16;

	count = numf;
	len = sb_fill_buffer(0,buff,count);
	count -= len;
	buff += len;
	len = sb_fill_buffer(1,buff,count);
	count -= len;
	buff += len;

	sb_play_buffer(0);
	bno = 1;
	while (sb_dma_active) {
		while (sb_read_counter() != 0xFFFF) /* loop */;
		sb_play_buffer(bno);
#ifdef EMO
		if (kbhit()) {
			if (getch() == 27) {
				sb_writedsp(SB_HALT_DMA);
		                sb_dma_active = 0;
				sb_buflen[0] = sb_buflen[1] = 0;
			}
		}
#endif
		bno = 1 - bno;
		len = sb_fill_buffer(bno,buff,count);
		count -= len;
		buff += len;
	}
		
	sb_voice(0);
	sb_reset();
	return(numf-count);
}

void	SB8Close(void)
{
	sb_DOS_free();
}

#ifdef EMO
#include <math.h>
short	sbuf[100000];
void	main()
{
	int	i;
	double	omega,t;

	omega = 500.0*8.0*atan(1.0)/10000.0;
	for (i=0,t=0;i<100000;i++,t+=omega)
		sbuf[i] = 20000*sin(t);
	for (i=0;i<20;i++) printf("%d,",sbuf[i]); printf("\n");

	if (SB8Init()) {
		i = SB8Play(sbuf,50000,10000,1);
		printf("played %d samples\n",i);
		SB8Close();
	}
	exit(0);
}
#endif
