/*====================== dp - dynamic time warp recognizer =================*/
/*======================     b.l.tan -- january 1987       =================*/
/*======================          version 1.0		   =================*/

/* version 2.0s - December 1987
	- SFS version
	- considerably changed
*/

/*--------------------------------------------------------------------------*/
/**MAN
.TH DP SFS1 UCL
.SH NAME
dp - dynamic programming recogniser
.SH SYNOPSIS
.B dp
(-r) (-a|-b|-c|-d|-e) (-i item) <token_files> file
.SH DESCRIPTION
.I dp
is an isolated word recognizer implemented using a dynamic programming algorithm.
It accepts a list of reference files ("token files") and a single unknown file and calculates
the which token file is "nearest" to the unknown.  The program can report the 
identity of the nearest token, the nearest filename, or a table of matching scores according
to selected options.
.PP
The 
.I dp
program operates using a Euclidean metric between vectors of a "Coefficient" (CO) data
set in the files (identified by the "-i item" switch).  The DP algorithm usually operates
without any slope constraints, but these may be incorporated with the "-r" switch.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program name and version number.
.TP 11
.B -r
Select a 1 to 2 ratio constraint on the length of the templates.
.TP 11
.B -a|-b|-c|-d|-e
Select "-a" for nearest filename, "-b" for table of scores, "-c" for
table of scores with filenames, "-d" as "-b" but sorted by score, "-e" as
"-c" but sorted by score.  Default is to report the
"head.token" field from the best match and test files.
.TP 11
.BI -i item
Select input item number.
.SH INPUT ITEMS
.IP CO.xx 11
Any Coefficient item.
.SH VERSION/AUTHOR
1.0 - b.l.tan
.br
2.0s - Mark Huckvale
*/
/*--------------------------------------------------------------------------*/

/* program credentials */
#define PROGNAME "dp"
#define PROGVERS "2.0s"
char	*progname=PROGNAME;

/* include files */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>	/* maths header */
#include "sfs.h"	/* database structures */

/* manifest constants */
#define MAXFILE 100	/* max number of reference files */
#define EDGE	10.0E6	/* large value for dp algorithm */

/* global data */
struct {
	char	*filename;
	char	token[80];
	float	distance;
} ftable[MAXFILE],hold;
int	ft;
char	testtoken[80];
struct main_header thead,rhead;	/* file headers */
struct item_header titem,ritem;	/* item headers */
struct co_rec	*tco,*rco;	/* CO data sets */
int	slope=0;		/* slope constraints off */
int	report=0;		/* match report option */
float	*dist1,*dist2;		/* distance scores */
float	*cudist1,*cudist2;	/* cumulative distance scores */
float   dtw();			/* dynamic time warping function */



/* distance metric for vocoder frames */
float metric(a,b,nc)
struct co_rec *a,*b;
int	nc;
{
	register int i;
	register float sumsq=(float)0.0,val,*ap,*bp;
	double	sqrt();

	ap = &a->data[0];
	bp = &b->data[0];
	for (i=nc;i;i--) {
		val = *ap++ - *bp++;
		sumsq += val*val;
	}
	return((float)sqrt(sumsq));
}

#define MINOF(x,y,z) (((x) < (y)) ? (((x) < (z)) ? (x) : (z)) : (((y) < (z)) ? (y) : (z)))

/************************************************************************/
/*									*/
/*	dtw -   dynamic time warping routine which takes two tables of	*/
/*		coeffs and returns a similarity measure.		*/
/*									*/
/************************************************************************/

float	dtw(slope,length1,length2,ncoeff)
int	slope;		/* 1:2 ratio constraint */
int	length1;	/* number of input vectors */
int	length2;
int	ncoeff;		/* no of coefficients per vector */
{
	register int	i,j;		/* loop counters */
	float		*tmp;		/* distance vector pointers */
	float		up,left,diag;	/* path values for three DP paths */
	float   	dtwresult;	/* result */

	/* check to see if word lengths are too dissimilar */
	if (slope) {
		if ((length1/length2 >= 2) || (length2/length1 >= 2))
			return((float)EDGE);
	}

	/* initialisation */
	tmp = cudist1;
	for (i=0;i<length1;i++) *tmp++ = (float)EDGE;

	/* DP match */
	for (i=0;i<length2;i++) {
		for (j=0;j<length1;j++) {
			if ((i==0) && (j==0)) {
				diag=(float)0.0;
				left=(float)EDGE;
			}
			else if (j==0) {
				diag=(float)EDGE;
				left=(float)EDGE;
			}
			else {
				diag=cudist1[j-1] + dist1[j-1];
				left=cudist2[j-1] + dist2[j-1];
			}
			up = cudist1[j] + dist1[j];
			dist2[j] = metric(&rco[i],&tco[j],ncoeff);
			cudist2[j] = dist2[j] + MINOF(diag,up,left);
		}
		/* swap over buffers */
		tmp=cudist1;
		cudist1=cudist2;
		cudist2=tmp;
		tmp=dist1;
		dist1=dist2;
		dist2=tmp;
	}

	/* return final distance */
	dtwresult = cudist1[length1-1] / (float)(length1+length2);
	return(dtwresult) ;
}

/* sort score table */
void sortab()
{
	int	i,j;


	for (i=1;i<ft;i++) {
		hold=ftable[i];
		j=i-1;
		while ((j>=0) && (hold.distance < ftable[j].distance)) {
			ftable[j+1]=ftable[j];
			j--;
		}
		ftable[j+1]=hold;
	}
}

/* main program */
void main(argc, argv)
int argc ;
char *argv[] ;
{
	/* option decoding variables */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;
	char		*ty;
	char		*cotype="0";	/* default data type = last */

	/* file variables */
	char	tfilename[SFSMAXFILENAME];	/* filename for the test template */
	char	rfilename[SFSMAXFILENAME];	/* filename for the reference template */
	int	fid;			/* file descriptor */

	/* processing */
	int	i,entry,ncoeff;
	float	min;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Irabcdei:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Dynamic Programming recogniser V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'r' : /* Turn on slope constrains */
			slope++ ;
			break ;
		case 'a' : /* display filenames */
			report = 1;
			break ;	
		case 'b' : /* display table of scores */
			report = 2;
			break ;
		case 'c' : /* display table of scores with filenames */
			report = 3;
			break ;
		case 'd' : /* display table of scores */
			report = 4;
			break ;
		case 'e' : /* display table of scores with filenames */
			report = 5;
			break ;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == CO_TYPE)
					cotype = ty;
				else 
					error("unsuitable item specifier %s",optarg);
		  	}
		  	else 
				error("illegal item specifier %s",optarg);
		  break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<3))
		error("usage: %s (-I) (-r) (-a|-b|-c|-d|-e) (-i item) <token_files> file",PROGNAME);

	if ((argc - optind) > MAXFILE)
		error ("Too many reference files", NULL) ;

	/* load test data */
	strcpy(tfilename,sfsfile(argv[argc-1]));
	if ((fid=sfsopen(tfilename,"r",&thead)) < 0)
		error("access error on %s",tfilename);
	if (!sfsitem(fid,CO_TYPE,cotype,&titem))
		error("cannot find input CO item in %s",tfilename);
	if ((tco=(struct co_rec *)sfsbuffer(&titem,titem.numframes))==NULL)
		error("unable to load data set from %s",tfilename);
	if (sfsread(fid,0,titem.numframes,tco) != titem.numframes)
		error("read error on %s",tfilename);
	sfsclose(fid);

	/* keep token name */
	if (thead.token[0] != '\0')
		strcpy(testtoken,thead.token);
	else
		strcpy(testtoken,tfilename);


	/* get buffers for distance scores */
	dist1=(float *)calloc(titem.numframes,sizeof(float));
	dist2=(float *)calloc(titem.numframes,sizeof(float));
	cudist1=(float *)calloc(titem.numframes,sizeof(float));
	cudist2=(float *)calloc(titem.numframes,sizeof(float));
	ncoeff = (titem.framesize*titem.datasize-sfsstruct[CO_TYPE])/sizeof(float);

	/* process each reference file in turn */
	argc--;
	while (optind < argc) {
		/* get data from ref file */
		ftable[ft].filename=argv[optind];
		strcpy(rfilename,sfsfile(argv[optind]));
		if ((fid=sfsopen(rfilename,"r",&rhead)) < 0)
			error("access error on %s",rfilename);
		if (!sfsitem(fid,CO_TYPE,cotype,&ritem))
			error("cannot find input CO item in %s",rfilename);
		if ((rco=(struct co_rec *)sfsbuffer(&ritem,ritem.numframes))==NULL)
			error("unable to load data set from %s",rfilename);
		if (sfsread(fid,0,ritem.numframes,rco) != ritem.numframes)
			error("read error on %s",rfilename);
		sfsclose(fid);

		/* check compatibility */
		if (titem.framesize != ritem.framesize)
			error("different framesize in %s",rfilename);

		/* keep token name */
		if (rhead.token[0] != '\0')
			strcpy(ftable[ft].token,rhead.token);
		else
			strcpy(ftable[ft].token,rfilename);

		/* calculate warping distance and store it */
		ftable[ft].distance=dtw(slope,titem.numframes,ritem.numframes,ncoeff);

		/* next file */
		free(rco);
		ft++;
		optind++;
	}

	/* find smallest entry in table */
	min=(float)(2*EDGE);
	entry=0;
	for (i=0;i<ft;i++) if (ftable[i].distance < min) {
		min = ftable[i].distance;
		entry = i;
	}

	/* print results */
	switch (report) {
	case 0:	/* print tokens */
		printf("%s %s\n",testtoken,ftable[entry].token);
		break;
	case 1:	/* print filename */
		printf("%s %s\n",tfilename,ftable[entry].filename);
		break;
	case 2:	/* print score table (tokens) */
		for (i=0;i<ft;i++) 
			printf("%10g %s %s\n",ftable[i].distance,testtoken,ftable[i].token);
		break;
	case 3:	/* print score table filenames */
		for (i=0;i<ft;i++) 
			printf("%10g %s %s\n",ftable[i].distance,tfilename,ftable[i].filename);
		break;
	case 4:	/* print score table (tokens) */
		sortab();
		for (i=0;i<ft;i++) 
			printf("%10g %s %s\n",ftable[i].distance,testtoken,ftable[i].token);
		break;
	case 5:	/* print score table filenames */
		sortab();
		for (i=0;i<ft;i++) 
			printf("%10g %s %s\n",ftable[i].distance,tfilename,ftable[i].filename);
		break;
	}

	/* that's all folks */
	exit(0);
}

