/* voc26 -- SRUBANK style 26-channel filterbank */

/* M.A.Huckvale - University College London */

/* version 1.0 - February 1995
	- 20kHz version only
*/
/* version 2.0 - February 1996
	- replace filtering with general purpose routines
*/

#define PROGNAME "voc26"
#define PROGVERS "1.0"
char	*progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH VOC26 SFS1 UCL SFS
.SH NAME
voc26 - SRUBANK style 26-channel vocoder
.SH SYNOPSIS
.B voc26
(-i item) (-f) (-q nlevel) file
.SH DESCRIPTION
.I voc26
is a straightforward implementation of a 26-channel filterbank analyser
using two second-order section Butterworth bandpass filters spaced as in
Holmes paper in IEE proceedings with
SRU extensions at the low and high frequencies.
Energy smoothing done at 50Hz to give default 100Hz framerate.
.PP
There is no 
limit to the length of the input signal.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version number.
.TP 11
.BI -i item
Select input item.
.TP 11
.B -f
Faster frame rate.  Sample output at 200 frames/sec, filtering remains at
50Hz to avoid pitch pulses.
.TP 11
.BI -q nlevels
Quantise the output energies to 'nlevels'.  The top 50dB of the signal is quantised.
.SH INPUT ITEMS
.IP SP
Any speech signal sampled at greater than 20kHz.
.SH OUTPUT ITEMS
.IP CO 11
Vocoder energies.
.SH VERSION/AUTHOR
2.0 - Mark Huckvale
.SH SEE ALSO
voc8(1), voc19(1)
*/
/*--------------------------------------------------------------------------*/

/* standard definitions */
#include "SFSCONFG.h"
#include <stdio.h>		/* standard io library */
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"		/* database structures */
#include "filter.h"

/* manifest constants */
#define DBRANGE 50.0
#define NCHAN 26		/* # channels */
int	window;			/* window size in samples */
int	fastframe=0;		/* faster frame rate */
int	qlevels=0;		/* # quantisation levels */

/* channel filters */
FILTER	*chan[NCHAN];
FILTER	*schan[NCHAN];

/* design filters */
void design(srate,frate)
double	srate;
double	frate;
{
	int	i;

	chan[ 0] = filter_design(FILTER_BAND_PASS,4,  60, 180,srate);
	chan[ 1] = filter_design(FILTER_BAND_PASS,4, 180, 300,srate);
	chan[ 2] = filter_design(FILTER_BAND_PASS,4, 300, 420,srate);
	chan[ 3] = filter_design(FILTER_BAND_PASS,4, 420, 541,srate);
	chan[ 4] = filter_design(FILTER_BAND_PASS,4, 541, 667,srate);
	chan[ 5] = filter_design(FILTER_BAND_PASS,4, 667, 795,srate);
	chan[ 6] = filter_design(FILTER_BAND_PASS,4, 795, 928,srate);
	chan[ 7] = filter_design(FILTER_BAND_PASS,4, 928,1066,srate);
	chan[ 8] = filter_design(FILTER_BAND_PASS,4,1066,1213,srate);
	chan[ 9] = filter_design(FILTER_BAND_PASS,4,1213,1367,srate);
	chan[10] = filter_design(FILTER_BAND_PASS,4,1367,1532,srate);
	chan[11] = filter_design(FILTER_BAND_PASS,4,1532,1708,srate);
	chan[12] = filter_design(FILTER_BAND_PASS,4,1708,1896,srate);
	chan[13] = filter_design(FILTER_BAND_PASS,4,1896,2097,srate);
	chan[14] = filter_design(FILTER_BAND_PASS,4,2097,2316,srate);
	chan[15] = filter_design(FILTER_BAND_PASS,4,2316,2552,srate);
	chan[16] = filter_design(FILTER_BAND_PASS,4,2552,2831,srate);
	chan[17] = filter_design(FILTER_BAND_PASS,4,2831,3153,srate);
	chan[18] = filter_design(FILTER_BAND_PASS,4,3153,3555,srate);
	chan[19] = filter_design(FILTER_BAND_PASS,4,3555,4093,srate);
	chan[20] = filter_design(FILTER_BAND_PASS,4,4093,4632,srate);
	chan[21] = filter_design(FILTER_BAND_PASS,4,4632,5344,srate);
	chan[22] = filter_design(FILTER_BAND_PASS,4,5344,6207,srate);
	chan[23] = filter_design(FILTER_BAND_PASS,4,6207,7221,srate);
	chan[24] = filter_design(FILTER_BAND_PASS,4,7221,8235,srate);
	chan[25] = filter_design(FILTER_BAND_PASS,4,8235,9249,srate);

	for (i=0;i<NCHAN;i++)
		schan[i] = filter_design(FILTER_LOW_PASS,2,frate/2,frate/2,srate);
	
}

/* filter processing */
void process(fid,window,numframes,ofid,co)
int		fid;		/* input file */
int		window;		/* input window size */
int		numframes;	/* # input frames */
int		ofid;		/* output file */
struct co_rec	*co;		/* output buffer */
{
	int	i;				/* sample */
	int	j;				/* channel */
	int	k;				/* window value */
	float	smrect;				/* smoothing filter output */
	int	cnt=1;				/* framecounter */
	int	winsize;			/* window size */
	short	*sp,*fsp,*rsp;
	static float maxamp = (float)-1000;
	int	iamp;

	/* get analysis window size */
	if (fastframe)
		winsize = window/2;
	else
		winsize = window;

	/* estimate # output frames */
	numframes /= winsize;

	sp = calloc(winsize,sizeof(short));
	fsp = calloc(winsize,sizeof(short));
	rsp = calloc(winsize,sizeof(short));

	/* process each window in turn */
	for (i=0;sfsread(fid,i,winsize,sp)==winsize;i+=winsize) {

		/* for each channel in turn */
		for (j=0;j<NCHAN;j++) {

			/* filter the window */
			filter_signal(chan[j],sp,fsp,winsize);

			/* rectify */
			for (k=0;k<winsize;k++)
				if (fsp[k]<0) fsp[k] = -fsp[k];

			/* smooth */
			filter_signal(schan[j],fsp,rsp,winsize);
				
			/* store energy for filter */
			smrect = rsp[winsize-1];
			if (smrect < 1) smrect = (float)0.5;
			co->data[j] = (float)(20.0 * log10(smrect));
			
			if (qlevels) {
				iamp = (int)(qlevels*(co->data[j] - maxamp + DBRANGE - 1.0e-6)/DBRANGE);
				if (iamp < 0) iamp = 0;
				co->data[j] = (float)(iamp*DBRANGE/qlevels + maxamp - DBRANGE);
			}
		}

		/* write output record */
		co->posn=i;
		co->size=winsize;
		co->flag=0;
		co->mix=(float)0.0;
		co->gain=(float)1.0;
		if (ofid) {
			if (sfswrite(ofid,1,co)!=1)
				error("write error on output file",NULL);
		}

		/* report progress */
		if (((++cnt%10)==0) && ttytest()) {
			printf("\rFrame %d/%d",cnt,numframes);
			fflush(stdout);
		}
	}

	free(sp);
	free(fsp);
	free(rsp);
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it=SP_TYPE;	/* item type selection */
	char		*ty="0";	/* item match selection */
	/* input file variables */
	char		filename[SFSMAXFILENAME]; /* sfs file name */
	int		fid;		/* input file descriptor */
	struct item_header spitem;	/* item header for speech data */
	/* output file variables */
	int		ofid;		/* output file descriptor */
	struct item_header coitem;	/* item header for vocoder data */
	struct co_rec	*co;
	double		srate;
	
	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:f")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: 26-channel vocoder V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it != SP_TYPE)
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'f' :	/* fast frame rate */
			fastframe++;
			break;
		case 'q' :	/* select quantisation */
			qlevels = atoi(optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-f) (-q nlevels) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open data file */
	if ((fid=sfsopen(filename,"w",NULL)) < 0) {
		if (fid==-1)
			error("cannot find file: %s",filename);
		else
			error("access error on %s",filename);
	}

	/* locate input speech */
	if (!sfsitem(fid,it,ty,&spitem))
		error("cannot find SP data in %s",filename);

	/* design filters */
	srate = 1.0/spitem.frameduration;
	window = (int)((0.5+(0.01/spitem.frameduration)));
	if (srate < 20000)
		error("input signal sampled at too low a rate");
	design(srate,100.0);

	/* create output item header */
	if (fastframe && qlevels) {
		sfsheader(&coitem,CO_TYPE,1,sizeof(float),
			NCHAN+sfsstruct[CO_TYPE]/sizeof(float),
			spitem.frameduration,spitem.offset,window/2,0,0);
		sprintf(coitem.history,"%s(%d.%02d;fastframe,quantised=%d)",
				PROGNAME,spitem.datatype,spitem.subtype,qlevels);
	}
	else if (fastframe) {
		sfsheader(&coitem,CO_TYPE,1,sizeof(float),
			NCHAN+sfsstruct[CO_TYPE]/sizeof(float),
			spitem.frameduration,spitem.offset,window/2,0,0);
		sprintf(coitem.history,"%s(%d.%02d;fastframe)",
				PROGNAME,spitem.datatype,spitem.subtype);
	}
	else if (qlevels) {
		sfsheader(&coitem,CO_TYPE,1,sizeof(float),
			NCHAN+sfsstruct[CO_TYPE]/sizeof(float),
			spitem.frameduration,spitem.offset,window,0,0);
		sprintf(coitem.history,"%s(%d.%02d;quantised=%d)",
				PROGNAME,spitem.datatype,spitem.subtype,qlevels);
	}
	else {
		sfsheader(&coitem,CO_TYPE,1,sizeof(float),
			NCHAN+sfsstruct[CO_TYPE]/sizeof(float),
			spitem.frameduration,spitem.offset,window,0,0);
		sprintf(coitem.history,"%s(%d.%02d)",
				PROGNAME,spitem.datatype,spitem.subtype);
	}
	strcpy(coitem.params,"labels=x2|120|360|603|859|1136|1445|1796|2198|2670|3315|4277|5700|7728");

	/* open output channel */
	if ((ofid=sfschannel(filename,&coitem)) < 0)
		error("cannot open output channel",NULL);

	/* get buffer */
	if ((co=(struct co_rec *)sfsbuffer(&coitem,1))==NULL)
		error("cannot get output buffer",NULL);

	/* do processing */
	if (qlevels) {
		process(fid,window,spitem.numframes,0,co);
		process(fid,window,spitem.numframes,ofid,co);
	}
	else
		process(fid,window,spitem.numframes,ofid,co);

	/* update data file */
	if (!sfsupdate(filename))
		error("update error on %s",filename);

	/* clear progress */
	if (ttytest()) {
		printf("\r                       \r");
		fflush(stdout);
	}

	/* that's all folks */
	exit(0);
}

