/* trload -- load a track item from an ASCII list of numbers */

/* M.A.Huckvale - January 1989 */

/* version 1.0 */
/* version 1.1
	- floating point sampling rate
	- optional offset
*/
/* version 1.2 - November 1989
	- optional offset and scaling factor to track
*/

/*-------------------------------------------------------------------*/
/**MAN
.TH TRLOAD SFS1 UCL
.SH NAME
trload -- load a track item from a text file of numbers
.SH SYNOPSIS
.B trload
(-f freq) (-h history) (-O timeoffset) (-o scaleoffset) (-s scalefactor) textfile file
.SH DESCRIPTION
.I trload
constructs a new track item in the given file from text numbers
a text file.  The numbers can be separated by any white-space.
.PP
.B Options
.TP 11
.B -I
Identify the program version.
.TP 11
.BI -f freq
Use a sampling interval based on 1/freq seconds for sampling the time of each
track value. Default 100.
.TP 11
.BI -h history
Place a "history=" indicator in the created item header, with the argument
.I history.
.TP 11
.BI -O timeoffset
Set "offset" field in output track item to given value.
.TP 11
.BI -o scaleoffset
Add this offset to all values before scaling and loading into track.
.TP 11
.BI -s scalefactor
Scale all values by this number before loading into track.
.SH OUTPUT ITEMS
.IP TR 11
Loaded track item
.SH VERSION/AUTHOR
1.2 - Mark Huckvale
*/
/*---------------------------------------------------------------*/

/* program name and version */
#define	PROGNAME "trload"
#define PROGVERS "1.2s"
char	*progname = PROGNAME;

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "sfs.h"		/* database filing system structures */

/* global data */
struct item_header 	tritem;			/* trcak item header  */
char			textname[SFSMAXFILENAME];	/* text file name */
char			filename[SFSMAXFILENAME];	/* dbase file name */
double			fdur=1.0E-2;		/* default sample duration */
char			*trhist;		/* track history */
double			offset=0;
double			scalefac=1.0;
double			scaleoff=0.0;

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;			/* option index */
	extern char	*optarg;		/* option argument ptr */
	int		errflg = 0;		/* option error flag */
	int		c;			/* option switch */

	/* processing variables */
	FILE		*ip;			/* input text file */
	int		ofid;			/* output file descriptors */
	float		val;			/* input value */
	double		srate,atof();

	/* decode switches */
	while ( (c = getopt(argc,argv,"If:h:O:o:s:")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Load track data from text V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'f' :	/* select sampling frequency */
			srate = atof(optarg);
			if (srate<=0)
				error("bad sampling frequency: %s",optarg);
			else
				fdur = 1.0/srate;
			break;
		case 'h' :	/* track history */
			trhist=optarg;
			break;
		case 'O' :	/* offset */
			offset=atof(optarg);
			break;
		case 'o' :	/* scale offset */
			scaleoff=atof(optarg);
			break;
		case 's' :	/* scale factor */
			scalefac=atof(optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	/* check command line */
	if (errflg || (argc<3))
		error("usage: %s (-I) (-f freq) (-h trhist) (-O timoffset) (-o scaleoffset) (-s scalefactor) textfile file",PROGNAME);

	/* get text file name */
	if (optind < argc) 
		strcpy(textname,argv[optind]);
	else
		error("no text file specified",NULL);
	optind++;

	/* get data filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open text file */
	if ((ip=fopen(textname,"r"))==NULL)
		error("cannot open %s",textname);

	/* open data file */
	if ((ofid=sfsopen(filename,"w",NULL)) < 0) {
		if (ofid==-1)
			error("cannot find file %s",filename);
		else
			error("access error on %s",filename);
	}
	sfsclose(ofid);

	/* create output item header */
	sfsheader(&tritem,TR_TYPE,1,4,1,fdur,offset,1,0,0);
	if ((scalefac==1.0) && (scaleoff==0.0)) {
		if (trhist) 
			sprintf(tritem.history,"%s(file=%s,rate=%g,offset=%g,history=%s)",PROGNAME,textname,1.0/fdur,offset,trhist);
		else
			sprintf(tritem.history,"%s(file=%s,rate=%g,offset=%g)",PROGNAME,textname,1.0/fdur,offset);
	}
	else {
		if (trhist) 
			sprintf(tritem.history,"%s(file=%s,rate=%g,offset=%g,scaleoff=%g,scalefac=%g,history=%s)",PROGNAME,textname,1.0/fdur,offset,scaleoff,scalefac,trhist);
		else
			sprintf(tritem.history,"%s(file=%s,rate=%g,offset=%g,scaleoff=%g,scalefac=%g)",PROGNAME,textname,1.0/fdur,offset,scaleoff,scalefac);
	}

	/* open output channel */
	if ((ofid=sfschannel(filename,&tritem)) < 0)
		error("unable to open temporary file",NULL);

	/* process input file */
	while (fscanf(ip,"%f",&val) == 1) {
		val += (float)scaleoff;
		val *= (float)scalefac;
		if (sfswrite(ofid,1,&val) != 1)
			error("write error on output file",NULL);
	}

	/* update data file */
	if (!sfsupdate(filename))
		error("backup error on %s",filename);

	/* that's all folks */
	exit(0);

}
