/* wave -- extract a waveform (and other unstructured data) from a database file */

/* m.a. huckvale - april 1985 */

/* version 2 headers - december 1985 */

/* version 3.0 - february 1987
	- add output to binary files
*/
/* version 3.1 - may 1987
	- add track item output 
*/
/* version 3.2 - march 1988
	- SFS version
*/
/* version 3.3 - march 1992
	- add -h HTK format output option
*/
/* version 3.4 - May 1992
	- add -a append binary option
*/
	
#define PROGNAME "wave"
#define PROGVERS "3.4s"
char *progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH WAVE 1 UCL
.SH NAME
wave - extract waveforms and other data from a database file
.SH SYNOPSIS
.B wave
(-i item) (-b datafile|-a datafile|-h htkfile) (-s) file
.SH DESCRIPTION
.I wave
is a program to copy data from speech, lx, tx, fx and track items.  The selected data set
is either printed on the standard output as a sequence of numbers or stored in a binary
datafile.  An option reverses the byte order in binary files for waveforms.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select input item.
.TP 11
.BI -b datafile
Store the contents of the data set in the binary file
.I datafile
rather than listing them on the standard output.  SP, LX and FX items are stored in 16-bit words,
TX items in 32-bit words.
.TP 11
.BI -h htkfile
Store the contents of the data set in an HTK format binary file.  As -b option
but with standard HTK header.
.TP 11
.B -a
(Binary file output only) With -b switch or -h switch, appends data to file.
.TP 11
.B -s
(Binary file output only) swap byte order in binary file. SP, LX and FX items will be stored <low byte>
<high byte>, TX items will be stored <low byte><byte><byte><high byte>.
.SH INPUT ITEMS
.IP 1.xx 11
Any speech item.
.IP 2.xx 11
Any excitation item.
.IP 3.xx 11
Any period data item.
.IP 4.xx 11
Any fundamental frequency item.
.IP 16.xx 11
Any parameter track item.
.SH VERSION/AUTHOR
3.3 - Mark Huckvale.
*/
/*--------------------------------------------------------------------------*/
/* global structures */

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include "sfs.h"

/* global data */
struct item_header	item;
int		binary=0, swapbyte=0, htkhead=0, append=0;
char		bfilename[SFSMAXFILENAME];
struct htk_header {
	long	nsamp;
	long	nperiod;
	short	ssize;
	short	skind;
} hhead;

/* swap bytes in buffer */
void swapbuf(buf,count,datasize)
char	*buf;
int	count;
int	datasize;
{
	int	i;
	char	temp;

	for (i=0;i<count;i+=datasize) {
		if (datasize==1)
			/* nothing */;
		else if (datasize==2) {
			temp=buf[i];
			buf[i]=buf[i+1];
			buf[i+1]=temp;
		}
		else if (datasize==4) {
			temp=buf[i];
			buf[i]=buf[i+3];
			buf[i+3]=temp;
			temp=buf[i+1];
			buf[i+1]=buf[i+2];
			buf[i+2]=temp;
		}
		else
			error("incorrect data size found",NULL);
	}
}

/* store data in binary file */
void storebin(fid,item,bfilename,swapbyte,append)
int		fid;		/* input data file */
struct item_header *item;
char		*bfilename;	/* output file name */
int		swapbyte;	/* swap output byte flag */
int		append;		/* append to existing file */
{
#define COPYBUFSIZE 2048
	char	buf[COPYBUFSIZE];
	int	count,i;
	FILE	*op;

	/* open output file */
	if ((op=fopen(bfilename,(append)?"ab":"wb")) == NULL)
		error("unable to open output file %s",bfilename);

	/* write HTK header if required */
	if (htkhead) {
		hhead.nsamp = item->numframes;
		hhead.nperiod = (long) (10000000.0*item->frameduration);
		hhead.ssize = item->framesize * item->datasize;
		hhead.skind = 0;
		if (fwrite((char *)&hhead,sizeof(struct htk_header),1,op) != 1)
			error("write error on %s",bfilename);
	}
	
	/* copy data in chunks */
	for (i=0;(count=sfsread(fid,i,COPYBUFSIZE/item->datasize,buf)) > 0;i+=count) {
		if (swapbyte) swapbuf(buf,count*item->datasize,item->datasize);
		if (fwrite(buf,item->datasize,count,op) != count)
			error("write error on %s",bfilename);
	}

	/* close output file */
	fclose(op);

}

/* print data on standard output */
void printdata(fid,numframes,framesize,datasize,floating)
int		fid;		/* input data file */
int		numframes;	/* # frames of data to store */
int		framesize;	/* # datum in frame */
int		datasize;	/* size of each datum in bytes */
int		floating;	/* integer/real flag */
{
	int	i,j;
	char	*buff;
	char	*cp;
	short	*sp;
	int	*ip;
	float	*fp;

	/* get input buffer */
	buff = sfsbuffer(&item,1);

	/* print out waveform */
	for (i=0;i<numframes;i++) {
		sfsread(fid,i,1,buff);
		cp = (char *) buff;
		sp = (short *) buff;
		ip = (int *) buff;
		fp = (float *) buff;
		for (j=0;j<framesize;j++) switch (datasize) {
			case 1:	printf("%d ",*cp++);
				break;
			case 2:	printf("%d ",*sp++);
				break;
			case 4:	if (floating)
					printf("%g ",*fp++);
				else
					printf("%d ",*ip++);
				break;
		}
		printf("\n");
	}
}

/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
	/* option decoding */
	extern int	optind;
	extern char	*optarg;
	int		errflg=0;
	int		c;
	int32		it;		/* selected item type */
	char		*ty;		/* item specification */
	int		initem=SP_TYPE;
	char		*intype="0";
	/* file variables */
	char		filename[SFSMAXFILENAME];	/* data file */
	int		fid;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:b:ash:")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Waveform extraction V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'h' :	/* htk header */
			htkhead++;
		case 'b' :	/* binary file output */
			binary++;
			strcpy(bfilename,optarg);
			break;
		case 'a' :	/* append to binary file */
			append++;
			break;
		case 's' :	/* swap byte order */
			swapbyte++;
			break;
		case 'i':	/* input item specification */
			if (itspec(optarg,&it,&ty) == 0) {
				if ((it == SP_TYPE) ||
				    (it == LX_TYPE) ||
				    (it == TX_TYPE) ||
				    (it == FX_TYPE) ||
				    (it == TR_TYPE)) {
					initem = it;
					intype = ty;
				}
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: wave (-I) (-b outfile|-h htkfile) (-a) (-s) dbase_file\n",NULL);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* check for binary output */
	if (swapbyte && !binary)
		error("byte-swapping on binary files only",NULL);

	if (swapbyte && (initem==TR_TYPE))
		error("byte-swapping for waveforms only",NULL);

	/* locate start of data set */
	if ((fid=sfsopen(filename,"r",NULL)) < 0)
		error("access error on '%s'",filename);
	if (!sfsitem(fid,initem,intype,&item))
		error("cannot find input item in '%s'",filename);

	if (binary)
		storebin(fid,&item,bfilename,swapbyte,append);
	else
		printdata(fid,item.numframes,item.framesize,item.datasize,item.floating);

	/* that's all folks */
	sfsclose(fid);
	exit(0);
}

