/* log2lin - convert 8-bit u-law signal to 12-bit linear */

/* M.A.Huckvale - University College London */

/* version 1.0 - June 1994 */

#define PROGNAME "log2lin"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH LOG2LIN SFS1 UCL
.SH NAME
log2lin - convert 8-but mu-law encoded signal to 16-bit linear
.SH SYNOPSIS
.B log2lin
(-i item) file
.SH DESCRIPTION
.I log2lin
is a program to convert 8-bit mu-law (logarithmic) encoded waveforms (as found
in Sun AU format files) into linear samples.  These are shifted
up to be suitable for 16-bit replay systems.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i item
Select input item number.
.SH INPUT ITEMS
.IP SP
8-bit mu-law (logarithmic) encoded waveforms
.SH OUTPUT ITEMS
.IP SP
12-bit linear waveform.
.SH VERSION/AUTHOR
.IP 1.0
M.A.Huckvale
*/
/*--------------------------------------------------------------------------*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"

/* global data */
struct item_header	spitem;
#define SPBUFSIZE	14096
short			sp[SPBUFSIZE];
struct item_header	opitem;

/* Mu-law data is stored in sign-magnitude bit-complemented format. The
   complement was originally specified for telecommunications applications to
   increase the density of 1 bits.  After complementing the bits, the byte is
   in sign-segment-mantissa format.
   mu-law data  complement  segment step-size       value
   0 000 xxxx   1 111 yyyy     7      256     -8031, ... , -4191 
   0 001 xxxx   1 110 yyyy     6      128     -3999, ... , -2079
   0 010 xxxx   1 101 yyyy     5       64     -1983, ... , -1023
   0 011 xxxx   1 100 yyyy     4       32      -975, ... , -495
   0 100 xxxx   1 011 yyyy     3       16      -471, ... , -231
   0 101 xxxx   1 010 yyyy     2        8      -219, ... , -99
   0 110 xxxx   1 001 yyyy     1        4       -93, ... , -33
   0 111 xxxx   1 000 yyyy     0        2       -30, ... ,  0
   1 000 xxxx   0 111 yyyy     7      256      8031, ... , 4191
   1 001 xxxx   0 110 yyyy     6      128      3999, ... , 2079
   1 010 xxxx   0 101 yyyy     5       64      1983, ... , 1023
   1 011 xxxx   0 100 yyyy     4       32       975, ... , 495
   1 100 xxxx   0 011 yyyy     3       16       471, ... , 231
   1 101 xxxx   0 010 yyyy     2        8       219, ... , 99
   1 110 xxxx   0 001 yyyy     1        4        93, ... , 33
   1 111 xxxx   0 000 yyyy     0        2        30, ... , 0

  The following table implements the conversion, including the complementing
  operation.
*/

static short utab[256] = {
  -8031, -7775, -7519, -7263, -7007, -6751, -6495, -6239,
  -5983, -5727, -5471, -5215, -4959, -4703, -4447, -4191,
  -3999, -3871, -3743, -3615, -3487, -3359, -3231, -3103,
  -2975, -2847, -2719, -2591, -2463, -2335, -2207, -2079,
  -1983, -1919, -1855, -1791, -1727, -1663, -1599, -1535,
  -1471, -1407, -1343, -1279, -1215, -1151, -1087, -1023,
   -975,  -943,  -911,  -879,  -847,  -815,  -783,  -751,
   -719,  -687,  -655,  -623,  -591,  -559,  -527,  -495,
   -471,  -455,  -439,  -423,  -407,  -391,  -375,  -359,
   -343,  -327,  -311,  -295,  -279,  -263,  -247,  -231,
   -219,  -211,  -203,  -195,  -187,  -179,  -171,  -163,
   -155,  -147,  -139,  -131,  -123,  -115,  -107,   -99,
    -93,   -89,   -85,   -81,   -77,   -73,   -69,   -65,
    -61,   -57,   -53,   -49,   -45,   -41,   -37,   -33,
    -30,   -28,   -26,   -24,   -22,   -20,   -18,   -16,
    -14,   -12,   -10,    -8,    -6,    -4,    -2,     0,
   8031,  7775,  7519,  7263,  7007,  6751,  6495,  6239,
   5983,  5727,  5471,  5215,  4959,  4703,  4447,  4191,
   3999,  3871,  3743,  3615,  3487,  3359,  3231,  3103,
   2975,  2847,  2719,  2591,  2463,  2335,  2207,  2079,
   1983,  1919,  1855,  1791,  1727,  1663,  1599,  1535,
   1471,  1407,  1343,  1279,  1215,  1151,  1087,  1023,
    975,   943,   911,   879,   847,   815,   783,   751,
    719,   687,   655,   623,   591,   559,   527,   495,
    471,   455,   439,   423,   407,   391,   375,   359,
    343,   327,   311,   295,   279,   263,   247,   231,
    219,   211,   203,   195,   187,   179,   171,   163,
    155,   147,   139,   131,   123,   115,   107,    99,
     93,    89,    85,    81,    77,    73,    69,    65,
     61,    57,    53,    49,    45,    41,    37,    33,
     30,    28,    26,    24,    22,    20,    18,    16,
     14,    12,    10,     8,     6,     4,     2,     0
};

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item selection */
	char		*ty;		/* item sub type */
	char		*sptype="0";

	/* file variables */
	char		filename[SFSMAXFILENAME]; /* SFS data file name */
	int		fid;		/* input file descriptor */
	int		i,pos,len;
	int		ofid;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Convert mu-law to 12-bit linear V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					sptype = ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open file */
	if ((fid=sfsopen(filename,"w",NULL)) < 0)
		error("access error on '%s'",filename);

	/* find item */
	if (!sfsitem(fid,SP_TYPE,sptype,&spitem))
		error("could not find input SP item in '%s'",filename);

	/* create output item header */
	sfsheader(&opitem,SP_TYPE,0,2,1,spitem.frameduration,spitem.offset,1,0,0);
	sprintf(opitem.history,"%s(%d.%02d)",
		PROGNAME,
		spitem.datatype,spitem.subtype);

	/* create output item */
	if ((ofid=sfschannel(filename,&opitem))<0)
		error("could not open output channel to '%s'",filename);

	/* process */
	pos=0;
	while ((len=sfsread(fid,pos,SPBUFSIZE,sp))>0) {
		for (i=0;i<len;i++) 
			sp[i] = utab[((unsigned)sp[i]&0xFF)] << 1;
		sfswrite(ofid,len,sp);
		pos += len;
	}

	/* update */
	if (!sfsupdate(filename))
		error("update fails on '%s'",filename);

	/* that's all folks */
	exit(0);
}
