/* spancat - Speech and Annotation concatenation */

/* M.A.Huckvale - University College London */

/* version 1.0 - February 1995 */

#define PROGNAME "spancat"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH SPANCAT SFS1 UCL
.SH NAME
spancat - concatenate speech and annotation items
.SH SYNOPSIS
.B spancat
(-f control-file) (-a) outfile
.SH DESCRIPTION
.I spancat
is a program to concatenate a number of speech and annotation items
from different files into a single speech and annotation item in an
output file.  The control file consists of one line per file to process
giving the filename, the speech item and the annotation item.  If the
item numbers are not given, the last appropriate item in the file
is chosen.  Speech items must match in sampling rate.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.B -a
Allow files with missing annotation items.  Default is to stop with error.
.SH INPUT ITEMS
.IP SP 11
Speech items
.IP AN 11
Annotation items
.SH OUTPUT ITEMS
.IP SP 11
Concatenated Speech.
.IP AN 11
Concatenated annotations.
.SH HISTORY
.IP file=control
Control file name.
.SH VERSION/AUTHOR
.IP 1.0
Mark Huckvale
*/
/*--------------------------------------------------------------------------*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <malloc.h>
#include <string.h>
#include "sfs.h"

/* global data */
struct item_header spitem;
struct item_header anitem;
struct item_header ospitem;
struct item_header oanitem;
struct an_rec	*an;

int	anmisserr=1;
char	cfilename[128];
char	fname[128];
char	*sptype;
char	*antype;
int	outrate=0;

#define SPCONVERT(ans)	(int)(((ans)*anitem.frameduration)/ospitem.frameduration)

/* read a line from control file */
int	getcontrol(ip,fname,sptype,antype)
FILE	*ip;
char	*fname;
char	**sptype;
char	**antype;
{
	int	count;
	char	line[256];
	static char	*zero="0";
	static char	it1[80],it2[80];
	int	it;

	if (!fgets(line,256,ip)) return(0);
	count = sscanf(line,"%s %s %s\n",fname,it1,it2);
	switch (count) {
		case 0:
			return(0);
		case 1:
			*sptype = zero;
			*antype = zero;
			return(1);
		case 2:
			*antype = zero;
			if ((itspec(it1,&it,sptype)==0) && (it==SP_TYPE))
				return(1);
			else
				error("bad SP item specification in control file");
		case 3:
			if ((itspec(it1,&it,sptype)==0) && (it==SP_TYPE)) {
				if ((itspec(it2,&it,antype)==0) && (it==AN_TYPE))
					return(1);
				else
					error("bad AN item specification in control file");
			}
			else
				error("bad SP item specification in control file");
	}
	return(0);
}
			
/* copy signal */
short	xferbuff[4096];
int	copysp(ifd,ofd,cnt)
int	ifd,ofd;
int	cnt;
{
	int	len;
	int	pos=0;

	while (cnt > 0) {
		len = sfsread(ifd,pos,(cnt>4096)?4096:cnt,xferbuff);
		sfswrite(ofd,len,xferbuff);
		pos += len;
		cnt -= len;
	}
	return(pos);
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */

	/* file variables */
	char		filename[SFSMAXFILENAME]; /* SFS data file name */
	int		fid;		/* input file descriptor */
	int		sfid=0;		/* output file descriptor */
	int		afid=0;		/* output file descriptor */
	FILE		*ip;
	int		splen,totsamp=0,anlast;
	int		inrate;
	int		i;
	struct an_rec	*anfill=NULL;

	/* decode switches */
	while ( (c = getopt(argc,argv,"If:a")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Concatenate speech signals with annotations V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'f' :
			strcpy(cfilename,optarg);
			break;
		case 'a' :
			anmisserr=0;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-a) (-f controlfile) sfsfile",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open control file */
	if (cfilename[0]) {
		if ((ip=fopen(cfilename,"r"))==NULL)
			error("could not open '%s'",cfilename);
	}
	else
		ip = stdin;

	/* get control items */
	while (getcontrol(ip,fname,&sptype,&antype)) {
printf("fname=%s sptype=%s antype=%s\n",fname,sptype,antype);
		/* open file */
		if ((fid=sfsopen(fname,"r",NULL)) < 0)
			error("could not open '%s'",fname);
		/* read in annotation */
		if (!sfsitem(fid,AN_TYPE,antype,&anitem)) {
			if (anmisserr)
				error("could not find annotation item in '%s'",fname);
			else
				an = NULL;
		}
		else {
			an = (struct an_rec *)sfsbuffer(&anitem,anitem.numframes);
			sfsread(fid,0,anitem.numframes,an);
		}
		/* locate speech */
		if (!sfsitem(fid,SP_TYPE,sptype,&spitem))
			error("could not find speech item in '%s'",fname);
		if (outrate) {
			inrate = (int)(0.5 + 0.1/spitem.frameduration);
			if (inrate != outrate)
				error("sample rate doesn't match in '%s'",fname);
		}
		else
			outrate = (int)(0.5 + 0.1/spitem.frameduration);

		/* if first time through, open output channel */
		if (sfid <= 0) {
			sfsheader(&ospitem,SP_TYPE,0,2,1,
					spitem.frameduration,0.0,1,0,0);
			sprintf(ospitem.history,"%s(file=%s)",
				PROGNAME,cfilename);
			if ((sfid=sfschannel(filename,&ospitem)) < 0)
				error("could not open output channel to '%s'",filename);
		}
		if (afid <= 0) {
			sfsheader(&oanitem,AN_TYPE,-1,1,-1,
					spitem.frameduration,0.0,0,0,1);
			sprintf(oanitem.history,"%s(file=%s)",
				PROGNAME,cfilename);
			if ((afid=sfschannel(filename,&oanitem)) < 0)
				error("could not open output channel to '%s'",filename);
			anfill = sfsbuffer(&oanitem,1);
			if (totsamp > 0) {
				anfill->posn = 0;
				anfill->size = totsamp;
				strcpy(anfill->label,"FILLER");
				sfswrite(afid,1,anfill);
			}
		}

		/* copy speech */
		splen = copysp(fid,sfid,spitem.numframes);

		if (an) {
			/* convert timing of annotations */
			for (i=0;i<anitem.numframes;i++) {
				an[i].posn = totsamp + SPCONVERT(an[i].posn);
				an[i].size = SPCONVERT(an[i].size);
			}
			/* check annotations */
			while ((anitem.numframes>0) && (an[anitem.numframes-1].posn >= (totsamp+splen)))
				anitem.numframes--;
			/* correct length of last annotation */
			if (anitem.numframes > 0) {
				an[anitem.numframes-1].size = totsamp+splen-an[anitem.numframes-1].posn;
				anlast = totsamp+splen;
			}
			else
				anlast = totsamp;
			/* write annotations */
			sfswrite(afid,anitem.numframes,an);
		}
		else
			anlast = totsamp;
		/* correct for any remaining An gap */
		if (anlast < (totsamp+splen)) {
			anfill->posn = totsamp;
			anfill->size = splen;
			strcpy(anfill->label,"FILLER");
			sfswrite(afid,1,anfill);
		}
		totsamp += splen;
		free(an);
		sfsclose(fid);
	}

	if (!sfsupdate(filename))
		error("update error on '%s'",filename);

	/* that's all folks */
	exit(0);
}

