/* harsyn -- interactive harmonic synthesis, display and replay */

/* M.A.Huckvale - October 1990 */

/* version 1.0 */

/* version 1.1 - October 1991
	- real valued amplitudes
	- examples by hidden commands
	- auto re-scaling
*/

/* version 1.2 - November 1993
	- bug fixes
*/
/* version 1.3 - October 1996
	- scaling to unity
	- logarithmic mode in decibels
*/
	
#define PROGNAME "harsyn"
#define PROGVERS "1.3"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH HARSYN UCL1 UCL SFS
.SH NAME
harsyn -- interactive harmonic synthesis and replay
.SH SYNOPSIS
.B harsyn
(-I) (-p) (-i initfile) (-s synthfile) (-t print_title)
.SH DESCRIPTION
.I harsyn
is a program to demonstrate harmonic synthesis of periodic waveforms.
The user can select harmonics by frequency, amplitude and phase, watch
the displayed wave shape and listen to the generated tone.  When executed
without the '-p' or '-s' options, the program is operated interactively
from the terminal with a mouse.  The '-p' option directs the initial
harmonic set-up (see '-i' option) directly to the printer.  The '-s' option
directs the generated waveform into an SFS file.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.B -p
Direct output to printer.  The program does not enter interactive mode.  The
screen image is recreated for the printer using an initial list of harmonics.
This can be set up from an initialisation file using the '-i' option.
.TP 11
.BI -i initfile
Specify the initial loading of the harmonics.  The format of the text file
must be the same as the format produced by the programs SAVE command (see below).
This is one line with the setting of the LOG and ANHARMONIC flags (0=off, 1=on), 
followed by the frequency, amplitude and phase of the harmonics required.
.TP 11
.BI -s synthfile
Specify that a generated 1 second tone of the default or loaded harmonics should
be saved into the givcen SFS file.  The program does not enter interactive mode.
.TP 11
.BI -t print_title
Specify a title to appear on the printout.  If no title is given then the
date and time is printed instead.
.SH "INTERACTIVE MODE"
In the interactive mode, harmonics may be added and moved on the upper amplitude
spectrum, causing the lower amplitude trace to be re-drawn.  Select a harmonic
by pointing to it and clicking the left mouse button.  To move it, point to its new position
and click the button again.  To delete a harmonic, select it then click off the amplitude
graph.  To add a new harmonic click on the NEW harmonic, then click on the position in
the graph where it is to be added.  To replay the waveform, press the middle mouse button.
To adjust the phases of the first four components, select one phase display then select
a phase value.
.SS "MAIN MENU"
.IP clear 11
Clear the current set of harmonics and restore the initial/default values.
.IP magic 11
Go to the Examples menu.
.IP save 11
Save the list of harmonics and values to a text file.
.IP options 11
Go to the Options menu.
.IP print 11
Send the current screen contents to the printer.
.IP quit 11
Stop the program.
.IP refresh 11
Re-draw the screen with current harmonics unchanged.
.IP scale 11
Adjust the harmonic amplitudes so that the peak amplitude of the generated wave is
equal to 1.0.
.SS "OPTIONS MENU"
.IP anharmonic 11
Allow harmonics to be placed at any frequency.
.IP harmonic 11
Only allow harmonics to be placed at multiples of 200Hz.
.IP logarithmic 11
Use a logarithmic amplitude scale for the harmonics.
.IP linear 11
Use a linear amplitude scale for the harmonics.
.IP components 11
Display on the amplitude waveform the individual sine-wave components.
.IP main 11
Go to the Main menu.
.SS "EXAMPLES MENU"
.IP square 11
Set up the harmonics for a square wave.
.IP triangle 11
Set up the harmonics for a triangular wave.
.IP i-vowel
Set up the harmonics for an /i/ like vowel sound.
.IP a-vowel
Set up the harmonics for an /a/ like vowel sound.
.IP u-vowel
Set up the harmonics for an /u/ like vowel sound.
.IP main 11
Go to the Main menu.
.SH VERSION/AUTHOR
.IP 1.0
Mark Huckvale
.SH BUGS
Fine control with the mouse is difficult.
*/
/*--------------------------------------------------------------------------*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <math.h>
#include <string.h>
#include <ctype.h>
#include "sfs.h"
#include "dig.h"
#include "digdata.h"
#include "harsyn.h"

/* constants */
#ifndef PI
#define PI	3.14159265358979323846
#endif
#define TWOPI	6.28318530717958647692
#define DEGTORAD 0.017453292

/* harmonic configuration */
#define MAXHARMONIC	25
#define MAXFREQ		3000.0
#define MAXAMP		100.0
#define NUMPHASE	4		/* display first few phase only */

/* waveform constants */
#define FUNDFREQ	200.0		/* default fundamental = 200Hz */
#define WAVETIME	0.01		/* display 10 ms of signal */

/* replay constants */
#define SAMPFREQ	10000.0		/* sampling frequency */
#define SBUFSIZE	60000		/* sine table size */
#define RBUFSIZE	10000		/* replay buffer size */
#define DBUFSIZE	3000		/* display buffer maximum size */
#define TAPER		500		/* replay taper to avoid clicks */
#define SINESCALE	8		/* binary floating point */

/* graphics configuration */
float SCREEN_WIDTH;
#define SCREEN_HEIGHT	(3*MAXAMP+200.0)
float FREQ_ORIGIN_X;
#define FREQ_ORIGIN_Y	350.0
float FREQ_AXIS_X;
#define FREQ_AXIS_Y	(FREQ_ORIGIN_Y+MAXAMP)
#define PHASE_ORIGIN_Y	280.0
#define PHASE_HEIGHT	20.0
#define PHASE_WIDTH	(MAXFREQ/(2*NUMPHASE))
#define AMP_ORIGIN_X	FREQ_ORIGIN_X
#define AMP_ORIGIN_Y	110.0
#define AMP_AXIS_X	FREQ_AXIS_X
#define AMP_AXIS_Y	(AMP_ORIGIN_Y+MAXAMP)
#define AMP_AXIS_L	(AMP_ORIGIN_Y-MAXAMP)

/* graphics constants */
#define OFF		0		/* black line */
#define DULL		20		/* red line */
#define BGLINE		81		/* light-green line */
#define ON		22		/* light-blue line */
#define BRIGHT		24		/* white line */
int	SELECT=28;			/* selected colour */
#define inarea(x,y,xl,yb,xr,yt) (((x) >= (xl)) && ((x) < (xr)) && ((y) >= (yb)) && ((y) < (yt)))

/* commands */
#define COM_NULL	0
#define COM_QUIT	1
#define COM_HARM	2
#define COM_NEW_HARM	3
#define COM_PHASE	4
#define COM_REPLAY	5
#define COM_CLEAR	6
#define COM_SCALE	7
#define COM_ANHARM	8
#define COM_DOHARM	9
#define COM_REFRESH	10
#define COM_COMPO	11
#define COM_SAVE	12
#define COM_LOG		13
#define COM_LIN		14
#define COM_OPTION	15
#define COM_MAIN	16
#define COM_EXAMPLE	17
#define COM_SQUARE	18
#define COM_TRIANGLE	19
#define COM_IVOWEL	20
#define COM_AVOWEL	21
#define COM_UVOWEL	22
#define COM_PRINT	23
#define COM_AUTODRAW	24
#define COM_MANDRAW	25

/* harmonic status table */
struct harmonic_rec {
	int	freq;			/* range 1..1000 */
	int	phase;			/* range 0..360 */
	float	amp;			/* floating point amplitude, range 0..1 */
} htab[MAXHARMONIC];
float	maxamp;				/* max amplitude of resultant wave */

/* control data */
int	anharmonic=1;			/* allow non-harmonically related lines */
int	logscale=0;			/* amplitude is log scaled */
int	mandraw=0;			/* automatic re-draw of waveform */
char	initfile[SFSMAXFILENAME];			/* initialisation file */
int	laser=0;			/* laser - one shot to printer */
int	sfs=0;				/* SFS - do one shot to SFS file */
char	laser_title[80];		/* title for laser output */

/* SFS interface */
char	sfsfilename[SFSMAXFILENAME];		/* SFS filename */
struct item_header spitem;		/* output item header */

/* data tables */
int	sinetab[SBUFSIZE];		/* sine table */
short	repbuff[RBUFSIZE];		/* replay buffer */
short	dspbuff[DBUFSIZE];		/* display buffer */
int	wavepoints;			/* # points in display buffer */
float	charwidth,charheight;		/* character width and height in screen units */

/* forward refernces */
double	flogcode();
double	flogdecode();

char *logname()
{
	static char *UNKNOWN="unknown";
	char *p,*getenv();
	if ((p=getenv("LOGNAME")))
		return(p);
	else
		return(UNKNOWN);
}
#ifdef DOS
char	*ttyname(int n)
{
	static char *term="PC";
	return(term);
}
#endif

static double mylog10(val)
double val;
{
	return(log(val)/log(10.0));
}
static double mypow10(val)
double val;
{
	return(pow(10.0,val));
}

/* log encoding of amplitude */
double flogcode(val)
double	val;
{
	if (val==0.0)
		return(0.0);
	else {
		val = (20.0*mylog10(val)+40)/40;
		if (val < 0) val=0;
		return(val);
	}
}
double flogdecode(val)
double	val;
{
	if (val==0.0)
		return(0.0);
	else
		return(mypow10((40*val-40)/20.0));
}

void beep()
{
	putchar('\007');
	fflush(stdout);
}

void loadharmstand()
{
	int	i;

	htab[0].freq = (int)FUNDFREQ;
	htab[0].phase = 0;
	htab[0].amp = 1.0;
	for (i=1;i<MAXHARMONIC;i++) {
		htab[i].freq = (int)((i+1)*FUNDFREQ);
		htab[i].phase = 0;
		htab[i].amp = 0.0;
	}
}

void loadharmfile(fname)
char	*fname;
{
	FILE	*ip;
	int	i=0;

	if ((ip=fopen(fname,"r"))==NULL)
		error("unable to open '%s'",fname);

	if (fscanf(ip,"%d %d\n",&anharmonic,&logscale)==2) {

		for (i=0;i<MAXHARMONIC;i++)
			if (fscanf(ip,"%d %f %d\n",&htab[i].freq,&htab[i].amp,&htab[i].phase)!=3)
				break;
	}

	for (;i<MAXHARMONIC;i++) {
		htab[i].freq = (int)((i+1)*FUNDFREQ);
		htab[i].amp = 0.0;
		htab[i].phase = 0;
	}

	fclose(ip);
}

void saveharmfile()
{
	char	filename[SFSMAXFILENAME];
	char	messg[80];
	FILE	*op;
	float	x,y;
	char	b;
	int	i;

	digprompt("Enter filename to save waveform: ");

	diggetmouse(filename,&b,&x,&y);

	if (filename[0] && (filename[0]!='\n')) {
		if ((op=fopen(filename,"w"))==NULL) {
			sprintf(messg,"Unable to open '%s'",filename);
			digprompt(messg);
			beep();
			sleep(2);
			return;
		}
		fprintf(op,"%d %d\n",anharmonic,logscale);
		for (i=0;i<MAXHARMONIC;i++)
			fprintf(op,"%d %f %d\n",htab[i].freq,htab[i].amp,htab[i].phase);
		fclose(op);
	}		
}

void hardcopy()
{
	char	filename[SFSMAXFILENAME];
	char	comstr[256];
	FILE	*op;
	float	x,y;
	char	b;
	int	i;

	digprompt("Enter title for print: ");

	diggetmouse(laser_title,&b,&x,&y);

	if (laser_title[0]) {
		digprompt("Printing ..");
		strcpy(filename,"/tmp/harsyn.XXXXXX");
		mktemp(filename);
		if ((op=fopen(filename,"w"))==NULL) {
			digprompt("Unable to open temporary file");
			beep();
			sleep(2);
			return;
		}
		fprintf(op,"%d %d\n",anharmonic,logscale);
		for (i=0;i<MAXHARMONIC;i++)
			fprintf(op,"%d %f %d\n",htab[i].freq,htab[i].amp,htab[i].phase);
		fclose(op);

#ifdef DOS
		sprintf(comstr,"harsyn -p -i %s -t \"%s\"",
				filename,laser_title);
#else
		sprintf(comstr,"(harsyn -p -i %s -t \"%s\";rm %s)&",
				filename,laser_title,filename);
#endif
		system(comstr);
#ifdef DOS
		sprintf(comstr,"del %s",filename);
		system(comstr);
#endif
		sleep(2);
		remove(filename);
	}
}

/* draw amplitude scale on frequency graph */
void drawfscale()
{
	int	amp;
	float	y;
	char	messg[16];

	if (logscale) {
		amp = -40;
		while (amp <= 0) {
			y = FREQ_ORIGIN_Y + MAXAMP + 2.5*amp;
			sprintf(messg,"%3d",amp);
			if ((amp % 10)==0) {
				digline(DULL,FREQ_ORIGIN_X,y,FREQ_ORIGIN_X-charwidth,y);
				digtext(DULL,FREQ_ORIGIN_X-9*charwidth/2,y-charheight/2,messg);
				if (y != FREQ_ORIGIN_Y)
					digline(BGLINE,FREQ_ORIGIN_X,y,FREQ_AXIS_X,y);
			}
			else
				digline(DULL,FREQ_ORIGIN_X,y,FREQ_ORIGIN_X-charwidth/2,y);
			amp += 5;
		}
		digtext(DULL,charwidth,FREQ_AXIS_Y+charheight,"Amp (dB)");
	}
	else {
		amp = 0;
		while (amp <= MAXAMP) {
			y = FREQ_ORIGIN_Y + amp;
			sprintf(messg,"%3.1f",(double)amp/100.0);
			if ((amp % 20)==0) {
				digline(DULL,FREQ_ORIGIN_X,y,FREQ_ORIGIN_X-charwidth,y);
				digtext(DULL,FREQ_ORIGIN_X-9*charwidth/2,y-charheight/2,messg);
				if (y != FREQ_ORIGIN_Y)
					digline(BGLINE,FREQ_ORIGIN_X,y,FREQ_AXIS_X,y);
			}
			else
				digline(DULL,FREQ_ORIGIN_X,y,FREQ_ORIGIN_X-charwidth/2,y);
			amp += 10;
		}
		digtext(DULL,charwidth,FREQ_AXIS_Y+charheight,"Amplitude");
	}
}

/* draw harmonic */
void drawharmonic(h,colour)
int	h;
int	colour;
{
	float	x,y;
	char	messg[8];

	if (h < 0) {
		/* draw harmonic prototype */
		x = (SCREEN_WIDTH+FREQ_AXIS_X)/2.0;
		y = FREQ_ORIGIN_Y + 2 * charheight;
		digline(colour,x,y,x,y+75);
		digtext(colour,x-digdata.chwidth/digdata.xscale,y+75,"NEW");
	}
	else if (htab[h].amp > 0) {
		x = FREQ_ORIGIN_X + htab[h].freq;
		y = FREQ_ORIGIN_Y + 100*htab[h].amp;
		if (x <= (FREQ_AXIS_X+digdata.chwidth/digdata.xscale)) {
			digline(colour,x,FREQ_ORIGIN_Y+0.5/digdata.yscale,x,y);
			if (anharmonic)
				sprintf(messg,"%d",h+1);
			else
				sprintf(messg,"%d",(int)(0.5+htab[h].freq/FUNDFREQ));
			digtext(colour,x-charwidth/2,y,messg);
		}
	}
	digflush();
}

/* graph of sine wave for phase picture */
#define NUMSINEPIC	16
float sinepic[NUMSINEPIC+1];
int	lenpic;

void drawphase(hnum,colour)
int	hnum;
int	colour;
{
	int	i;
	float	orgx,stepx;
	float	x,y,lastx,lasty;
	char	messg[20];

	/* first time, calculate picture */
	if (lenpic==0) {
		for (i=0;i<=NUMSINEPIC;i++)
			sinepic[i] = PHASE_ORIGIN_Y + PHASE_HEIGHT * sin(TWOPI*(float)i/(float)NUMSINEPIC);
		lenpic = NUMSINEPIC+1;
	}

	/* get co-ordinates */
	orgx = FREQ_ORIGIN_X + hnum * 2 * PHASE_WIDTH;
	stepx = PHASE_WIDTH/NUMSINEPIC;

	/* draw graph */
	digfillbox(0,orgx,PHASE_ORIGIN_Y-PHASE_HEIGHT-charheight,
			orgx+2*PHASE_WIDTH-2.0/digdata.xscale,PHASE_ORIGIN_Y+PHASE_HEIGHT+charheight);

	if (htab[hnum].amp > 0) {
		x = lastx = orgx;
		lasty = sinepic[0];
		for (i=1;i<lenpic;i++) {
			x += stepx;
			y = sinepic[i];
			digline(colour,lastx,lasty,x,y);
			lastx = x;
			lasty = y;
		}

		/* draw phase marker */
		x = orgx + htab[hnum].phase * PHASE_WIDTH / 360.0;
		i = (int)(0.5+(float)NUMSINEPIC*htab[hnum].phase/360.0);
		i = (i < 0) ? 0 : i;
		i = (i > NUMSINEPIC) ? NUMSINEPIC : i;
		y = sinepic[i];
		digline(colour,x,y-charheight/2,x,y+charheight/2);

		/* draw title */
		if (anharmonic)
			sprintf(messg,"Phase %d",hnum+1);
		else
			sprintf(messg,"Phase %d",(int)(0.5+htab[hnum].freq/FUNDFREQ));
		digtext(colour,orgx+PHASE_WIDTH+2.0/digdata.xscale,PHASE_ORIGIN_Y-charheight/2,messg);
	}
	digflush();
}

/* draw complete frequency graph */
void drawfrequency()
{
	int	i,freq;
	float	x;
	char	messg[16];

	digfillbox(0,0.0,FREQ_ORIGIN_Y-3*charheight,FREQ_AXIS_X+2*charwidth,SCREEN_HEIGHT);
	digline(DULL,FREQ_ORIGIN_X,FREQ_ORIGIN_Y,FREQ_AXIS_X,FREQ_ORIGIN_Y);
	digline(DULL,FREQ_ORIGIN_X,FREQ_ORIGIN_Y,FREQ_ORIGIN_X,FREQ_AXIS_Y);
	freq = 0;
	while (freq <= MAXFREQ) {
		x = FREQ_ORIGIN_X + freq;
		sprintf(messg,"%4d",freq);
		if ((freq % 1000)==0) {
			digline(DULL,x,FREQ_ORIGIN_Y,x,FREQ_ORIGIN_Y-charheight);
			digtext(DULL,x-2*charwidth,
				FREQ_ORIGIN_Y-9*charheight/4,messg);
		}
		else if ((freq % 500)==0) {
			digline(DULL,x,FREQ_ORIGIN_Y,x,FREQ_ORIGIN_Y-2*charheight/3);
			digtext(DULL,x-2*charwidth,
				FREQ_ORIGIN_Y-9*charheight/4,messg);
		}
		else
			digline(DULL,x,FREQ_ORIGIN_Y,x,FREQ_ORIGIN_Y-charheight/3);
		freq += 100;
	}
	drawfscale();
	for (i=0;i<MAXHARMONIC;i++)
		drawharmonic(i,ON);
	if (!laser) drawharmonic(-1,ON);
	digtext(DULL,FREQ_AXIS_X+charwidth,FREQ_ORIGIN_Y-charheight/2,"Freq (Hz)");
}

/* generate waveform from harmonics */
void genwave(buff,len,hstart,hend,ffactor)
short	*buff;
int	len;
int	hstart,hend;
double	ffactor;
{
	int		i;
	register int	j;
	register int	fp,fstep;
	register short	*sp;
	int		amp;
	
	if (digdata.setup && !laser)
		digprompt("Calculating ..");

	/* clear buffer */
	memset((char *)buff,0,len*sizeof(short));

	/* over each harmonic in turn */
	for (i=hstart;i<hend;i++) if (htab[i].amp > 0) {

		/* start at beginning of buffer */
		sp = buff;

		/* get frequency step */
		fstep = (int)(0.49 + (float)SBUFSIZE*(float)htab[i].freq*ffactor/SAMPFREQ);

		/* get phase offset */
		fp = (int)(0.5+(float)SBUFSIZE*htab[i].phase/360.0);

		/* get amplitude */
		if (logscale)
			amp = (int)(0.5+100*flogdecode(htab[i].amp));
		else
			amp = (int)(0.5+100*htab[i].amp);

		/* add in harmonic */
		for (j=0;j<len;j++,sp++) {
			*sp += (amp * sinetab[fp]) >> SINESCALE;
			fp = (fp + fstep)%SBUFSIZE;
		}	
	}

	/* do taper */
	if (buff==repbuff) {
		sp = buff;
		for (i=0;i<TAPER;i++,sp++)
			*sp = (short)(*sp * ((float)i/(float)TAPER));
		sp = buff+len-1;
		for (i=0;i<TAPER;i++,sp--)
			*sp = (short)(*sp * ((float)i/(float)TAPER));
	}
}

/* draw amplitude scale on amplitude graph */
void drawascale()
{
	int	amp;
	float	y;
	char	messg[16];

	amp = (int)-MAXAMP;
	while (amp <= MAXAMP) {
		y = AMP_ORIGIN_Y + amp;
		sprintf(messg,"%3.1f",amp/50.0);
		if ((amp % 20)==0) {
			digline(DULL,AMP_ORIGIN_X,y,AMP_ORIGIN_X-charwidth,y);
			digtext(DULL,AMP_ORIGIN_X-11*charwidth/2,y-charheight/2,messg);
		}
		else
			digline(DULL,AMP_ORIGIN_X,y,AMP_ORIGIN_X-charwidth/2,y);
		amp += 10;
	}
	digtext(DULL,charwidth,AMP_AXIS_Y+charheight,"Amplitude");
}

/* draw amplitude */
void drawamplitude(colour)
int	colour;
{
	int	i,numtick;
	float	lastx=0.0,lasty=0.0,x,y;

	if (colour >= 0) {
		if (!laser) digprompt("Drawing wave ..");
		digfillbox(0,AMP_ORIGIN_X,AMP_AXIS_L,AMP_AXIS_X,AMP_AXIS_Y);
		digline(DULL,AMP_ORIGIN_X,AMP_ORIGIN_Y,AMP_AXIS_X,AMP_ORIGIN_Y);
		digline(DULL,AMP_ORIGIN_X,AMP_AXIS_L,AMP_ORIGIN_X,AMP_AXIS_Y);
		digclip(0.0,AMP_AXIS_L,AMP_AXIS_X,AMP_AXIS_Y);
		drawascale();
		numtick = (int)(0.5+WAVETIME/0.001);
		for (i=1;i<=numtick;i++) {
			x = AMP_ORIGIN_X + (float)i * (AMP_AXIS_X-AMP_ORIGIN_X) * 0.001 / WAVETIME;
			digline(DULL,x,AMP_ORIGIN_Y-charheight/2,x,AMP_ORIGIN_Y+charheight/2);
		}
		digtext(DULL,AMP_AXIS_X+charwidth,AMP_ORIGIN_Y-charheight/2,"Time (ms)");
	}

	maxamp=0.0;
	for (i=0;i<wavepoints;i++) {
		x = AMP_ORIGIN_X + 2.0*(float)i/digdata.xscale;
		y = (double)((int)dspbuff[i] << SINESCALE)/(32767.0*2.0);
		if ((y > maxamp) || (y < -maxamp)) maxamp = fabs(y);
		y += AMP_ORIGIN_Y;
		if (i && (colour>=0)) digline(colour,lastx,lasty,x,y);
		lastx = x;
		lasty = y;
	}

	digclip(0.0,0.0,SCREEN_WIDTH,SCREEN_HEIGHT);
	if (colour==0) digline(DULL,AMP_ORIGIN_X,AMP_ORIGIN_Y,AMP_AXIS_X,AMP_ORIGIN_Y);
	digflush();
}

/* draw components */
void drawcomponents(colour)
int	colour;
{
	int	i,j;
	float	lastx=0.0,lasty=0.0,x,y;

	digclip(AMP_ORIGIN_X,AMP_AXIS_L,AMP_AXIS_X,AMP_AXIS_Y);

	for (j=0;j<MAXHARMONIC;j++) if (htab[j].amp > 0) {

		genwave(dspbuff,wavepoints,j,j+1,SAMPFREQ*WAVETIME/wavepoints);

		for (i=0;i<wavepoints;i++) {
			x = AMP_ORIGIN_X + 2.0*(float)i/digdata.xscale;
			y = (double)((int)dspbuff[i] << SINESCALE)/(32767.0*2.0);
			if ((y > maxamp) || (y < -maxamp)) maxamp = fabs(y);
			y += AMP_ORIGIN_Y;
			if (i && colour) digline(colour,lastx,lasty,x,y);
			lastx = x;
			lasty = y;
		}

	}
	genwave(dspbuff,wavepoints,0,MAXHARMONIC,SAMPFREQ*WAVETIME/wavepoints);
	digclip(0.0,0.0,SCREEN_WIDTH,SCREEN_HEIGHT);
	digflush();
}

int stringcomp(s,t,l)
char	*s;
char	*t;
int	l;
{
	int	c=0;

	while ((l-- > 0) && ((c=(toupper(*t)-toupper(*s)))==0)) {
		s++;
		t++;
	}
	return(c);
}

/* command table */
struct command_rec {
	char	*comname;
	int	comval;
	int	menu;
} comtab[]={
	/* main menu */
	{ "clear",	COM_CLEAR,	0 },
	{ "components",	COM_COMPO,	0 },
	{ "magic",	COM_EXAMPLE,	0 },
	{ "scale",	COM_SCALE,	0 },
	{ "save",	COM_SAVE,	0 },
	{ "options",	COM_OPTION,	0 },
	{ "print",	COM_PRINT,	0 },
	{ "quit",	COM_QUIT,	0 },
	{ "refresh",	COM_REFRESH,	0 },
	/* options menu */
	{ "anharmonic",	COM_ANHARM,	1 },
	{ "harmonise",	COM_DOHARM,	1 },
	{ "logarithmic", COM_LOG,	1 },
	{ "linear",	COM_LIN,	1 },
	{ "auto-draw",	COM_AUTODRAW,	1 },
	{ "manual-draw", COM_MANDRAW,	1 },
	/* examples menu */
	{ "square",	COM_SQUARE,	2 },
	{ "triangle",	COM_TRIANGLE,	2 },
	{ "ivowel",	COM_IVOWEL,	2 },
	{ "avowel",	COM_AVOWEL,	2 },
	{ "uvowel",	COM_UVOWEL,	2 },
	{ "i-vowel",	COM_IVOWEL,	2 },
	{ "a-vowel",	COM_AVOWEL,	2 },
	{ "u-vowel",	COM_UVOWEL,	2 },

};
#define NUMCOMMAND (sizeof(comtab)/sizeof(struct command_rec))

int getcommand(h)
int	*h;
{
	int	i,l;
	int	com=COM_NULL;
	float	xm,ym;
	int	freq,amp,idx,dst;
	char	but;
	char	string[80];
	int	menu=0;

	*h=0;
	do {
		switch (menu) {
		case 0:		/* top menu */
			digprompt("Main: Clear, Components, Options, Refresh, Print, Save, Scale, Quit: ");
			break;
		case 1:		/* options menu */
			sprintf(string,"Options: %s, %s, %s: ",
					(anharmonic)?"Harmonise":"Anharmonic",
					(logscale)?"Linear":"Logarithmic",
					(mandraw)?"Auto-Draw":"Manual-Draw"
					);
			digprompt(string);
			break;
		case 2:		/* examples menu */
			digprompt("Examples: Square, Triangle, I-Vowel, A-Vowel, U-Vowel: ");
			break;
		}
		diggetmouse(string,&but,&xm,&ym);
		if (string[0]) {
			l = strlen(string);
			/* search commands table */
			for (i=0;i<NUMCOMMAND;i++)
				if ((menu==comtab[i].menu) && (stringcomp(string,comtab[i].comname,l)==0)) {
					com = comtab[i].comval;
					break;
				}
			if (com==COM_NULL) {
				menu=0;
				beep();
			}
			/* process menu switch internally */
			if (com==COM_MAIN) {
				menu = 0;
				com = COM_NULL;
			}
			else if (com==COM_OPTION) {
				menu = 1;
				com = COM_NULL;
			}
			else if (com==COM_EXAMPLE) {
				menu = 2;
				com = COM_NULL;
			}
		}
		else if (but==4) {
			if (inarea(xm,ym,FREQ_ORIGIN_X,FREQ_ORIGIN_Y,FREQ_AXIS_X+charwidth,FREQ_AXIS_Y+charheight)) {
				freq = (int)(xm-FREQ_ORIGIN_X);
				freq = (int)((freq > MAXFREQ) ? MAXFREQ : freq);
				amp  = (int)(ym-FREQ_ORIGIN_Y);
				amp  = (int)((amp > MAXAMP) ? MAXAMP : amp);
				idx=0;
				dst=abs(freq-htab[0].freq);
				for (i=1;i<MAXHARMONIC;i++)
					if ((htab[i].amp > 0) && (abs(freq-htab[i].freq) < dst)) {
						dst = abs(freq-htab[i].freq);
						idx = i;
					}
				*h = idx;
				com = COM_HARM;
			}
			else if (inarea(xm,ym,FREQ_AXIS_X+charwidth,FREQ_ORIGIN_Y,SCREEN_WIDTH,FREQ_AXIS_Y+charheight)) {
				*h = 0;
				while ((*h < MAXHARMONIC) && (htab[*h].amp > 0)) (*h)++;
				if (*h < MAXHARMONIC)
					com = COM_NEW_HARM;
				else
					beep();
			}
			else if (inarea(xm,ym,FREQ_ORIGIN_X,AMP_AXIS_Y,FREQ_AXIS_X,FREQ_ORIGIN_Y)) {
				*h = (int)((xm-FREQ_ORIGIN_X)/(2*PHASE_WIDTH));
				if (htab[*h].amp > 0)
					com = COM_PHASE;
				else
					beep();
			}
			else
				com = COM_REPLAY;
		}
		else if (but==2)
			com = COM_REPLAY;
		else if (but==1)
			drawamplitude(ON);
	} while (com==COM_NULL);

	return(com);
}

int getharmpos(freq,amp)
int	*freq;
float	*amp;
{
	float	xm,ym;
	char	but;
	char	string[80];

	do {
		digprompt("Select Harmonic Frequency and Amplitude");
		diggetmouse(string,&but,&xm,&ym);
		if (string[0]) {
			return(0);
		}
		else if (but) {
			if (inarea(xm,ym,FREQ_ORIGIN_X,FREQ_ORIGIN_Y,FREQ_AXIS_X,FREQ_AXIS_Y)) {
				*freq = (int)(xm-FREQ_ORIGIN_X);
				*amp  = (ym-FREQ_ORIGIN_Y)/100;
				if (!anharmonic)
					*freq = ((int)(*freq+FUNDFREQ/2)/(int)FUNDFREQ)*(int)FUNDFREQ;
				return(1);
			}
			else
				return(0);
		}
	} while (1);
}

int getphase(phase)
int	*phase;
{
	float	xm,ym;
	char	but;
	char	string[80];
	int	h;

	do {
		digprompt("Select Phase for Component");
		diggetmouse(string,&but,&xm,&ym);
		if (string[0]) {
			return(0);
		}
		else if (but) {
			if (inarea(xm,ym,FREQ_ORIGIN_X,AMP_AXIS_Y,FREQ_AXIS_X,FREQ_ORIGIN_Y)) {
				h = (int)((xm - FREQ_ORIGIN_X+digdata.chwidth/digdata.xscale)/(2*PHASE_WIDTH));
				*phase = (int)(360.0*(xm-FREQ_ORIGIN_X-h*2*PHASE_WIDTH)/PHASE_WIDTH);
				*phase = ((*phase+45)/90)*90;
				*phase = (*phase < 0) ? 0 : *phase;
				*phase = (*phase > 359) ? 0 : *phase;
				return(1);
			}
			else
				return(0);
		}
	} while (1);
}

int harmsortok()
{
	int	i;
	int	lastfreq=0;

	/* check phases */
	for (i=0;i<NUMPHASE;i++)
		if ((htab[i].amp==0) &&
		    (htab[i+1].amp > 0))
			return(0);
	/* check harmonic sequence */
	for (i=0;i<MAXHARMONIC;i++)
		if (htab[i].amp > 0)
			if (htab[i].freq <= lastfreq)
				return(0);
			else
				lastfreq = htab[i].freq;
	return(1);
}

void harmsort()
{
	int	i,j;
	struct harmonic_rec htemp;

	/* check not in order already */
	if (harmsortok())
		return;

	/* turn them all off */
	for (i=0;i<MAXHARMONIC;i++)
		if (htab[i].amp > 0)
			drawharmonic(i,OFF);
		else
			htab[i].freq = (int)MAXFREQ;
	for (i=0;i<NUMPHASE;i++)
		drawphase(i,OFF);

	/* sort in order of frequency */
	for (i=1;i<MAXHARMONIC;i++) {
		j=i;
		htemp = htab[j];
		while ((j > 0) && (htemp.freq < htab[j-1].freq)) {
			htab[j] = htab[j-1];
			j--;
		}
		htab[j] = htemp;
	}

	/* eliminate duplicates */
	j=0;
	for (i=1;i<MAXHARMONIC;i++) {
		if (htab[i].freq == htab[j].freq) 
			htab[j].amp += htab[i].amp;
		else {
			j++;
			htab[j] = htab[i];
		}
	}
	for (j++;j<MAXHARMONIC;j++)
		htab[j].amp=0.0;

	/* redraw harmonics */
	for (i=0;i<MAXHARMONIC;i++)
		drawharmonic(i,ON);
	for (i=0;i<NUMPHASE;i++)
		drawphase(i,ON);
}


void replaywave()
{
	static int doplay = -10;
	digprompt("Replay ..");
	if (doplay==-10)
		doplay = dac_open(NULL);

	if (doplay >= 0)
		dac_playback(repbuff,RBUFSIZE,SAMPFREQ,16,1,1);
}

/* set up harmonics for demonstrations */
void setharmonics(demo)
int	demo;
{
	int	i,idx;
	double	*hp=NULL;

	switch (demo) {
	case COM_SQUARE:
		logscale=1;
		hp = square_demo;
		break;
	case COM_TRIANGLE:
		logscale=1;
		hp = triangle_demo;
		break;
	case COM_IVOWEL:
		logscale=1;
		hp = ivowel_demo;
		break;
	case COM_AVOWEL:
		logscale=1;
		hp = avowel_demo;
		break;
	case COM_UVOWEL:
		logscale=1;
		hp = uvowel_demo;
		break;
	}
	idx=0;
	for (i=0;i<NUMDEMO_HARMONIC;i++) {
		htab[i].amp = 0.0;
		htab[i].phase = 0;
		htab[idx].freq = (int)((i+1)*FUNDFREQ);
		if (hp[i] > 0) {
			htab[idx].amp = hp[i]/100;
			htab[idx].phase = 0;
			idx++;
		}
		else if (hp[i] < 0) {
			htab[idx].amp = -hp[i]/100;
			htab[idx].phase = 180;
			idx++;
		}
	}
	anharmonic=0;
}

/* save replay buffer to SFS file */
void sfssynth(fname)
char	*fname;
{
	genwave(repbuff,RBUFSIZE,0,MAXHARMONIC,1.0);
	sfsheader(&spitem,SP_TYPE,0,2,1,1.0/SAMPFREQ,0.0,1,0,0);
	sprintf(spitem.history,"%s(file=%s)",PROGNAME,initfile);
	putitem(fname,&spitem,RBUFSIZE,(void *)repbuff);
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */

	/* processing */
	int	i,hnum;
	int	com;
	int	freq,phase;
	float	amp;
	float	scale;
	char	messg[256],*logname(),*ttynme();
	time_t	tim;
	char	timbuf[32];
	
	/* decode switches */
	while ( (c = getopt(argc,argv,"Ipi:t:s:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Harmonic synthesis V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'p' :	/* one shot to printer */
			laser++;
			break;
		case 'i' :	/* initialisation file */
			strncpy(initfile,optarg,80);
			break;
		case 't' :	/* print title */
			strncpy(laser_title,optarg,80);
			break;
		case 's' :	/* SFS synthesis */
			strncpy(sfsfilename,optarg,80);
			sfs++;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg)
		error("usage: %s (-I) (-p) (-t title) (-i initfile) (-s sfsfile)",PROGNAME);

	/* generate sine table */
	for (i=0;i<SBUFSIZE;i++)
		sinetab[i] = (int)(32767.0 * sin(TWOPI*(double)i/(double)SBUFSIZE));

	/* start-up configuration */
	if (initfile[0])
		loadharmfile(initfile);
	else
		loadharmstand();

	/* check not just SFS synthesis */
	if (sfs) {
		sfssynth(sfsfilename);
		exit(0);
	}

	/* initialise screen */
	digstart((char)((laser)?DIG_DEFAULT_PRINTER:DIG_DEFAULT_TERM),NULL,1);
	charwidth = digdata.chwidth/digdata.xscale;
	SCREEN_WIDTH = MAXFREQ / (1 - 16 * charwidth);
	digscale(SCREEN_WIDTH,SCREEN_HEIGHT,0);
	digclearscreen();
	charwidth = digdata.chwidth/digdata.xscale;
	charheight = digdata.chheight/digdata.yscale;
	FREQ_ORIGIN_X = 6 * charwidth;
	FREQ_AXIS_X = FREQ_ORIGIN_X + MAXFREQ;
	if (digdata.device=='b') SELECT=60;		/* on BBC use dotted line */

	/* draw frequency graph */
	drawfrequency();

	/* draw phases */
	for (i=0;i<NUMPHASE;i++)
		drawphase(i,ON);

	/* draw amplitude graph */
	wavepoints = (int)(1 + ((AMP_AXIS_X - AMP_ORIGIN_X)*digdata.xscale)/2);	/* 1pt per 2 pixels */
	if (wavepoints > DBUFSIZE) {
		digquit(0);
		error("displayed waveform too large");
	}
	genwave(repbuff,RBUFSIZE,0,MAXHARMONIC,1.0);
	genwave(dspbuff,wavepoints,0,MAXHARMONIC,SAMPFREQ*WAVETIME/wavepoints);
	drawamplitude(ON);

	/* check for non-interactive */
	if (laser) {
		tim = time((time_t *)0);
		strcpy(timbuf,ctime(&tim));
		timbuf[19]='\0';
		if (*laser_title)
			sprintf(messg,"Harmonic Synthesis.  User: %s  Terminal: %s  Title: %s",logname(),ttyname(1),laser_title);
		else
			sprintf(messg,"Harmonic Synthesis.  User: %s  Terminal: %s  Date: %s",logname(),ttyname(1),timbuf);

		digtext(24,0.0,SCREEN_HEIGHT-charheight,messg);
		digbox(24,0.0,0.0,SCREEN_WIDTH,SCREEN_HEIGHT-3*charheight/2);
		digbox(24,0.0,AMP_AXIS_Y+5*charheight/2,SCREEN_WIDTH,FREQ_ORIGIN_Y-3*charheight);
		digquit(0);
		exit(0);
	}

	/* initialise mouse */
	diginitmouse(SCREEN_WIDTH/2.0,SCREEN_HEIGHT/2.0);

	/* command loop */
	while ((com=getcommand(&hnum)) != COM_QUIT) switch (com) {

	case COM_HARM:		/* harmonic selected */
		drawharmonic(hnum,SELECT);
		if (getharmpos(&freq,&amp)) {
			/* move harmonic */
			drawharmonic(hnum,OFF);
			if (!anharmonic && (htab[hnum].freq != freq) && (hnum < NUMPHASE)) {
				htab[hnum].freq=freq;
				drawphase(hnum,ON);
			}
			else
				htab[hnum].freq = freq;
			htab[hnum].amp = amp;
			drawharmonic(hnum,ON);
		}
		else {
			/* delete harmonic */
			drawharmonic(hnum,OFF);
			if (hnum < NUMPHASE) drawphase(hnum,OFF);
			htab[hnum].amp=0.0;
			htab[hnum].freq=(int)MAXFREQ;
		}
		harmsort();
		genwave(repbuff,RBUFSIZE,0,MAXHARMONIC,1.0);
		genwave(dspbuff,wavepoints,0,MAXHARMONIC,SAMPFREQ*WAVETIME/wavepoints);
		drawamplitude((mandraw)?-1:ON);
		if (maxamp > MAXAMP) goto rescale;
		break;

	case COM_NEW_HARM:		/* harmonic selected */
		drawharmonic(-1,SELECT);
		if (getharmpos(&freq,&amp)) {
			/* add a new harmonic */
			drawharmonic(-1,ON);
			htab[hnum].freq = freq;
			htab[hnum].phase = 0;
			htab[hnum].amp = amp;
			drawharmonic(hnum,ON);
			if (hnum < NUMPHASE) drawphase(hnum,ON);
			harmsort();
			genwave(repbuff,RBUFSIZE,0,MAXHARMONIC,1.0);
			genwave(dspbuff,wavepoints,0,MAXHARMONIC,SAMPFREQ*WAVETIME/wavepoints);
			if (!mandraw) drawamplitude(ON);
			if (maxamp > MAXAMP) goto rescale;
		}
		else {
			beep();
			drawharmonic(-1,ON);
		}
		break;

	case COM_PHASE:		/* phase selected */
		drawphase(hnum,SELECT);
		if (getphase(&phase)) {
			drawphase(hnum,OFF);
			htab[hnum].phase = phase;
			drawphase(hnum,ON);
			genwave(repbuff,RBUFSIZE,0,MAXHARMONIC,1.0);
			genwave(dspbuff,wavepoints,0,MAXHARMONIC,SAMPFREQ*WAVETIME/wavepoints);
			drawamplitude((mandraw)?-1:ON);
			if (maxamp > MAXAMP) goto rescale;
		}
		else {
			beep();
			drawphase(hnum,ON);
			
		}
		break;

	case COM_REPLAY:	/* replay waveform */
		replaywave();
		break;

	case COM_SAVE:		/* save waveform to file */
		saveharmfile();
		break;

	case COM_CLEAR:			/* clear harmonics */
		if (initfile[0])
			loadharmfile(initfile);
		else
			loadharmstand();

	case COM_REFRESH:		/* clear & redraw screen */
		digclearscreen();
		drawfrequency();
		for (i=0;i<NUMPHASE;i++)
			drawphase(i,ON);
		genwave(repbuff,RBUFSIZE,0,MAXHARMONIC,1.0);
		genwave(dspbuff,wavepoints,0,MAXHARMONIC,SAMPFREQ*WAVETIME/wavepoints);
		drawamplitude(ON);
		break;

	case COM_SCALE:		/* rescale harmonics */
		drawamplitude(-1);
rescale:
		if (logscale) {
			scale = 0.5*MAXAMP/(float)maxamp;
			for (i=0;i<MAXHARMONIC;i++)
				if (flogdecode((double)htab[i].amp)*scale > MAXAMP)
					scale = MAXAMP/flogdecode((double)htab[i].amp);
			for (i=0;i<MAXHARMONIC;i++) if (htab[i].amp > 0) {
				drawharmonic(i,OFF);
				htab[i].amp = flogcode(flogdecode((double)htab[i].amp)*scale);
				htab[i].amp = (htab[i].amp < 0) ? 0 : htab[i].amp;
				htab[i].amp = (htab[i].amp > MAXAMP) ? MAXAMP : htab[i].amp;
				drawharmonic(i,ON);
			}
		}
		else {
			scale = 0.5*MAXAMP/maxamp;
			for (i=0;i<MAXHARMONIC;i++)
				if (htab[i].amp*scale > MAXAMP)
					scale = MAXAMP/(float)htab[i].amp;
			for (i=0;i<MAXHARMONIC;i++) {
				drawharmonic(i,OFF);
				htab[i].amp *= scale;
				drawharmonic(i,ON);
			}
		}
		genwave(repbuff,RBUFSIZE,0,MAXHARMONIC,1.0);
		genwave(dspbuff,wavepoints,0,MAXHARMONIC,SAMPFREQ*WAVETIME/wavepoints);
		drawamplitude((mandraw)?-1:ON);
		break;

	case COM_ANHARM:	/* allow anharmonics */
		if (anharmonic) break;
		for (i=0;i<MAXHARMONIC;i++) {
			anharmonic=0;
			drawharmonic(i,OFF);
			anharmonic=1;
			drawharmonic(i,ON);
		}
		break;

	case COM_DOHARM:	/* force harmonics */
		if (!anharmonic) break;
		for (i=0;i<MAXHARMONIC;i++) {
			anharmonic=1;
			drawharmonic(i,OFF);
			htab[i].freq = ((int)(htab[i].freq+FUNDFREQ/2)/(int)FUNDFREQ)*(int)FUNDFREQ;
			anharmonic=0;
			drawharmonic(i,ON);
		}
		harmsort();
		genwave(repbuff,RBUFSIZE,0,MAXHARMONIC,1.0);
		genwave(dspbuff,wavepoints,0,MAXHARMONIC,SAMPFREQ*WAVETIME/wavepoints);
		drawamplitude((mandraw)?-1:ON);
		if (maxamp > MAXAMP) goto rescale;
		break;

	case COM_LOG:		/* log amplitude scale */
		if (logscale) break;
		logscale=1;
		for (i=0;i<MAXHARMONIC;i++)
			htab[i].amp = flogcode(htab[i].amp);
		drawfrequency();
		break;

	case COM_LIN:		/* linear amplitude scale */
		if (!logscale) break;
		logscale=0;
		for (i=0;i<MAXHARMONIC;i++)
			htab[i].amp = flogdecode(htab[i].amp);
		drawfrequency();
		break;

	case COM_COMPO:		/* draw components */
		drawcomponents(BGLINE);
		break;		

	case COM_PRINT:		/* do hard copy of current setting */
		hardcopy();
		break;

	case COM_SQUARE:	/* set up example waveforms */
	case COM_TRIANGLE:
	case COM_IVOWEL:
	case COM_AVOWEL:
	case COM_UVOWEL:
		setharmonics(com);
		drawfrequency();
		for (i=0;i<NUMPHASE;i++)
			drawphase(i,ON);
		genwave(repbuff,RBUFSIZE,0,MAXHARMONIC,1.0);
		genwave(dspbuff,wavepoints,0,MAXHARMONIC,SAMPFREQ*WAVETIME/wavepoints);
		drawamplitude((mandraw)?-1:ON);
		break;

	case COM_AUTODRAW:
		mandraw=0;
		drawamplitude(ON);
		break;
	case COM_MANDRAW:
		mandraw=1;
		drawamplitude(OFF);
		drawamplitude(-1);
		break;
	}

	digkillmouse();
	digclearscreen();
	digquit(0);
	exit(0);
}

