/* bitmap -- DIG routines for BITMAP output */

/* Mark Huckvale - University College London */

/* version 1.0 - May 1996 */

#include "SFSCONFG.h"
#ifdef	EVB

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <ctype.h>
#include <time.h>
#include <malloc.h>
#include <string.h>
#include "dig.h"
#include "digdata.h"

extern void encodegif(int,FILE *,int (*)());

/* default image size */
#define DEFWIDTH	600
#define DEFHEIGHT	450

/* default font */
#include "pc6x8.h"
#define DEFCHARWIDTH	6
#define DEFCHARHEIGHT	8

/* operating modes */
#define BM_INITIALISED 1

/* local storage of parameters */
struct {
	int	mode;		/* bitmap modes */
	int	subtype;	/* subtype as supplied from digmap */
	int	myfile;		/* PS system opened output file */
	int	col;
	int	width,height;
	unsigned char **bmap;	/* bitmap array */
	int	currow,curcol;
} digDEVB;

/* bitmap colours */
#define NUM_COLOUR	32
static struct rgb_rec {
	unsigned char	r,g,b;
} colmap[NUM_COLOUR]={
	{ 255,255,255},	/* 0 = background */
	{ 200, 50, 50},	/* 1 */
	{ 100, 75,100},	/* 2 */
	{ 100, 75, 50},	/* 3 */
	{ 150,200,  0},	/* 4 */
	{  25, 25,100},	/* 5 = was red - mapped to dark grey/blue */
	{  50,200,100},	/* 6 */
	{  50,100,200},	/* 7 */
	{ 150,150,  0},	/* 8 */
	{  50, 50, 50},	/* 9 = was white - mapped to dark grey */
	{ 200,100,  0},	/* 10 */
	{ 100,  0,100},	/* 11 */
	{   0,100,100},	/* 12 */
	{ 200,100,100},	/* 13 */
	{  50,100,200},	/* 14 */
	{   0,  0,  0},	/* 15 = foreground */
	{ 250,250,250},	/* 16 */
	{ 248,248,248},	/* 17 */
	{ 244,244,244},	/* 18 */
	{ 238,238,238},	/* 19 */
	{ 230,230,230},	/* 20 */
	{ 220,220,220},	/* 21 */
 	{ 208,208,208},	/* 22 */
	{ 194,194,194},	/* 23 */
	{ 178,178,178},	/* 24 */
	{ 160,160,160},	/* 25 */
	{ 140,140,140},	/* 26 */
	{ 118,118,118},	/* 27 */
	{  94, 94, 94},	/* 28 */
	{  68, 68, 68},	/* 29 */
	{  40, 40, 40},	/* 30 */
	{  10, 10, 10},	/* 31 */
};

/* forward declarations */
void bm_save_bitmap(FILE *op);
static void bm_colour();

/* get graphics parameters */
int bm_params(subtype,digdev)
int	subtype;
struct digdevice *digdev;
{
	char	*p;
	char	cbuf[32];

	/* record operating mode */
	digDEVB.subtype = subtype;

	/* set up image dimensions */
	if ((p=getenv("GSIZE"))!=NULL) {
		strncpy(cbuf,p,31);
		digDEVB.width = atoi(strtok(cbuf,"x"));
		digDEVB.height = atoi(strtok(NULL,""));
	}
	else {
		digDEVB.width = DEFWIDTH;
		digDEVB.height = DEFHEIGHT;
	}
	if (digDEVB.width > 10)
		digdev->nhoriz = digDEVB.width;
	else
		digdev->nhoriz = digDEVB.width = DEFWIDTH;
	if (digDEVB.height > 10)
		digdev->nvert = digDEVB.height;
	else
		digdev->nvert = digDEVB.height = DEFHEIGHT;

	/* return other image characteristics */
	digdev->aspect = (float)1.0;
 	digdev->chwidth = DEFCHARWIDTH;
	digdev->chheight = DEFCHARHEIGHT;
	digdev->greylevels = 16;
	digdev->greypixwidth = 1;
	digdev->greypixheight = 1;

	return(0);
}

/* open output file */
int bm_outfile(fp)
FILE	**fp;
{
	char	*p,ofilename[256];

	if ((p=getenv("GFILE"))!=NULL)
		strncpy(ofilename,p,255);
	else
		strcpy(ofilename,"dig.gif");
#ifdef DOS
	if ((*fp = fopen(ofilename,"wb"))==NULL)
#else
	if ((*fp = fopen(ofilename,"w"))==NULL)
#endif
		error("could not open '%s'",ofilename);
	return(0);
}

/* open output */
int bm_open()
{
	int	i;

	if (digDEVB.mode & BM_INITIALISED) return(0);

	/* allocate memory for bitmap */
	if ((digDEVB.bmap = (unsigned char **)calloc(digDEVB.height,sizeof(unsigned char *)))==NULL)
		error("could not get memory for bitmap");
	for (i=0;i<digDEVB.height;i++)
		if ((digDEVB.bmap[i]=(unsigned char *)calloc(digDEVB.width,1))==NULL)
			error("could not get memory for bitmap");

	digDEVB.mode |= BM_INITIALISED;
	return(0);
}

/* interrupt */
void bm_inter()
{
	int	i;
	if (digDEVB.mode & BM_INITIALISED) {
		for (i=0;i<digDEVB.height;i++)
			free(digDEVB.bmap[i]);
		free(digDEVB.bmap);
	}
}

/* close and save bitmap */
void bm_close()
{
	if (digDEVB.mode & BM_INITIALISED) {

		/* save bitmap */
		bm_save_bitmap(digdata.outfile);

		/* free bitmap */
		bm_inter();

		/* shut down output stream */
		fclose(digdata.outfile);
	}
	digDEVB.mode = 0;
}

/* clear screen */
int bm_clear()
{
	int	i;
	for (i=0;i<digDEVB.height;i++)
		memset(digDEVB.bmap[i],0,digDEVB.width);
	return(0);
}

/* set up colour from bundle */
static void bm_colour(bundle)
int	bundle;
{
	if (bundle==0)
		digDEVB.col=0;	/* background */
	else if ((bundle >= 100) && (bundle < 115))
		digDEVB.col=bundle-84;	/* grey scale */
	else {
		bundle = bundle % 15;
		if (bundle==0)
			digDEVB.col=15;
		else
			digDEVB.col=bundle;
	}
}

/* plot line on bitmap */
static void bm_plotline(int x1,int y1,int x2,int y2)
{
	int	xt,yt;
	int	i;
	double	slope,val;

	if ((x1==x2) && (y1==y2)) {
		/* plot point */
		digDEVB.bmap[y1][x1]=digDEVB.col;
		return;
	}
	xt = x2 - x1;
	yt = y2 - y1;
	if (xt==0) {
		/* vertical line */
		if (y1 > y2) { yt = y1; y1 = y2; y2 = yt; }
		for (i=y1;i<=y2;i++) digDEVB.bmap[i][x1]=digDEVB.col;
	}
	else if (yt==0) {
		/* horizontal line */
		if (x1 > x2) { xt = x1; x1 = x2; x2 = xt; }
		for (i=x1;i<=x2;i++) digDEVB.bmap[y1][i]=digDEVB.col;
	}
	else if (abs(xt) > abs(yt)) {
		/* use x as a base */
		slope = (float)yt/(float)xt;
		if (x2 < x1) {
			val = y2+0.5;
			for (i=x2;i<=x1;i++,val+=slope)
				digDEVB.bmap[(int)val][i]=digDEVB.col;
		}
		else {
			val = y1+0.5;
			for (i=x1;i<=x2;i++,val+=slope)
				digDEVB.bmap[(int)val][i]=digDEVB.col;
		}
	}
	else {
		/* use y as a base */
		slope = (float)xt/(float)yt;
		if (y2 < y1) {
			val = x2+0.5;
			for (i=y2;i<=y1;i++,val+=slope)
				digDEVB.bmap[i][(int)val]=digDEVB.col;
		}
		else {
			val = x1+0.5;
			for (i=y1;i<=y2;i++,val+=slope)
				digDEVB.bmap[i][(int)val]=digDEVB.col;
		}
	}
}

/* plot poly-line */
void bm_polyline(bundle,buff,len)
int bundle;
short *buff;
int32	len;
{
	int     x1,y1,x2,y2;

	/* set up plot colour */
	bm_colour(bundle);

	/* move to first point */
	x1 = *buff++;
	y1 = *buff++;
	len-=(int32)2;

	/* plot lines */
	while (len > (int32)0) {
		x2 = *buff++;
		y2 = *buff++;
		len -= (int32)2;
		bm_plotline(x1,y1,x2,y2);
		x1 = x2;
		y1 = y2;
	}
}

/* draw character */
static void bm_char(int x,int y,int ch)
{
	unsigned char	*cbuf;
	int		i,j;
	unsigned char	mask;

	/* locate character in stored font */
	cbuf = pc6x8_font[ch];

	/* row by row */
	for (i=DEFCHARHEIGHT-1;i>=0;i--) {
		/* get row mask */
		mask = *cbuf++;
		/* display row */
		for (j=0;j<DEFCHARWIDTH;j++) {
			if (mask & 0x80)
				digDEVB.bmap[y+i][x+j] = digDEVB.col;
			mask <<= 1;
		}
	}

}

/* draw text */
void bm_text(bundle,x,y,str)
int bundle;
int x;
int y;
char *str;
{
	/* set up plot colour */
	bm_colour(bundle);

	/* write string character by character */
	while (*str) {
		if (x > (digDEVB.width-DEFCHARWIDTH)) break;
		bm_char(x,y,*str);
		str++;
		x += DEFCHARWIDTH;
	}
}

/* draw a filled rectangle */
void bm_fillrect(bundle,x1,y1,x2,y2)
int bundle;
int x1;
int y1;
int x2;
int y2;
{
	int	i,j;

	/* set up plot colour */
	bm_colour(bundle);

	for (i=x1;i<=x2;i++)
		for (j=y1;j<=y2;j++)
			digDEVB.bmap[j][i] = digDEVB.col;

}

/* grey-level display */
void bm_gscale(x,y,height,buff)
int x;
int y;
int height;
unsigned char *buff;
{
	int	i;

	for (i=height-1;i>=0;i--,buff++)
		digDEVB.bmap[y+i][x] = 16 + *buff;

}

/* wait for key routine - NULL don't wait */
void bm_pause()
{
}

/* server routine for GIF compression */
int bm_get_pixel()
{
	int	pixel;
	if (digDEVB.currow<0) return(-1);
	pixel = digDEVB.bmap[digDEVB.currow][digDEVB.curcol++];
	if (digDEVB.curcol==digDEVB.width) {
		digDEVB.curcol=0;
		digDEVB.currow--;
	}
	return(pixel);
}

/* save bitmap to GIF file */
void bm_save_bitmap(FILE *op)
{
	int	i,codesize;

	fwrite("GIF87a",6,1,digdata.outfile);
	fputc(digDEVB.width & 0xFF,digdata.outfile);
	fputc(digDEVB.width >> 8,digdata.outfile);
	fputc(digDEVB.height & 0xFF,digdata.outfile);
	fputc(digDEVB.height >> 8,digdata.outfile);
	fputc(0xF7,digdata.outfile);
	fputc(0,digdata.outfile);
	fputc(0,digdata.outfile);
	for (i=0;i<NUM_COLOUR;i++)
		fwrite((char *)&colmap[i],3,1,digdata.outfile);
	for (;i<256;i++)
		fwrite((char *)&colmap[0],3,1,digdata.outfile);
	fputc(',',digdata.outfile);
	fputc(0,digdata.outfile);
	fputc(0,digdata.outfile);
	fputc(0,digdata.outfile);
	fputc(0,digdata.outfile);
	fputc(digDEVB.width & 0xFF,digdata.outfile);
	fputc(digDEVB.width >> 8,digdata.outfile);
	fputc(digDEVB.height & 0xFF,digdata.outfile);
	fputc(digDEVB.height >> 8,digdata.outfile);
	fputc(7,digdata.outfile);
	codesize = 8;
	fputc(codesize,digdata.outfile);
	codesize++;
	digDEVB.currow = digDEVB.height-1;
	digDEVB.curcol = 0;
	encodegif(codesize,digdata.outfile,bm_get_pixel);
/*	fputc(0,digdata.outfile); - superfluous ? */
	fputc(0,digdata.outfile);
	fputc(';',digdata.outfile);
	fflush(digdata.outfile);
}
#endif
